//*****************************************************************************
//* Name:
//*      12pmapii.h
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file contains the declaration of the implementation class for the
//*      Korg 1212 I/O driver API class.  The API class uses a Cheshire cat
//*      implementation to reduce the impact of changes to the applications
//*      developed around this API. 
//*
//* Modification Log:
//*
//*      1.2   07/31/97 Bill
//*      Added RegisterMsgBoxFunc() declaration.  
//*
//*      1.1   02/27/97 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1997 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#ifndef K12PMAPII_H
#define K12PMAPII_H

#ifndef K1212STAT_H
#include "1212stat.h"
#endif
#ifndef K1212API_H
#include "1212api.h" 
#endif
#ifndef K12PMCTRL_H
#include "12pmctrl.h" 
#endif


// ------------------------------------------------------------------------------
// The following strings are used for naming each wave device on a card.
// ------------------------------------------------------------------------------
#define SR_K1212_WAVE_DEVICE1 "1212 I/O Adat 1-2"
#define SR_K1212_WAVE_DEVICE2 "1212 I/O Adat 3-4"
#define SR_K1212_WAVE_DEVICE3 "1212 I/O Adat 5-6"
#define SR_K1212_WAVE_DEVICE4 "1212 I/O Adat 7-8"
#define SR_K1212_WAVE_DEVICE5 "1212 I/O Analog"
#define SR_K1212_WAVE_DEVICE6 "1212 I/O S/PDIF"
extern  char __far* waveDevNameTable[];  // stores the names of each wave device

// ------------------------------------------------------------------------------
// The following strings are used for naming each wave device source on a card.
// ------------------------------------------------------------------------------
#define SR_K1212_WAVE_SOURCE1 "Adat 1-2"
#define SR_K1212_WAVE_SOURCE2 "Adat 3-4"
#define SR_K1212_WAVE_SOURCE3 "Adat 5-6"
#define SR_K1212_WAVE_SOURCE4 "Adat 7-8"
#define SR_K1212_WAVE_SOURCE5 "Analog"
#define SR_K1212_WAVE_SOURCE6 "S/PDIF"
extern  char __far* waveDevSourceNameTable[];  // stores the names of each wave device source


// ------------------------------------------------------------------------------
// The following strings are used for naming each channel on a card. 
// ------------------------------------------------------------------------------
#define SR_K1212_CHANNEL1         "Adat 1"
#define SR_K1212_CHANNEL2         "Adat 2"
#define SR_K1212_CHANNEL3         "Adat 3"
#define SR_K1212_CHANNEL4         "Adat 4"
#define SR_K1212_CHANNEL5         "Adat 5"
#define SR_K1212_CHANNEL6         "Adat 6"
#define SR_K1212_CHANNEL7         "Adat 7"
#define SR_K1212_CHANNEL8         "Adat 8"
#define SR_K1212_CHANNEL9         "Analog Left"
#define SR_K1212_CHANNEL10        "Analog Right"
#define SR_K1212_CHANNEL11        "S/PDIF Left"
#define SR_K1212_CHANNEL12        "S/PDIF Right" 
#define SR_K1212_CHANNEL_INVALID  "Invalid Slider ID" 
extern  char __far* channelNameTable[];  // stores the names of each volume slider


// ------------------------------------------------------------------------------
// The following constant is used to mark an unconfigured card data structure.
// As of today, the device number comes from the system as 8 bits, so we define
// a larger number for the null device ID.
// ------------------------------------------------------------------------------
const int nullDeviceNumber = 32000;


// ------------------------------------------------------------------------------
// The following struct is used to store information about each configured card.
// ------------------------------------------------------------------------------
struct K1212CardData {
   DWORD            cardHandle;    // this stores the card handle
   DWORD            ref2Index;     // this maps the card reference number
                                   //   to the driver's card index.
   k1212CardInfoPM  cardInfo;      // this struct contains information on the card.
   
   DWORD            componentType; // component type of the entire card.  This is set
                                   //   to the speaker destination for the mixer driver

   DWORD            waveDevCompType[k1212NumWaveDevices]; 
                                   // the above array contains the source component type  
                                   // (used by the mixer) for each wave device on the card.

   LPMXDCLIENT      lpMixerClients[k1212NumWaveDevices];   
                                   // the above array contains the head pointer of
                                   //    the linked list of mixer clients for each
                                   //    wave device.

   CKorg1212MixControl masterVolume;
                                   // master volume control for all of the card's
                                   // wave devices

   CKorg1212MixControl mixControls[k1212NumWaveDevices][K1212_CTRL_MAX];
                                   // the above array contains the controls for each
                                   // of the wave devices on a card
};


// ------------------------------------------------------------------------------
// Private implementation details for the CKorg1212Api class.
// ------------------------------------------------------------------------------
class CKorg1212ApiImpl {
   
   friend class CKorg1212Api;
   
   private:
    // ---------------------
    // private data members
    // ---------------------
      CKorg1212Api*  parentPtr;           // contains the pointer to the API object that
                                          //    this class provides private implementation for
      APIHandle      apiHandle;           // handle assigned to us by the driver.  Must be
                                          //    passed in all calls subsequent to registering.
      k1212PMIn*     pmInputBufPtr;       // input buffer for DeviceIoControl calls
      WORD           pmInputBufSize;      // size of the input buffer
      k1212PMOut*    pmOutputBufPtr;      // output buffer for DeviceIoControl calls
      WORD           pmOutputBufSize;     // size of the output buffer

      FARPROC        vxdAddress;          // driver address obtained from get PM address call
      WORD           numBuffers;          // set in the constructor to the number of buffers the card
                                          //    uses for record and playback
      WORD           numCardsInSystem;    // set in the constructor to the number of cards
                                          // in the system reported by the driver.  This can
                                          // be used to indicate the maximum usable index into 
                                          // the card info array.

      K1212CardData  kcdCardData[k1212MaxCards];    // card specific information for all cards

    // -------------------------
    // private member functions
    // -------------------------
      CKorg1212ApiImpl();                     // constructor - creates resources
      ~CKorg1212ApiImpl();                    // destructor.

      void     SetPMInputBufPtr    (void);    // allocates memory for the DIOC input buffer
                                              //    and assigns the pointer
      void     SetPMOutputBufPtr   (void);    // allocates memory for the DIOC output
                                              //    buffer and assigns the pointer
      void     SetVxDAddress       (void);    // opens a connection to the Korg1212 driver
                                              //    and stores the handle to the VxD
      BOOL     RegisterAPI         (void);    // registers the connection to the driver
                                              //    and stores the API channel handle.

      BOOL     RegisterMsgBoxFunc  (DWORD funcPtr);    // registers a msg box display function.


         // ------------------------------------------------------------
         // Data member access functions
         // ------------------------------------------------------------
      APIHandle      GetApiHandle        (void)  {return apiHandle;}
      k1212PMIn*     GetPMInputBufPtr    (void)  {return pmInputBufPtr;}
      WORD           GetPMInputBufSize   (void)  {return pmInputBufSize;}
      k1212PMOut*    GetPMOutputBufPtr   (void)  {return pmOutputBufPtr;}
      WORD           GetPMOutputBufSize  (void)  {return pmOutputBufSize;}
      FARPROC        GetVxDAddress       (void)  {return vxdAddress;}         
      WORD           GetNumCardsInSystem (void)  {return numCardsInSystem;}
      WORD           GetNumBuffers       (void)  {return numBuffers;}

      CardState      GetCardState       (k1212CardRef refNum)  {return (CardState)(kcdCardData[refNum].cardInfo.state);}
      WORD           GetCardBusNum      (k1212CardRef refNum)  {return kcdCardData[refNum].cardInfo.busNum;}
      WORD           GetCardDevNum      (k1212CardRef refNum)  {return kcdCardData[refNum].cardInfo.deviceNum;}
      WORD           GetCardIrqNum      (k1212CardRef refNum)  {return kcdCardData[refNum].cardInfo.irqNum;}
      DWORD          GetCardDevNode     (k1212CardRef refNum)  {return kcdCardData[refNum].cardInfo.devNode;}
      WORD           GetCardIndex       (k1212CardRef refNum)  {return (WORD)kcdCardData[refNum].ref2Index;}
      DWORD          GetCardHandle      (k1212CardRef refNum)  {return kcdCardData[refNum].cardHandle;}
      DWORD          GetCardCompType    (k1212CardRef refNum)  {return kcdCardData[refNum].componentType;}
      DWORD          GetWaveCompType    (k1212CardRef refNum, WORD deviceId)  
                                                               {return kcdCardData[refNum].waveDevCompType[deviceId];}
      LPMXDCLIENT    GetMixerClientPtr  (k1212CardRef refNum, WORD deviceId)  
                                                               {return kcdCardData[refNum].lpMixerClients[deviceId];}

      CKorg1212MixControl __far* GetMixControl(k1212CardRef refNum,
                                               WORD         deviceId,
                                               WORD         controlId)   
                           {return &(kcdCardData[refNum].mixControls[deviceId][controlId]);}

      CKorg1212MixControl __far* GetMasterVolumeControl(k1212CardRef refNum)
                           {return &(kcdCardData[refNum].masterVolume);}


         // ------------------------------------------------------------
         // The following functions set the card data members
         // ------------------------------------------------------------
      void SetNumBuffers       (WORD       wNumBufs)  {numBuffers       = wNumBufs;}
      void SetNumCardsInSystem (WORD       wNum)      {numCardsInSystem = wNum;}
      void SetApiHandle        (APIHandle  apiHdl)    {apiHandle        = apiHdl;}
      void IncNumCardsInSystem (void)                 {numCardsInSystem++;}

      void SetCardState      (k1212CardRef refNum, CardState  newState)  
                                 {kcdCardData[refNum].cardInfo.state = newState;}

      void SetCardBusNum     (k1212CardRef refNum, WORD       newBusNum)  
                                 {kcdCardData[refNum].cardInfo.busNum = newBusNum;}

      void SetCardDevNum     (k1212CardRef refNum, WORD       newDevNum)  
                                 {kcdCardData[refNum].cardInfo.deviceNum = newDevNum;}

      void SetCardIrqNum     (k1212CardRef refNum, WORD       newIrqNum)  
                                 {kcdCardData[refNum].cardInfo.irqNum = newIrqNum;}

      void SetCardDevNode    (k1212CardRef refNum, DWORD      newDevNode)  
                                 {kcdCardData[refNum].cardInfo.devNode = newDevNode;}

      void SetCardIndex      (k1212CardRef refNum, WORD       cardIndex)  
                                 {kcdCardData[refNum].ref2Index = cardIndex;}

      void SetCardHandle     (k1212CardRef refNum, DWORD      cdHndl)    
                                 {kcdCardData[refNum].cardHandle = cdHndl;}

      void SetCardCompType   (k1212CardRef refNum, DWORD      type)    
                                 {kcdCardData[refNum].componentType = type;}

      void SetWaveCompType   (k1212CardRef refNum, WORD deviceId, DWORD type)    
                                 {kcdCardData[refNum].waveDevCompType[deviceId] = type;}

      void SetMixerClientPtr (k1212CardRef refNum, 
                              WORD         deviceId, 
                              LPMXDCLIENT  lpMixerClient)    
                                 {kcdCardData[refNum].lpMixerClients[deviceId] = lpMixerClient;}


         // -----------------------------------------------------------------
         // other/utility member functions
         // -----------------------------------------------------------------
      K1212CmdRet   UpdateCardInfo     (k1212CardRef   refNum);
      K1212CmdRet   GetCardInfo        (k1212CardRef                cardRefNum,
                                        k1212PublicCardInfo __far & kciCardInfo);

      BOOL     VerifyRefNum       (k1212CardRef refNum);
      BOOL     VerifyBufNum       (DWORD        bufNum);

      void     InitCardData       (k1212CardRef refNum); // this function inits our local card data
      void     ClearCardData      (k1212CardRef refNum); // this function clears out our local
                                                               // data structures for the specified card.
                                                               // it is used after a card reboot.

      void     QueryAllCards      (void);  // this function fills in the cardInfo array
                                                 // when this object is constructed
      
};

#endif // K12PMAPII_H
