//*****************************************************************************
//* Name:
//*      1212dnld.c
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file contains the definition of the functions required for DSP
//*      microcode download to the card.
//*
//* Modification Log:
//*
//*      1.2   12/30/96 Bill
//*      Modified to support multiple cards.  
//*
//*      1.1   11/15/96 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1996 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <vtoolsc.h>
#include <vmm.h>
#include <debug.h>
#include <vmmreg.h>
#include <configmg.h>
#include <winerror.h>

#define  _K1212_DSP_DOWNLOADER
#include "1212dnld.h"
#undef   _K1212_DSP_DOWNLOADER

#ifndef  K1212INTF_H
#include "1212intf.h"
#endif
#ifndef  K1212CARD_H
#include "1212card.h"
#endif
#ifndef  K1212CFG_H
#include "1212cfg.h"
#endif
#ifndef  K1212FILE_H
#include "1212file.h"
#endif


// ----------------------------------------------------------------------------
// downloadDSPCode 
//
//    This function starts the DSP download.  Note, the card's IRQ must be
//    hooked before calling this function, so that we'll know when the download
//    is complete and we can release the host memory storing the DSP code, and
//    know that the card is now smart and API capable.
//
//    We start by opening the file, which is fixed at the path given in 
//    dspCodeFileName, getting its size, allocating memory, reading the file 
//    into memory, then notifying the card so that it may commence the 
//    download.
//
//    Returns: CR_SUCCESS if all goes well, or a CONFIGRET error code.
// ----------------------------------------------------------------------------
CONFIGRET downloadDSPCode(DWORD cardIndex)
{
   HANDLE      dspFileHandle;
   DWORD       dspFileSize;
   DWORD       numDSPPages;
   int         returnVal;
   MEMHANDLE   dspMemHdl;
   PVOID       dspMemPtr;
   PVOID       dspPhysMemPtr;
   char        fileName[255];
   
   #ifdef DEBUG
      DWORD*  ptr;
   #endif // DEBUG

   // ---------------------------------------------------------------
   // verify the state of the card before proceeding.
   // ---------------------------------------------------------------
   if (getCardState(cardIndex) >= K1212_STATE_DSP_IN_PROCESS) {
      return CR_FAILURE;
   }
   
   // ---------------------------------------------------------------
   // open the file and make sure no error occurred while opening.
   // ---------------------------------------------------------------
   dspFileHandle  = openFileRO(dspCodeFilePath);

   if (dspFileHandle == NULL) {
      // ------------------------------------------------------------
      // last ditch effort.
      // ------------------------------------------------------------
      strcpy(fileName, Get_Config_Directory());
      strcat(fileName, "SYSTEM\\");
      strcat(fileName, dspCodeFileName);
      dspFileHandle  = openFileRO(fileName);
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: trying file %s\n", cardIndex, fileName);
      #endif //DEBUG
      
      if (dspFileHandle == NULL) {
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: Open DSP file failed\n", cardIndex);
         #endif //DEBUG
         return CR_FAILURE;
      }
   }
   
   // ---------------------------------------------------------------
   // get the size of the file and then allocate enough memory to 
   // hold the file's contents.  Make sure the allocation succeeded.
   // In case we get called via the reboot command, check for the pre-
   // existence of a DSP memory buffer and free it before making a
   // new one.
   // ---------------------------------------------------------------
   dspFileSize = getFileSize(dspFileHandle);
   numDSPPages = ((dspFileSize / PAGESIZE) + 1);    // plus 1 for fractional part

   freeDSPucodeImageMemory(cardIndex);
   
   if (_PageAllocate(numDSPPages,           // number of pages to allocate
                     PG_SYS,                // allocate in system shared area
                     NULL,                  // no VM specified for sys memory
                     0,                     // 4K alignment 
                     0,                     // min acceptable page number 
                     MAXSYSTEMPAGE,         // max acceptable page number
                     &dspPhysMemPtr,        // pointer for physical address returned
                     (PAGECONTIG   |        // specify: contiguous memory
                      PAGEFIXED    |        //          can't be moved
                      PAGEUSEALIGN |        //          use align required for contig
                      PAGEZEROINIT),        //          initialize to all zeroes
                     &dspMemHdl,            // receiver for the memory block's handle
                     &dspMemPtr             // receiver for the memory block's linear address
       )) {

      setDSPMemPtr    (cardIndex, dspMemPtr);
      setDSPPhysMemPtr(cardIndex, dspPhysMemPtr);
      setDSPMemHdl    (cardIndex, dspMemHdl);
   }
   else {
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: Allocate memory for DSP code failed\n",
                              cardIndex
         );
      #endif //DEBUG
      closeFile(dspFileHandle);
      return CR_FAILURE;
   }

   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: DSP mem hdl 0x%x linear: 0x%x phys: 0x%x\n",
                           cardIndex,
                           getDSPMemHdl(cardIndex),
                           getDSPMemPtr(cardIndex),
                           getDSPPhysMemPtr(cardIndex)
      );
   #endif // DEBUG

   // ---------------------------------------------------------------
   // read the file into memory and then tell the card  to start
   // downloading it.  If this fails, close the file and release 
   // memory.  Otherwise, change the card's state so we know we are
   // now expecting a download acknowledgement (can't do anything
   // else with the card until the acknowledgement is received).
   // ---------------------------------------------------------------
   if (readFile(dspFileHandle, 
                getDSPMemPtr(cardIndex), 
                dspFileSize
       )) {
      Send1212Command(cardIndex,
                      K1212_DB_StartDSPDownload, 
                      UpperWordSwap((DWORD)getDSPPhysMemPtr(cardIndex)),
                      DUMMY_PARAMETER,
                      DUMMY_PARAMETER,
                      DUMMY_PARAMETER
      );
      setCardState(cardIndex, K1212_STATE_DSP_IN_PROCESS);
      returnVal = CR_SUCCESS;
   }
   else {
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: Read DSP file failed\n",
                              cardIndex
         );
      #endif //DEBUG
      _PageFree(getDSPMemHdl(cardIndex), 0);   // zero means no flags - just a regular free
      setDSPMemPtr(cardIndex, 0);
      returnVal = CR_FAILURE;
   }

   closeFile(dspFileHandle);

   return returnVal;
}

