//*****************************************************************************
//* Name:
//*      1212crdp.c
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file contains the definition of the card configuration structure
//*      functions.
//*
//* Modification Log:
//*
//*      1.8   08/20/97 Bill
//*      (v 1.0B6) Added functions and initialization for autoWaveSyncMode.
//*
//*      1.7   08/05/97 Bill
//*      (v 1.0B5) Added functions and initialization for idleMonitorOn and
//*      idleMonPending.  Added member for setting the new card command register 
//*      in the shared buffer.
//*
//*      1.6   06/18/97 Bill
//*      (v 1.11) Added initialization for the wave output numBufsPrefilled member
//*      and wave input/output buffer return linked list head pointers.
//*
//*      1.5   06/16/97 Bill
//*      (v 1.09) Added calculation for numMsPerInt when the clock rate is set.
//*
//*      1.4   06/09/97 Bill
//*      (v 1.08) Added initialization for the wave output looping members.
//*
//*      1.3   05/29/97 Bill
//*      (v 1.04) Added initialization for the new wave output preFillBufIndex 
//*      member.  
//*
//*      1.2   05/15/97 Bill
//*      (v 1.02) Added initialization for wave output filter members.  
//*
//*      1.1   01/21/97 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1996 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <vtoolsc.h>
#include <vmm.h>
#include <debug.h>
#include <vmmreg.h>
#include <configmg.h>
#include <vpicd.h>

#ifndef  K1212IFPR_H
#include "1212ifpr.h"
#endif
#ifndef  K1212CARD_H
#include "1212card.h"
#endif
#ifndef  K1212CFG_H
#include "1212cfg.h"
#endif
#ifndef  K1212WAVESTRUCT_H
#include "1212wavestruct.h"
#endif

// ----------------------------------------------------------------------------
// the following array of card configuration data structures is defined in
// 1212card.c
// ----------------------------------------------------------------------------
extern cardConfig cardData[k1212MaxCards];


DWORD stereoPairOffset[] = {K1212Adat1Offset,      // wave device 1
                            K1212Adat3Offset,      // wave device 2
                            K1212Adat5Offset,      // wave device 3
                            K1212Adat7Offset,      // wave device 4
                            K1212AnlgLeftOffset,   // wave device 5
                            K1212SPDIFLeftOffset,  // wave device 6
      };


#ifdef DEBUG
// ----------------------------------------------------------------------------
// state string definition for debugging purposes
// ----------------------------------------------------------------------------
char* stateName[] = {"Non-existent",
                     "Uninitialized",
                     "DSP download in process",
                     "DSP download complete",
                     "Ready",
                     "Open",
                     "Setup for play",
                     "Playing",
                     "Monitor mode on",
                     "Calibrating"
                     "Invalid"
      };
#endif // DEBUG


// -------------------------------------------------------------------------
// cardData utility functions
// -------------------------------------------------------------------------
BOOL  VerifyCardIndex(DWORD cardIndex)
{
   if (cardIndex >= k1212MaxCards) {
      return FALSE;
   }
   else {
      return TRUE;
   }
}

BOOL  VerifyDeviceIndex(DWORD deviceId)
{
   if (deviceId >= k1212NumWaveDevices) {
      return FALSE;
   }
   else {
      return TRUE;
   }
}



// -------------------------------------------------------------------------
// -------------------------------------------------------------------------
// cardData accessor functions
// -------------------------------------------------------------------------
// -------------------------------------------------------------------------
#ifdef DEBUG
#define VERIFY_INDEX                                                             \
         if (cardIndex >= k1212MaxCards) {                                       \
            Debug_Printf_Service("Korg 1212 Driver: Bad index specified - %d\n", \
                                 cardIndex                                       \
            );                                                                   \
            return 0;                                                            \
         }
#else
#define VERIFY_INDEX
#endif

DEVNODE getDevNode(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].devNode;
}

DWORD getBusNum(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].busNum;
}

DWORD getDeviceNum(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].deviceNum;
}

const char* getRegistryPath(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].registryPath;
}

char* getRegPathPtr(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].registryPath;
}

THREADHANDLE getFillThreadHdl(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].fillThreadHdl;
}

PVOID getFillFunc(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].fillFunc;
}

DWORD getFillParam(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].fillParam;
}

APIHandle getApiHandle(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].apiHandle;
}

APIMode getApiMode(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].apiMode;
}

int getIrqNum(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].irqNum;
}

ULONG getIrqHandle(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].irqHandle;
}

ULONG getIrqCount(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].irqCount;
}

DWORD getCurFillBufNum(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].curFillBufNum;
}

DWORD getCurRecBufNum(DWORD cardIndex)       // v 1.11
{
   VERIFY_INDEX
   return cardData[cardIndex].curRecBufNum;
}

char* getPhysMemBase(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].physMemBase;
}

char* getMemBase(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].memBase;
}

WORD getIoBase(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].ioBase;
}

DWORD getMemLength(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].memLength;
}

MEMHANDLE getDSPMemHdl(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].dspMemBufHdl;
}

char* getDSPMemPtr(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].dspMemBufPtr;
}

char* getDSPPhysMemPtr(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].dspPhysBufPtr;
}

CardState getCardState(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].state;
}

DWORD getOpenCount(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].openCount;
}

DWORD getCmdRetryCount(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].cmdRetryCount;
}

MEMHANDLE getSharedBufferHdl(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufHdl;
}

KorgSharedBuffer* getSharedBufferPtr(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPtr;
}

KorgSharedBuffer* getSharedPhyBufPtr(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPhys;
}

WORD* getVolumeTable(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPtr->volumeData;
}

WORD* getRoutingTable(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPtr->routeData;
}

WORD* getVolumeTablePhy(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPhys->volumeData;
}

WORD* getRoutingTablePhy(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPhys->routeData;
}

KorgAudioBuffer* getPlayBuf0Address(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPtr->playDataBufs;
}

KorgAudioBuffer* getPlayBufXAddress(DWORD cardIndex,
                                    DWORD bufferIndex)
{
   KorgAudioBuffer* bufPtr;
   VERIFY_INDEX
   bufPtr = cardData[cardIndex].sharedBufPtr->playDataBufs;
   bufPtr += bufferIndex;
   return bufPtr;
}

KorgAudioBuffer* getRecBuf0Address(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPtr->recordDataBufs;
}

KorgAudioBuffer* getRecBufXAddress(DWORD cardIndex,
                                   DWORD bufferIndex)
{
   KorgAudioBuffer* bufPtr;
   VERIFY_INDEX
   bufPtr = cardData[cardIndex].sharedBufPtr->recordDataBufs;
   bufPtr += bufferIndex;
   return bufPtr;
}

KorgAudioBuffer* getPlayDataPhy(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPhys->playDataBufs;
}

KorgAudioBuffer* getRecDataPhy(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPhys->recordDataBufs;
}

DWORD getAdatTimeCode(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPtr->AdatTimeCode;
}

DWORD* getAdatTimeCodePtr(DWORD cardIndex)
{
   VERIFY_INDEX
   return &(cardData[cardIndex].sharedBufPtr->AdatTimeCode);
}

DWORD* getAdatTimeCodePhy(DWORD cardIndex)
{
   VERIFY_INDEX
   return &(cardData[cardIndex].sharedBufPhys->AdatTimeCode);
}

DWORD getNumMsPerInt(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].numMsPerInt;
}

ClockSourceIndex getClkSrcRate(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].clkSrcRate;
}

short getChannelVolume(DWORD cardIndex,
                       WORD  channel)
{
   VERIFY_INDEX
   return (short)(getVolumeTable(cardIndex)[channel]);
}

DWORD getCardCommand(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].sharedBufPtr->cardCommand;
}

WORD getChannelRoute(DWORD cardIndex,
                     WORD  channel)
{
   VERIFY_INDEX
   return (short)(getRoutingTable(cardIndex)[channel]);
}

WORD getLeftADCInSens(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].leftADCInSens;
}

WORD getRightADCInSens(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].rightADCInSens;
}

k1212WaveInDevice* getWaveInDevPtr(DWORD cardIndex, DWORD deviceId)
{
   VERIFY_INDEX
   if (deviceId < k1212NumWaveDevices) {
      return &(cardData[cardIndex].waveInDevice[deviceId]);
   }
   else {
      return 0;
   }
}

k1212WaveOutDevice* getWaveOutDevPtr(DWORD cardIndex, DWORD deviceId)
{
   VERIFY_INDEX
   if (deviceId < k1212NumWaveDevices) {
      return &(cardData[cardIndex].waveOutDevice[deviceId]);
   }
   else {
      return 0;
   }
}

WORD getMasterVolume(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].masterVolume;
}

HANDLE getFileHandle(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].fileHandle;
}

BOOL getIdleMonitorState(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].idleMonitorOn;
}

BOOL getAutoWaveSyncMode(DWORD cardIndex)
{
   VERIFY_INDEX
   return cardData[cardIndex].autoWaveSyncMode;
}



// -------------------------------------------------------------------------
// -------------------------------------------------------------------------
// cardData mutator functions
// -------------------------------------------------------------------------
// -------------------------------------------------------------------------
#ifdef DEBUG
#define VERIFY_INDEX_NO_RETVAL                                                   \
         if (cardIndex >= k1212MaxCards) {                                       \
            Debug_Printf_Service("Korg 1212 Driver: Bad index specified - %d\n", \
                                 cardIndex                                       \
            );                                                                   \
            return;                                                              \
         }
#else
#define VERIFY_INDEX_NO_RETVAL
#endif

void initWaveDeviceData(DWORD cardIndex)
{
   DWORD                deviceId;
   WORD                 sampleNum;
   k1212WaveInDevice*   waveInDevPtr;
   k1212WaveOutDevice*  waveOutDevPtr;
   
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      waveInDevPtr  = getWaveInDevPtr (cardIndex, deviceId);
      waveOutDevPtr = getWaveOutDevPtr(cardIndex, deviceId);
      
      // --------------------------------------------------------
      // init wave input device data members 
      // --------------------------------------------------------
      waveInDevPtr->state                 = K1212_WAVEINSTATE_DISABLED;
      waveInDevPtr->format                = WAVFMT_UNSUPPORTED;
      waveInDevPtr->waveBufferQueue       = 0;
      waveInDevPtr->curWaveSamplePtr      = 0;
      waveInDevPtr->bufLength             = 0;
      waveInDevPtr->numBytesRecordedInBuf = 0;
      waveInDevPtr->numSamplesRecorded    = 0;
      waveInDevPtr->lastEstimatedOffset   = 0;
      waveInDevPtr->stereoPairOffset      = stereoPairOffset[deviceId];
      waveInDevPtr->chainLinkSync         = FALSE;
      if (deviceId != K1212_DEVICE_SPDIF) {
         waveInDevPtr->spdifDevice = SPDIF_FALSE;
      }
      else {
         waveInDevPtr->spdifDevice = SPDIF_TRUE;
      }
      waveInDevPtr->ring3Callback         = 0;
      waveInDevPtr->clientHandle          = 0;
      waveInDevPtr->translateFunctionPtr  = recordXlateFunctions[WAVFMT_UNSUPPORTED];
      
      // --------------------------------------------------------
      // init wave output device data members
      // --------------------------------------------------------
      waveOutDevPtr->state                 = K1212_WAVEOUTSTATE_DISABLED;
      waveOutDevPtr->format                = WAVFMT_UNSUPPORTED;
      waveOutDevPtr->waveBufferQueue       = 0;
      waveOutDevPtr->retBufQueue           = 0;
      waveOutDevPtr->curWaveSamplePtr      = 0;
      waveOutDevPtr->remainingBytesInBuf   = 0;
      waveOutDevPtr->numSamplesPlayed      = 0;
      waveOutDevPtr->lastEstimatedPosition = 0;
      waveOutDevPtr->stereoPairOffset      = stereoPairOffset[deviceId];
      waveOutDevPtr->numBufsPrefilled      = 0;
      waveOutDevPtr->numSamplesPrefilled   = 0;
      waveOutDevPtr->leftVolumeLevel       = k1212MaxWaveVolume;
      waveOutDevPtr->rightVolumeLevel      = k1212MaxWaveVolume;
      waveOutDevPtr->muted                 = FALSE;
      waveOutDevPtr->volumeControlEnabled  = FALSE;
      waveOutDevPtr->chainLinkSync         = FALSE;
      if (deviceId != K1212_DEVICE_SPDIF) {
         waveOutDevPtr->spdifDevice = SPDIF_FALSE;
      }
      else {
         waveOutDevPtr->spdifDevice = SPDIF_TRUE;
      }
      waveOutDevPtr->ring3Callback         = 0;
      waveOutDevPtr->clientHandle          = 0;
      waveOutDevPtr->translateFunctionPtr  = playXlateFunctions[WAVFMT_UNSUPPORTED];
      waveOutDevPtr->inLoop                = FALSE;      // v1.08
      waveOutDevPtr->loopStartBuffer       = 0;          // v1.08

      // 1.02
      waveOutDevPtr->curSampleSubscript    = 0;
      for (sampleNum = 0; sampleNum < MAX_NUM_WAVEOUT_FILTER_SAMPLES; sampleNum++) {
         waveOutDevPtr->prevLeftSample[sampleNum] = 0;
         waveOutDevPtr->prevRightSample[sampleNum] = 0;
      }
   }
}

void initCardData(DWORD cardIndex)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].devNode        = 0;
   cardData[cardIndex].busNum         = 0;
   cardData[cardIndex].deviceNum      = 0;
   cardData[cardIndex].fillThreadHdl  = 0;
   cardData[cardIndex].fillParam      = 0;
   cardData[cardIndex].apiHandle      = 0;
   cardData[cardIndex].apiMode        = K1212_NATIVE_API_MODE;
   cardData[cardIndex].irqNum         = 0;
   cardData[cardIndex].irqFlags       = 0;
   cardData[cardIndex].irqHandle      = 0;
   cardData[cardIndex].irqCount       = 0;
   cardData[cardIndex].curFillBufNum  = 0;
   cardData[cardIndex].physMemBase    = 0;
   cardData[cardIndex].memBase        = 0;
   cardData[cardIndex].ioBase         = 0;
   cardData[cardIndex].memLength      = 0;
   cardData[cardIndex].dspMemBufHdl   = 0;
   cardData[cardIndex].dspMemBufPtr   = 0;
   cardData[cardIndex].dspPhysBufPtr  = 0;
   cardData[cardIndex].state          = K1212_STATE_NONEXISTENT;
   cardData[cardIndex].idleMonitorOn  = FALSE;
   cardData[cardIndex].openCount      = 0;
   cardData[cardIndex].sharedBufHdl   = 0;
   cardData[cardIndex].sharedBufPtr   = 0;
   cardData[cardIndex].sharedBufPhys  = 0;
   cardData[cardIndex].numMsPerInt    = ((1000 * kPlayBufferFrames) / 44100);
   cardData[cardIndex].clkSrcRate     = K1212_CLKIDX_LocalAt44_1K;
   cardData[cardIndex].leftADCInSens  = k1212MaxADCSens;
   cardData[cardIndex].rightADCInSens = k1212MaxADCSens;
   initCardInterfaceRegs(cardIndex);   
   initWaveDeviceData(cardIndex);   
   cardData[cardIndex].cmdRetryCount    = 0;
   cardData[cardIndex].autoWaveSyncMode = FALSE;
   cardData[cardIndex].masterVolume     = k1212MaxWaveVolume;
}

void setDevNode(DWORD cardIndex, DEVNODE devNode)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].devNode = devNode;
}

void setBusNum(DWORD cardIndex, DWORD busNum)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].busNum = busNum;
}

void setRegistryPath(DWORD cardIndex, const char* regPath)
{
   VERIFY_INDEX_NO_RETVAL
   strncpy(cardData[cardIndex].registryPath, 
           regPath, 
           MAX_REGPATH_SIZE
   );
}

void setDeviceNum(DWORD cardIndex, DWORD deviceNum)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].deviceNum = deviceNum;
}

void setFillThreadHdl(DWORD cardIndex, THREADHANDLE fillThreadHdl)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].fillThreadHdl = fillThreadHdl;
}

void setFillFunc(DWORD cardIndex, PVOID fillFunc)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].fillFunc = fillFunc;
}

void setFillParam(DWORD cardIndex, DWORD fillParam)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].fillParam = fillParam;
}

void setApiHandle(DWORD cardIndex, APIHandle apiHandle)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].apiHandle = apiHandle;
}

void setApiMode(DWORD cardIndex, APIMode apiMode)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].apiMode = apiMode;
}

void setIrqNum(DWORD cardIndex, int iIrqNum)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].irqNum = iIrqNum;
}

void setIrqHandle(DWORD cardIndex, ULONG iIrqHandle)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].irqHandle = iIrqHandle;
}

void incIrqCount(DWORD cardIndex)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].irqCount++;
}

void clearIrqCount(DWORD cardIndex)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].irqCount = 0;
}

void clearCmdRetryCount(DWORD cardIndex)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].cmdRetryCount = 0;
}

void setCurFillBufNum(DWORD cardIndex, DWORD bufNum)
{
   VERIFY_INDEX_NO_RETVAL
   if (bufNum >= kNumBuffers) {
      bufNum = kNumBuffers - 1;
   }
   cardData[cardIndex].curFillBufNum = bufNum;
}

void incCurFillBufNum(DWORD cardIndex)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].curFillBufNum++;
   if (cardData[cardIndex].curFillBufNum >= kNumBuffers) {
      cardData[cardIndex].curFillBufNum = 0;
   }
}

void setCurRecBufNum(DWORD cardIndex, DWORD bufNum)   // v1.11
{
   VERIFY_INDEX_NO_RETVAL
   if (bufNum >= kNumBuffers) {
      bufNum = kNumBuffers - 1;
   }
   cardData[cardIndex].curRecBufNum = bufNum;
}

void incCurRecBufNum(DWORD cardIndex)                 // v1.11
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].curRecBufNum++;
   if (cardData[cardIndex].curRecBufNum >= kNumBuffers) {
      cardData[cardIndex].curRecBufNum = 0;
   }
}

void setPhysMemBase(DWORD cardIndex, char* pMemBase)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].physMemBase = pMemBase;
}

void setMemBase(DWORD cardIndex, char* pMemBase)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].memBase = pMemBase;
}

void setIoBase(DWORD cardIndex, WORD wIoBase)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].ioBase = wIoBase;
}

void setMemLength(DWORD cardIndex, DWORD dMemLength)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].memLength = dMemLength;
}

void setDSPMemHdl(DWORD cardIndex, MEMHANDLE dspMemHdl)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].dspMemBufHdl = dspMemHdl;
}

void setDSPMemPtr(DWORD cardIndex, char* dspMemPtr)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].dspMemBufPtr = dspMemPtr;
}

void setDSPPhysMemPtr(DWORD cardIndex, char* dspPhysMemPtr)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].dspPhysBufPtr = dspPhysMemPtr;
}

void incOpenCount(DWORD cardIndex)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].openCount++;
}

void setOpenCount(DWORD cardIndex, DWORD value)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].openCount = value;
}

void addToCmdRetryCount(DWORD cardIndex,
                        DWORD numRetries)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].cmdRetryCount += numRetries;
}

void setCardState(DWORD cardIndex, CardState csState)
{
   VERIFY_INDEX_NO_RETVAL

   // ---------------------------------------------------------------
   // do new state dependent processing here
   // ---------------------------------------------------------------
   switch (csState) {

      case K1212_STATE_READY:
         K1212TurnOnIdleMonitor(cardIndex);
         break;

      case K1212_STATE_OPEN:
         K1212TurnOffIdleMonitor(cardIndex);
         break;
   }
   
   // ---------------------------------------------------------------
   // make sure state is updated after idle monitor is set!
   // ---------------------------------------------------------------
   cardData[cardIndex].state = csState;

//   #ifdef DEBUG
//      Debug_Printf_Service("Korg 1212 Card %d: State change to %s \n",
//                           cardIndex,
//                           stateName[csState]
//      );
//   #endif //DEBUG
}

void setSharedBufferHdl(DWORD cardIndex, MEMHANDLE handle)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].sharedBufHdl = handle;
}

void setSharedBufferPtr(DWORD cardIndex, KorgSharedBuffer* bufPtr)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].sharedBufPtr = bufPtr;
}

void setSharedPhyBufPtr(DWORD cardIndex, KorgSharedBuffer* bufPtr)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].sharedBufPhys = bufPtr;
}

void setAdatTimeCode(DWORD cardIndex, DWORD timeCodeVal)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].sharedBufPtr->AdatTimeCode = timeCodeVal;
}

BOOL setClkSrcRate(DWORD cardIndex, ClockSourceIndex clkSrcSel)
{
   VERIFY_INDEX
   if (clkSrcSel < K1212_CLKIDX_Invalid) {
      cardData[cardIndex].clkSrcRate = clkSrcSel;
      
      // -----------------------------------------------------------
      // determine number of milliseconds per interrupt and store
      // -----------------------------------------------------------
      switch (clkSrcSel) {
         case K1212_CLKIDX_AdatAt44_1K:
         case K1212_CLKIDX_WordAt44_1K:
         case K1212_CLKIDX_LocalAt44_1K:
            cardData[cardIndex].numMsPerInt = ((1000 * kPlayBufferFrames) / 44100);
            break;
            
         case K1212_CLKIDX_AdatAt48K:
         case K1212_CLKIDX_WordAt48K:
         case K1212_CLKIDX_LocalAt48K:
            cardData[cardIndex].numMsPerInt = ((1000 * kPlayBufferFrames) / 48000);
            break;
      }
      
      return TRUE;
   }
   else {
      cardData[cardIndex].clkSrcRate = K1212_CLKIDX_AdatAt48K;
      cardData[cardIndex].numMsPerInt = ((1000 * kPlayBufferFrames) / 48000);
      return FALSE;
   }
}

void setChannelVolume(DWORD cardIndex, WORD channel, short volume)
{
   VERIFY_INDEX_NO_RETVAL
   getVolumeTable(cardIndex)[channel] = volume;
}

void setCardCommand(DWORD cardIndex, DWORD command)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].sharedBufPtr->cardCommand = command;
}

void setChannelRoute(DWORD cardIndex, WORD channel, WORD route)
{
   VERIFY_INDEX_NO_RETVAL
   getRoutingTable(cardIndex)[channel] = route;
}

void setLeftADCInSens(DWORD cardIndex, WORD lSens)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].leftADCInSens = lSens;
}

void setRightADCInSens(DWORD cardIndex, WORD rSens)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].rightADCInSens = rSens;
}

void setMasterVolume(DWORD cardIndex,
                     WORD  volume)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].masterVolume = volume;
}

void setFileHandle(DWORD  cardIndex,
                   HANDLE fileHandle)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].fileHandle = fileHandle;
}

void setIdleMonitorState(DWORD cardIndex,
                         BOOL  newState)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].idleMonitorOn = newState;
}
   
void setAutoWaveSyncMode(DWORD cardIndex,
                         BOOL  setting)
{
   VERIFY_INDEX_NO_RETVAL
   cardData[cardIndex].autoWaveSyncMode = setting;
}



// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
// private functions for setting and reading the card interface register
// addresses.
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------

void     initCardInterfaceRegs(DWORD cardIndex)
{
   cardData[cardIndex].intfRegs.statusRegPtr     = 0;
   cardData[cardIndex].intfRegs.outDoorbellPtr   = 0;
   cardData[cardIndex].intfRegs.inDoorbellPtr    = 0;
   cardData[cardIndex].intfRegs.mailbox0Ptr      = 0;
   cardData[cardIndex].intfRegs.mailbox1Ptr      = 0;
   cardData[cardIndex].intfRegs.mailbox2Ptr      = 0;
   cardData[cardIndex].intfRegs.mailbox3Ptr      = 0;
   cardData[cardIndex].intfRegs.controlRegPtr    = 0;
   cardData[cardIndex].intfRegs.sensRegPtr       = 0;
   cardData[cardIndex].intfRegs.idRegPtr         = 0;
}


// --------------------------------------------------------------------------
// the following functions return memory mapped card register addresses
// --------------------------------------------------------------------------
DWORD*   statusRegAddress(DWORD cardIndex)   
{
   return cardData[cardIndex].intfRegs.statusRegPtr;
}
DWORD*   outDoorbellAddress(DWORD cardIndex)   
{
   return cardData[cardIndex].intfRegs.outDoorbellPtr;
}
DWORD*   inDoorbellAddress(DWORD cardIndex)   
{
   return cardData[cardIndex].intfRegs.inDoorbellPtr;
}
DWORD*   mailbox0Address(DWORD cardIndex)  
{
   return cardData[cardIndex].intfRegs.mailbox0Ptr;
}
DWORD*   mailbox1Address(DWORD cardIndex) 
{
   return cardData[cardIndex].intfRegs.mailbox1Ptr;
}
DWORD*   mailbox2Address(DWORD cardIndex) 
{
   return cardData[cardIndex].intfRegs.mailbox2Ptr;
}
DWORD*   mailbox3Address(DWORD cardIndex)   
{
   return cardData[cardIndex].intfRegs.mailbox3Ptr;
}
DWORD*   controlRegAddress(DWORD cardIndex)   
{
   return cardData[cardIndex].intfRegs.controlRegPtr;
}
WORD*    sensRegAddress(DWORD cardIndex)   
{
   return cardData[cardIndex].intfRegs.sensRegPtr;
}
DWORD*   idRegAddress(DWORD cardIndex)   
{
   return cardData[cardIndex].intfRegs.idRegPtr;
}


// --------------------------------------------------------------------------
// the following functions assign the memory mapped card register addresses
// --------------------------------------------------------------------------
void     setStatusRegAddress(DWORD cardIndex, DWORD* address) 
{
   cardData[cardIndex].intfRegs.statusRegPtr = address;
}
void     setOutDoorbellAddress(DWORD cardIndex, DWORD* address) 
{
   cardData[cardIndex].intfRegs.outDoorbellPtr = address;
}
void     setInDoorbellAddress(DWORD cardIndex, DWORD* address) 
{
   cardData[cardIndex].intfRegs.inDoorbellPtr = address;
}
void     setMailbox0Address(DWORD cardIndex, DWORD* address) 
{
   cardData[cardIndex].intfRegs.mailbox0Ptr = address;
}
void     setMailbox1Address(DWORD cardIndex, DWORD* address) 
{
   cardData[cardIndex].intfRegs.mailbox1Ptr = address;
}
void     setMailbox2Address(DWORD cardIndex, DWORD* address) 
{
   cardData[cardIndex].intfRegs.mailbox2Ptr = address;
}
void     setMailbox3Address(DWORD cardIndex, DWORD* address) 
{
   cardData[cardIndex].intfRegs.mailbox3Ptr = address;
}
void     setControlRegAddress(DWORD cardIndex, DWORD* address) 
{
   cardData[cardIndex].intfRegs.controlRegPtr = address;
}
void     setSensRegAddress(DWORD cardIndex, WORD* address) 
{
   cardData[cardIndex].intfRegs.sensRegPtr = address;
}
void     setIdRegAddress(DWORD cardIndex, DWORD* address) 
{
   cardData[cardIndex].intfRegs.idRegPtr = address;
}


// --------------------------------------------------------------------------
// the following functions are for the remaining card interface members
// --------------------------------------------------------------------------
BOOL     regPtrsAssigned(DWORD cardIndex)
{
   if (outDoorbellAddress(cardIndex) == 0) {
      return FALSE;
   }
   else {
      return TRUE;
   }
}
