// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//  Portions Copyright (c) 1998-1999 Microsoft Corporation. 
//
// **************************************************************************

#include <limits.h>
#include "shared.h"

#include "xx_protocol.h"
#include "piocommands.h"

#include "mintopo.h"

#include "pcxerr_e.h"

// These are the values passed to the property handlers in the instance
// parameter that normally represents the channel.
const LONG CHAN_LEFT   = 0;
const LONG CHAN_RIGHT  = 1;
const LONG CHAN_MASTER = -1;

#pragma code_seg("PAGE")

// These macros should not be used in paged code
//
#undef BOARD_LOCK
#undef BOARD_UNLOCK

#define IS_DIGITAL	TRUE
#define IS_ANALOG	FALSE


/*****************************************************************************
 * CMiniportTopologyICH::StartActualNodeSettings
 *****************************************************************************
 * Starts passing node settings down to the board (requires a pipe is defined in the DSP)
 */
STDMETHODIMP_(NTSTATUS) CMiniportTopologyICH::StartActualNodeSettings
(
    IN      WORD    PmChannelNum,
    IN      BOOLEAN PmCapture
)
{
    PAGED_CODE ();

    NTSTATUS ntStatus = STATUS_SUCCESS;

	TopoNodes   LcNode ;

    if( PmCapture )
    {
		// Record settings
		//
		m_bIsInputPipeReady = TRUE ;

        ASSERT( PmChannelNum < (2 * AdapterCommon->GetWaveInNumber()));

		LcNode = NODE_DIGITALIN_VOLUME;

        for ( LONG i = 0; i < m_wChannelCount; i++)
        {
            if ( (stMultiNodeCache[LcNode].dwValidityMask & ( 1L << i)) == 0)
            {
                stMultiNodeCache[LcNode].dwValidityMask |= ( 1L << i);
                stMultiNodeCache[LcNode].lvalue[i] = 0;      // 0 dB
            }
            if( (stMultiNodeCache[LcNode].lvalue[i] != 0) || m_bCopyProtectFlag )
                this->SetDigitalLevel( LcNode, i, &stMultiNodeCache[LcNode].lvalue[i] );
        }
	}
    else
	{
		// Play settings
		//
		m_bIsOutputPipeReady = TRUE ;

        ASSERT( PmChannelNum < (2 * AdapterCommon->GetWaveOutNumber()));

		LcNode = NODE_MASTEROUT_VOLUME;

        for ( LONG i = 0; i < m_wChannelCount; i++)
        {
            if ( (stMultiNodeCache[LcNode].dwValidityMask & ( 1L << i)) == 0)
            {
                stMultiNodeCache[LcNode].dwValidityMask |= ( 1L << i);
                stMultiNodeCache[LcNode].lvalue[i] = 0;      // 0 dB
            }
            if( stMultiNodeCache[LcNode].lvalue[i] != 0 )
                this->SetDigitalLevel( LcNode, i, &stMultiNodeCache[LcNode].lvalue[i] );
        }

		LcNode = NODE_MASTEROUT_MUTE;	// ----------Master Mute------
		// default after pipe creation is not mute, program it only if mute :
		if (stMultiNodeCache[LcNode].dwValidityMask == 0)
        {
            stMultiNodeCache[LcNode].dwValidityMask = 1;
            stMultiNodeCache[LcNode].lvalue[0] = 0;
        }
        if( (stMultiNodeCache[LcNode].lvalue[0] != 0) || m_bDigitalOutputDisable )
		{
			this->SetMasterMute( LcNode, (PBOOL)&stMultiNodeCache[LcNode].lvalue[0]);
		}
	} // end play and record settings

	DOUT (DBG_VSR|DBG_PRINT, ("MinTopo:StartActualNodeSettings (pipe %s @channel %d) OK\n",PmCapture ? "IN":"OUT",PmChannelNum));

	return STATUS_SUCCESS;
}

/*****************************************************************************
 * CMiniportTopologyICH::StopActualNodeSettings
 *****************************************************************************
 * Stops passing node settings down to the board (requires a pipe is defined in the DSP)
 */
void CMiniportTopologyICH::StopActualNodeSettings
(
    IN      WORD    PmChannelNum,
    IN      BOOLEAN PmCapture
)
{
    PAGED_CODE ();

    BOOL        LcMuteOn;
    
	DOUT (DBG_PRINT, ("MinTopo:StopActualNodeSettings (pipe %s @channel %d)\n",PmCapture ? "IN":"OUT", PmChannelNum));

    if (PmCapture)
    {
        m_bIsInputPipeReady = FALSE ;
    }
    else
    {
	/*	LcMuteOn = TRUE ;
        this->SetMasterMute( NODE_MASTEROUT_MUTE, &LcMuteOn);*/

        m_bIsOutputPipeReady = FALSE ;
    }
}



/*****************************************************************************
 * CMiniportTopologyICH::GetDBValues
 *****************************************************************************
 * This function is used internally and does no parameter checking. The only
 * parameter that could be invalid is the node.
 * It returns the dB values (means minimum, maximum, step) of the node control,
 * mainly for the property call "basic support". Sure, the node must be a
 * volume or tone control node, not a mute or mux node.
 */
NTSTATUS CMiniportTopologyICH::GetDBValues
(
    IN PADAPTERCOMMON AdapterCommon,
    IN TopoNodes Node,
    OUT LONG *plMinimum,
    OUT LONG *plMaximum,
    OUT ULONG *puStep
)
{
    PAGED_CODE ();

    DOUT (DBG_PROPERTY, ("[CMiniportTopologyICH::GetDBValues]"));

    // This is going to be simple. Check the node and return the parameters.
    switch (Node)
    {
        // All digital levels
        //
        case NODE_DIGITALIN_VOLUME:

            //if (plMaximum)  *plMaximum = AdapterCommon->ConvertDriver2DSound(0x1ff, Node); // digital max driver level (+18 dB)
			// MBR (10/06/04) limit digital volumes to max 0 dB !
            if (plMaximum)  *plMaximum = AdapterCommon->ConvertDriver2DSound(0x1b7, Node); // digital max driver level limited to (0 dB)
            if (plMinimum)  *plMinimum = AdapterCommon->ConvertDriver2DSound(0,     Node); // digital min driver level (-110 dB)
            if (puStep)     *puStep    = 65536 / 4;     // 1/4 dB
            break;

        // Analog levels
        //
        case NODE_MASTEROUT_VOLUME:

            if (plMaximum)  *plMaximum = AdapterCommon->ConvertDriver2DSound(0x1b7, Node); // digital max driver level (0 dB)
            if (plMinimum)  *plMinimum = AdapterCommon->ConvertDriver2DSound(0,     Node); // digital min driver level (-110 db)
            if (puStep)     *puStep    = 65536 / 4;     // 1/4 dB
            break;

        default:
            // poeser pupe, tu.
            DOUT (DBG_ERROR, ("GetDBValues: Invalid node requested."));
            return STATUS_INVALID_PARAMETER;
    }

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportTopologyICH::PropertyHandler_OnOff
 *****************************************************************************
 * Accesses a KSAUDIO_ONOFF value property.
 * This function (property handler) is called by portcls every time there is a
 * get or a set request for the node. The connection between the node type and
 * the property handler is made in the automation table which is referenced
 * when you register the node.
 * We use this property handler for all nodes that have a checkbox, means mute
 * controls and the special checkbox controls under advanced properties, which
 * are AGC and LOUDNESS.
 */
NTSTATUS CMiniportTopologyICH::PropertyHandler_OnOff
(
    IN      PPCPROPERTY_REQUEST   PropertyRequest
)
{
    PAGED_CODE ();

    ASSERT (PropertyRequest);

    NTSTATUS        ntStatus = STATUS_INVALID_PARAMETER;
    LONG            channel;
    TopoNodes       NodeDef;
    // The major target is the object pointer to the topology miniport.
    CMiniportTopologyICH *that =
        (CMiniportTopologyICH *) PropertyRequest->MajorTarget;

    ASSERT (that);

    DOUT (DBG_PROPERTY, ("[CMiniportTopologyICH::PropertyHandler_OnOff]"));

    // validate node
    if (PropertyRequest->Node == (ULONG)-1)
        return ntStatus;

    // do the appropriate action for the request.

    // we should do a get or a set?
    if ((PropertyRequest->Verb & KSPROPERTY_TYPE_GET) ||
        (PropertyRequest->Verb & KSPROPERTY_TYPE_SET))
    {
        // validate parameters
        if ((PropertyRequest->InstanceSize < sizeof(LONG)) ||
            (PropertyRequest->ValueSize < sizeof(BOOL)))
            return ntStatus;

        // get channel
        channel = *(PLONG)PropertyRequest->Instance;

		NodeDef = that->TransNodeNrToNodeDef (PropertyRequest->Node);

        // check channel types, return when unknown
        // as you can see, we have no multichannel support.
        // We have only mono mutes or On/Off checkboxes although they might control
        // a stereo path. For example, we have a 1-bit mute for CD Volume. This
        // mute controls both CD Volume channels.
        if ((channel != CHAN_LEFT) &&
            (channel != CHAN_MASTER))
		{
            DOUT (DBG_PROPERTY, ("PropertyHandler_OnOff: Invalid channel %d requested (%s)\n", channel, NodeStrings[NodeDef]));
            return ntStatus;
		}

        // get the buffer
        PBOOL OnOff = (PBOOL)PropertyRequest->Value;

        // Switch on the node id. This is just for parameter checking.
        // If something goes wrong, we will immediately return with
        // ntStatus, which is STATUS_INVALID_PARAMETER.
        switch (NodeDef)
        {
            // these are mono channels, don't respond to a right channel
            // request.
            // these are stereo channels; a get on left or master
            // should return the mute.
            case NODE_MASTEROUT_MUTE:

                // just check the type.
                if (PropertyRequest->PropertyItem->Id != KSPROPERTY_AUDIO_MUTE)
                    return ntStatus;
                break;

            case NODE_INVALID:
            default:
                // Ooops.
				DOUT (DBG_ERROR, ("PropertyHandler_OnOff: Invalid node requested (%s)\n", NodeStrings[NodeDef]));
                return ntStatus;
        }

        // Now, do some action!

        if (PropertyRequest->Verb & KSPROPERTY_TYPE_GET)
        {
            ULONG   LcRegister;

            // get the register and read it.
            // Note: mono channels are 'master' here (see fake above).
            // this makes sure that left and right channel is prg. for the virt.
            // controls. On controls that only have the right channel, the left
            // channel programming does nothing cause the mask will be zero.
            if ( that->stMultiNodeCache[NodeDef].dwValidityMask )
            {
                // read only left value to the node cache.
                LcRegister = that->stMultiNodeCache[NodeDef].lvalue[0];
            }
            else 
            {
				// Assume no mute
				LcRegister = FALSE;
            }

            // "normalize"
            *OnOff = LcRegister ? TRUE : FALSE;
            PropertyRequest->ValueSize = sizeof(BOOL);

            DOUT (DBG_PROPERTY, ("GET: %s = 0x%x", NodeStrings[NodeDef], *OnOff));
            // ntStatus was set with the read call! whatever this is, return it.
            ntStatus = STATUS_SUCCESS;

        }
        else    // this must be a set.
        {
            // get the register + mask and write it.
            DOUT (DBG_PROPERTY, ("SET: %s -> 0x%x", NodeStrings[NodeDef], *OnOff));

            //  case NODE_MASTEROUT_MUTE:

			// forward request to the board
			that->SetMasterMute( NodeDef, OnOff );  

            // write the stuff (with mask!).
			//
            Property_ValidateValues( that, NodeDef, channel, *OnOff);

            ntStatus = STATUS_SUCCESS;
            // ntStatus was set with the write call! whatever this is, return it.
        }
    }
    return ntStatus;
}

/*****************************************************************************
 * CMiniportTopologyICH::BasicSupportHandler
 *****************************************************************************
 * Assists in BASICSUPPORT accesses on level properties.
 * This function is called internally every time there is a "basic support"
 * request on a volume or tone control. The basic support is used to retrieve
 * some information about the range of the control (from - to dB, steps) and
 * which type of control (tone, volume).
 * Basically, this function just calls GetDBValues to get the range information
 * and fills the rest of the structure with some constants.
 */
NTSTATUS CMiniportTopologyICH::BasicSupportHandler
(
    IN      PPCPROPERTY_REQUEST   PropertyRequest
)
{
    PAGED_CODE ();

    ASSERT (PropertyRequest);

    DOUT (DBG_PROPERTY, ("[CMiniportTopologyICH::BasicSupportHandler]"));

    NTSTATUS ntStatus = STATUS_INVALID_DEVICE_REQUEST;
    // The major target is the object pointer to the topology miniport.
    CMiniportTopologyICH *that =
        (CMiniportTopologyICH *) PropertyRequest->MajorTarget;

    ASSERT (that);
	DOUT (DBG_PROPERTY, ("==>Device number : %d ", that->m_sDeviceNumber));


    // if there is enough space for a KSPROPERTY_DESCRIPTION information
    if (PropertyRequest->ValueSize >= (sizeof(KSPROPERTY_DESCRIPTION)))
    {
		// on LX6464ES : no m/c prop handler : no volume control , only mute in fact.
		//
        BOOLEAN IsMultiChannelNode = /*( that->m_wChannelCount > 2) ? 1 :*/ 0;

        WORD    LcMembers =    IsMultiChannelNode ? that->m_wChannelCount : 1;

        // we return a KSPROPERTY_DESCRIPTION structure.
        PKSPROPERTY_DESCRIPTION PropDesc = (PKSPROPERTY_DESCRIPTION)PropertyRequest->Value;

        PropDesc->AccessFlags = KSPROPERTY_TYPE_BASICSUPPORT |
                                KSPROPERTY_TYPE_GET |
                                KSPROPERTY_TYPE_SET;
        PropDesc->DescriptionSize   = sizeof(KSPROPERTY_DESCRIPTION) +
                                      sizeof(KSPROPERTY_MEMBERSHEADER) +
                                      LcMembers*sizeof(KSPROPERTY_STEPPING_LONG);
        PropDesc->PropTypeSet.Set   = KSPROPTYPESETID_General;
        PropDesc->PropTypeSet.Id    = VT_I4;
        PropDesc->PropTypeSet.Flags = 0;
        PropDesc->MembersListCount  = 1;
        PropDesc->Reserved          = 0;

        // if return buffer can also hold a range description, return it too
        if (PropertyRequest->ValueSize >= (sizeof(KSPROPERTY_DESCRIPTION) +
            sizeof(KSPROPERTY_MEMBERSHEADER) + LcMembers*sizeof(KSPROPERTY_STEPPING_LONG)))
        {

            // fill in the members header
            PKSPROPERTY_MEMBERSHEADER Members = (PKSPROPERTY_MEMBERSHEADER)(PropDesc + 1);

            Members->MembersFlags   = KSPROPERTY_MEMBER_STEPPEDRANGES;
            Members->MembersSize    = sizeof(KSPROPERTY_STEPPING_LONG);
            Members->MembersCount   = LcMembers;
            Members->Flags          = IsMultiChannelNode ? KSPROPERTY_MEMBER_FLAG_BASICSUPPORT_MULTICHANNEL : 0;

            // fill in the stepped range
            PKSPROPERTY_STEPPING_LONG Range = (PKSPROPERTY_STEPPING_LONG)(Members + 1);

            for (ULONG i = 0; i <LcMembers; i++) {
                ntStatus = GetDBValues (that->AdapterCommon,
                                    that->TransNodeNrToNodeDef (PropertyRequest->Node),
                                    &(Range[i].Bounds.SignedMinimum),
                                    &(Range[i].Bounds.SignedMaximum),
                                    &(Range[i].SteppingDelta));
            }

            Range->Reserved         = 0;

            // set the return value size
            PropertyRequest->ValueSize = sizeof(KSPROPERTY_DESCRIPTION) +
                                         sizeof(KSPROPERTY_MEMBERSHEADER) +
                                         LcMembers*sizeof(KSPROPERTY_STEPPING_LONG);

            DOUT (DBG_PROPERTY, ("BASIC_SUPPORT: %s max=0x%x min=0x%x step=0x%x",
                NodeStrings[that->TransNodeNrToNodeDef (PropertyRequest->Node)],
                Range->Bounds.SignedMaximum, Range->Bounds.SignedMinimum,
                Range->SteppingDelta));
        } else
        {
            // we hadn't enough space for the range information; 
            // set the return value size
            PropertyRequest->ValueSize = sizeof(KSPROPERTY_DESCRIPTION);
        }

        ntStatus = STATUS_SUCCESS;
    }
    else if (PropertyRequest->ValueSize >= sizeof(ULONG))
    {
        // if return buffer can hold a ULONG, return the access flags
        PULONG AccessFlags = (PULONG)PropertyRequest->Value;

        *AccessFlags = KSPROPERTY_TYPE_BASICSUPPORT |
                       KSPROPERTY_TYPE_GET |
                       KSPROPERTY_TYPE_SET;

        // set the return value size
        PropertyRequest->ValueSize = sizeof(ULONG);
        ntStatus = STATUS_SUCCESS;
    }

    // In case there was not even enough space for a ULONG in the return buffer,
    // we fail this request with STATUS_INVALID_DEVICE_REQUEST.
    // Any other case will return STATUS_SUCCESS.
    return ntStatus;
}

/*****************************************************************************
 * CMiniportTopologyICH::PropertyHandler_Level
 *****************************************************************************
 * Accesses a KSAUDIO_LEVEL property.
 * This function (property handler) is called by portcls every time there is a
 * get, set or basic support request for the node. The connection between the
 * node type and the property handler is made in the automation table which is
 * referenced when you register the node.
 * We use this property handler for all volume controls (and virtual volume
 * controls for recording).
 */
NTSTATUS CMiniportTopologyICH::PropertyHandler_Level
(
    IN      PPCPROPERTY_REQUEST   PropertyRequest
)
{
    PAGED_CODE ();

    ASSERT (PropertyRequest);

    DOUT (DBG_PROPERTY, ("[CMiniportTopologyICH::PropertyHandler_Level]"));

    NTSTATUS        ntStatus = STATUS_INVALID_PARAMETER;
    TopoNodes       NodeDef;
    LONG            channel;

    // The major target is the object pointer to the topology miniport.
    CMiniportTopologyICH *that =
        (CMiniportTopologyICH *) PropertyRequest->MajorTarget;

    ASSERT (that);

    // validate node
    if (PropertyRequest->Node == (ULONG)-1)
    {
        // Ooops
        DOUT (DBG_ERROR, ("PropertyHandler_Level: Invalid node requested."));
        return ntStatus;
    }

    NodeDef = that->TransNodeNrToNodeDef (PropertyRequest->Node);

    // do the appropriate action for the request.

    // we should do a get or a set?
    if ((PropertyRequest->Verb & KSPROPERTY_TYPE_GET) ||
        (PropertyRequest->Verb & KSPROPERTY_TYPE_SET))
    {
        // validate parameters
        if ((PropertyRequest->InstanceSize < sizeof(LONG)) ||
            (PropertyRequest->ValueSize < sizeof(LONG)))
        {
            // Ooops
            DOUT (DBG_ERROR, ("PropertyHandler_Level: Invalid parameter size."));
            return ntStatus;
        }

        // get channel information
        channel = *((PLONG)PropertyRequest->Instance);

        // check channel types, return when unknown
        // as you can see, we have multichannel support.
        if ((channel != CHAN_MASTER) &&
            ((channel < 0) || (channel >= that->m_wChannelCount)) )
        {
            // Ooops
            DOUT (DBG_ERROR, ("PropertyHandler_Level: Invalid channel requested."));
            return ntStatus;
        }

        // get the buffer
        PLONG Level = (PLONG)PropertyRequest->Value;

        // Switch on the node id. This is just for parameter checking.
        // If something goes wrong, we will immideately return with
        // ntStatus, which is STATUS_INVALID_PARAMETER.
        switch(NodeDef = that->TransNodeNrToNodeDef (PropertyRequest->Node))
        {
            // These are stereo channels.
            case NODE_DIGITALIN_VOLUME:
            case NODE_MASTEROUT_VOLUME:

                // check type
                if (PropertyRequest->PropertyItem->Id != KSPROPERTY_AUDIO_VOLUMELEVEL)
                {
                    // Ooops
                    DOUT (DBG_ERROR, ("PropertyHandler_Level: Invalid property requested."));
                    return ntStatus;
                }

                // check channel; we don't support a get with master
                if ((channel == CHAN_MASTER) &&
                    (PropertyRequest->Verb & KSPROPERTY_TYPE_GET))
                {
                    // Ooops
                    DOUT (DBG_ERROR, ("PropertyHandler_Level: we don't support a get with master"));
                    return ntStatus;
                }
                break;

            case NODE_INVALID:
            default:
                // Ooops
				DOUT (DBG_ERROR, ("PropertyHandler_Level: Invalid node requested (%s)\n", NodeStrings[NodeDef]));
                return ntStatus;
        }

        // Now, do some action!

        // do a get
        if (PropertyRequest->Verb & KSPROPERTY_TYPE_GET)
        {
            // $$ FS
	        DOUT (DBG_PROPERTY, ("GET %s ==> %d",NodeStrings[NodeDef],channel));

            // thats all, good bye.
            PropertyRequest->ValueSize = sizeof(LONG);

            // get the register and read it.
            // Note: mono channels are 'master' here (see fake above).
            // this makes sure that left and right channel is prg. for the virt.
            // controls. On controls that only have the right channel, the left
            // channel programming does nothing cause the mask will be zero.

            // read only one channel value to the node cache.
            *Level = that->stMultiNodeCache[NodeDef].lvalue[channel];

            ntStatus = STATUS_SUCCESS ;
        }
        else        // this must be a set
        {
            LONG    lLevel = *Level;

            // $$ FS
	        DOUT (DBG_PROPERTY, ("SET %s ==> %d",NodeStrings[NodeDef],channel));

            //    case NODE_MASTEROUT_VOLUME:
            //    case NODE_DIGITALIN_VOLUME:

			// forward request to the board
			ntStatus = that->SetDigitalLevel( NodeDef, channel, &lLevel );

            // We must always return STATUS_SUCCESS. The GET command will retrieve
            // the actual value set
            //
            ntStatus = STATUS_SUCCESS;

            // write the stuff (with mask!).
            Property_ValidateValues( that, NodeDef, channel, lLevel);

            // ntStatus was set with the read call! whatever this is, return it.
        }
    }
    else
    {
        if (PropertyRequest->Verb & KSPROPERTY_TYPE_BASICSUPPORT)
        {

	        DOUT (DBG_PROPERTY, ("BASICSUPPORT"));

            ntStatus = BasicSupportHandler (PropertyRequest);
        }
    }

    ASSERT(ntStatus==STATUS_SUCCESS);

    return ntStatus;
}

/*****************************************************************************
 * CMiniportTopologyICH::Property_ValidateValues
 *****************************************************************************
* write the stuff (with mask!).
* Note: mono channels are 'master' here (see fake above).
* this makes sure that left and right channel is prg. for the virt.
* controls. On controls that only have the right channel, the left
* channel programming does nothing cause the mask will be zero
*/

void CMiniportTopologyICH::Property_ValidateValues
(
    CMiniportTopologyICH*   PmThat,
    TopoNodes               PmNodeDef,
    LONG                    PmChannel,
    LONG                    PmValue
)
{
    tMultichannelNodeCache* LcNodeCache = &PmThat->stMultiNodeCache[PmNodeDef];

    for (LONG i = 0; i < PmThat->m_wChannelCount; i++)
    {
        if (( i == PmChannel ) || ( PmChannel == CHAN_MASTER)) 
        {
            // write only left value to the node cache.
            LcNodeCache->lvalue[i] = PmValue;
            LcNodeCache->dwValidityMask |= (1 << i);
        }
    }
}	// end of Property_ValidateValues



/*****************************************************************************
 * CMiniportTopologyICH::PropertyHandler_Ulong
 *****************************************************************************
 * Accesses a ULONG value property. For MUX and DEMUX.
 * This function (property handler) is called by portcls every time there is a
 * get, set or basic support request for the node. The connection between the
 * node type and the property handler is made in the automation table which is
 * referenced when you register the node.
 * We use this property handler for all muxer controls.
 */
NTSTATUS CMiniportTopologyICH::PropertyHandler_Ulong
(
    IN      PPCPROPERTY_REQUEST   PropertyRequest
)
{
    PAGED_CODE ();

    ASSERT (PropertyRequest);

    DOUT (DBG_PROPERTY, ("[CMiniportTopologyICH::PropertyHandler_Ulong]"));

    NTSTATUS        ntStatus = STATUS_INVALID_PARAMETER;
    TopoNodes       NodeDef;
 
    // The major target is the object pointer to the topology miniport.
    CMiniportTopologyICH *that =
        (CMiniportTopologyICH *) PropertyRequest->MajorTarget;

    ASSERT (that);

    // validate node instance
    if (PropertyRequest->Node == (ULONG)-1)
        return ntStatus;

    // if we should do a get or set.
    if ((PropertyRequest->Verb & KSPROPERTY_TYPE_GET) ||
        (PropertyRequest->Verb & KSPROPERTY_TYPE_SET))
    {
        // validate buffer size.
        if (PropertyRequest->ValueSize < sizeof(ULONG))
            return ntStatus;

        // get the pointer to the buffer.
        PLONG PropValue = (PLONG)PropertyRequest->Value;

        // Now do some action!
        NodeDef = that->TransNodeNrToNodeDef (PropertyRequest->Node);

        // Switch on the node id. This is just for parameter checking.
        // If something goes wrong, we will immideately return with
        // ntStatus, which is STATUS_INVALID_PARAMETER.
        switch(NodeDef)
        {
            case NODE_INPUT_SELECT:
              // check the type
                if (PropertyRequest->PropertyItem->Id != KSPROPERTY_AUDIO_MUX_SOURCE)
                    return ntStatus;
                break;
                
            case NODE_INVALID:
            default:
                // Ooops
                DOUT (DBG_ERROR, ("PropertyHandler_Ulong: Invalid node requested\n"));
                return ntStatus;
        }

        // should we return the value?
        if (PropertyRequest->Verb & KSPROPERTY_TYPE_GET)
        {
            if( that->stMultiNodeCache[NodeDef].dwValidityMask )
            {
                *PropValue = that->stMultiNodeCache[NodeDef].lvalue[0];
            }
            else
            {
                *PropValue = KSNODEPIN_SUM_MUX_IN;
            }

            // we return a LONG
            PropertyRequest->ValueSize = sizeof(LONG);
            DOUT (DBG_PROPERTY, ("GET: %s = 0x%x", NodeStrings[NodeDef], *PropValue));

            // ntStatus was set with the read call! whatever this is, return it.
            ntStatus = STATUS_SUCCESS ;
        }
        else        // that must be a set
        {
            LONG   lSelect = *PropValue;

            if( (NodeDef != NODE_INPUT_SELECT) || (lSelect != KSNODEPIN_SUM_MUX_IN ) )
			{
				DOUT (DBG_ERROR, ("PropertyHandler_Ulong : cannot change %d to %d\n", NodeDef, lSelect));
                return ntStatus;
			}

            ntStatus = STATUS_SUCCESS ;

            // write the stuff (with mask!).
            // write only left value to the node cache.
            if ( ntStatus == STATUS_SUCCESS )
            {
                that->stMultiNodeCache[NodeDef].lvalue[0] = lSelect;
                that->stMultiNodeCache[NodeDef].dwValidityMask = 0x01;
            }

            DOUT (DBG_PROPERTY, ("SET: %s -> 0x%x (Status = 0x%lx)", NodeStrings[NodeDef],
                    *PropValue, ntStatus));
            // ntStatus was set with the write call! whatever this is, return it.
        }
    }

    return ntStatus;
}

/*****************************************************************************
 * CMiniportTopologyICH::PropertyHandler_CpuResources
 *****************************************************************************
 * Propcesses a KSPROPERTY_AUDIO_CPU_RESOURCES request
 * This property handler is called by the system for every node and every node
 * must support this property. Basically, this property is for performance
 * monitoring and we just say here that every function we claim to have has HW
 * support (which by the way is true).
 */
NTSTATUS CMiniportTopologyICH::PropertyHandler_CpuResources
(
    IN      PPCPROPERTY_REQUEST   PropertyRequest
)
{
    PAGED_CODE ();

    ASSERT (PropertyRequest);

    DOUT (DBG_PROPERTY, ("[CMiniportTopologyICH::PropertyHandler_CpuResources]"));

    CMiniportTopologyICH *that =
        (CMiniportTopologyICH *) PropertyRequest->MajorTarget;
    NTSTATUS ntStatus = STATUS_INVALID_DEVICE_REQUEST;

    ASSERT (that);

    // validate node
    if (PropertyRequest->Node == (ULONG)-1)
        return ntStatus;

    // validate the node def.
    if (that->TransNodeNrToNodeDef (PropertyRequest->Node) == NODE_INVALID)
        return ntStatus;
    
    // we should do a get
    if (PropertyRequest->Verb & KSPROPERTY_TYPE_GET)
    {
        // just return the flag.
        if (PropertyRequest->ValueSize >= sizeof(LONG))
        {
            *((PLONG)PropertyRequest->Value) = KSAUDIO_CPU_RESOURCES_NOT_HOST_CPU;
            PropertyRequest->ValueSize = sizeof(LONG);
            ntStatus = STATUS_SUCCESS;
        }
        else    // not enough buffer.
        {
            ntStatus = STATUS_BUFFER_TOO_SMALL;
        }
    }

    DOUT (DBG_PROPERTY, ("[CMiniportTopologyICH::PropertyHandler_CpuResources] ntStatus = 0x%x",ntStatus));

    return ntStatus;
}


/*****************************************************************************
 * CMiniportTopologyICH::UpdateRecordMute
 *****************************************************************************
 * Updates the record mute control. This is used to have DRM functionality.
 * In the case that we play a DRM file that is copy protected, we have to
 * mute the record if stereo or mono mix is selected. We also have to update
 * the record mute every time the DRM content changes or the playback stream
 * goes away. The property handler also calls this function to update the
 * record mute in case stereo or mono mix is selected.
 */
void CMiniportTopologyICH::SetDrmFlags (BOOL CopyProtect, BOOL DigitalOutputDisable)
{
    PAGED_CODE ();

    DOUT (DBG_PRINT, ("[CMiniportTopologyICH::SetDrmFlags] CopyProtect=%d DigitalOutputDisable=%d", CopyProtect, DigitalOutputDisable));

    if (m_bCopyProtectFlag != CopyProtect)
    {
        m_bCopyProtectFlag = CopyProtect;

		BOOL		LcUseSyncConnector;
		TopoNodes   LcNode;

		LcNode = NODE_DIGITALIN_VOLUME;

        for( LONG i = 0; i < m_wChannelCount; i++ )
        {
            if ( (stMultiNodeCache[LcNode].dwValidityMask & ( 1L << i)) == 0)
            {
                stMultiNodeCache[LcNode].dwValidityMask |= ( 1L << i);
                stMultiNodeCache[LcNode].lvalue[i] = 0;      // 0 dB
            }
            this->SetDigitalLevel( LcNode, i, &stMultiNodeCache[LcNode].lvalue[i] );
        }
    }
    if (m_bDigitalOutputDisable != DigitalOutputDisable)
    {
        m_bDigitalOutputDisable = DigitalOutputDisable;

		TopoNodes   LcNode = NODE_MASTEROUT_MUTE;

		if (stMultiNodeCache[LcNode].dwValidityMask == 0)
        {
            stMultiNodeCache[LcNode].dwValidityMask = 1;
            stMultiNodeCache[LcNode].lvalue[0] = 0;
        }
		this->SetMasterMute( LcNode, (PBOOL)&stMultiNodeCache[LcNode].lvalue[0]);
    }
}



// **********************************************************************************
//
// **********************************************************************************
//
// **********************************************************************************
//
#pragma code_seg()

// These macros are only defined below
//
#define BOARD_LOCK()\
    KIRQL   LcOldBoardIrql; \
    \
	DOUT (DBG_SYSINFO, ("[Prop BOARD_LOCK]"));\
    this->AdapterCommon->AcquireBoardLock(&LcOldBoardIrql);\
	DOUT (DBG_SYSINFO, ("[Prop BOARD LOCKED]"))

#define BOARD_UNLOCK()\
	DOUT (DBG_SYSINFO, ("[Prop BOARD_UNLOCK]"));\
    this->AdapterCommon->ReleaseBoardLock(&LcOldBoardIrql)

#define BOARD_LOCK_AGAIN()\
    DOUT (DBG_SYSINFO, ("[Wave BOARD_LOCK]"));\
    this->AdapterCommon->AcquireBoardLock(&LcOldBoardIrql);\
    DOUT (DBG_SYSINFO, ("[Wave BOARD_LOCKED]"))


/*****************************************************************************
 * CMiniportTopologyICH::SetDigitalLevel
 *****************************************************************************
 * Translate the Set command of the digital level for specified input or output
 * and send the corresponding write order to the board 
 *
 */
NTSTATUS CMiniportTopologyICH::SetDigitalLevel
(
    IN      TopoNodes   PmNode,
    IN      LONG        PmChannel,
    INOUT   PLONG       PmPLevel
)
{
    LEVEL_AUDIO_INFO    LcLevelAudioInfo;
    TARGET_INFO         LcTarget ;
    WORD                LcDriverLevel;
    DWORDLONG           LcAudioMask;
    WORD                LcRet = SUCCESS ;
    BOOLEAN             LcIsInput ;

    LcIsInput = (BOOLEAN)( PmNode >= NODE_WAVEOUT_NUMBER ) ;

    // Does output pipe exist ?
    //
    if ( ( !LcIsInput ) && ( !m_bIsOutputPipeReady ) )
    {
        return STATUS_UNSUCCESSFUL;
    }
    // Does input pipe exist ?
    //
    if ( LcIsInput && ( !m_bIsInputPipeReady ) )
    {
        return STATUS_UNSUCCESSFUL;
    }

    CProtocol*	Lc_pDsp	=	(CProtocol*)AdapterCommon->GetProtocolPtrAsVoid();

    LcDriverLevel = AdapterCommon->ConvertDSound2Driver( *PmPLevel, PmNode );

    DOUT (DBG_PROPERTY, ("[CMiniportTopologyICH::SetDigitalLevel %d dB = 0x%x driver ]", (*PmPLevel)>>16, LcDriverLevel));

    // Specify target audio channel
    //
    RtlZeroMemory( &LcTarget, sizeof(LcTarget) );
    LcTarget.tgCarte = m_wBoardIndex ;
    LcTarget.tgPipe = (WORD) m_wChannelNumber ;
    LcTarget.tgCaracPipeVoie = LcIsInput ? OPER_REC : OPER_PLAY ;

    if ( PmChannel == CHAN_MASTER )
    { 
       LcAudioMask = Channels2AudioMask(m_wChannelCount);
    }
    else
    {
       LcAudioMask = UTIMask64bit(PmChannel);
    }

	LcAudioMask <<= m_wChannelNumber;

    BOARD_LOCK();

    // Specify target level settings
    //
    RtlZeroMemory( &LcLevelAudioInfo, sizeof(LcLevelAudioInfo) );
    LcLevelAudioInfo.gaiHasDigitalLevel = TRUE;
    LcLevelAudioInfo.gaiDigitalLevel = LcDriverLevel;

    LcLevelAudioInfo.gaiHasMuteLevel = TRUE;
    if( LcIsInput )
    {
        LcLevelAudioInfo.gaiMuteLevel = (WORD)( m_bCopyProtectFlag != FALSE );
    }
    else
    {
        LcLevelAudioInfo.gaiMuteLevel = (WORD)( m_bDigitalOutputDisable != FALSE );
    }

    // if this audio is not controlled by DHS
    if( APHChkIsAudioDHSControlled(m_wBoardIndex, m_wChannelNumber, LcIsInput) == FALSE )
    {
        // Make the settings effective on the board
        //
        LcRet = Lc_pDsp->ILevel_AudioSetDigitalLevel( &LcTarget, LcAudioMask, &LcLevelAudioInfo );
    }
    else
    {
        LcRet = ED_INVALID_PIPE_AUDIO;
    }

    BOARD_UNLOCK();

    // Read back the actual value we've just set
    //
    if ( LcRet == SUCCESS )
    {
        return STATUS_SUCCESS;
    }

    return STATUS_UNSUCCESSFUL;
}


/*****************************************************************************
 * CMiniportTopologyICH::SetMasterMute
 *****************************************************************************
 * Translate the Set command of the mute for specified output
 * and send the corresponding write order to the board 
 *
 */
NTSTATUS CMiniportTopologyICH::SetMasterMute
(
    IN      TopoNodes   PmNode,
    INOUT   PBOOL       PmPOnOff
)
{
    LEVEL_AUDIO_INFO    LcLevelAudioInfo;
    TARGET_INFO         LcTarget ;
    WORD                LcRet = SUCCESS ;
    DWORDLONG           LcConsolidatedChannelMask  = 0;


    // should not happen
    //
    if ( PmNode >= NODE_WAVEOUT_NUMBER ) 
        return STATUS_UNSUCCESSFUL;

    DOUT (DBG_PROPERTY, ("[CMiniportTopologyICH::SetMasterMute %s]", *PmPOnOff ? "ON" : "off"));
    
    // Does output pipe exist ?
    //
    if ( !m_bIsOutputPipeReady )  {
        DOUT (DBG_ERROR, ("SetMasterMute: OutputPipe not ready."));
        return STATUS_UNSUCCESSFUL;
    }

    // Specify target audio type
    //
    RtlZeroMemory( &LcTarget, sizeof(LcTarget) );
    LcTarget.tgCaracPipeVoie = OPER_PLAY ;

    // Specify target level settings
    //
    RtlZeroMemory( &LcLevelAudioInfo, sizeof(LcLevelAudioInfo) );
    LcLevelAudioInfo.gaiHasMuteLevel = TRUE;

	// DRM : muter la sortie SPDIF; pas besoin de muter la sortie Analog,
	// pour l'instant pas de separation possible -> mute all

    LcLevelAudioInfo.gaiMuteLevel = (WORD)((*PmPOnOff) || m_bDigitalOutputDisable);

	CProtocol* Lc_pDsp	= (CProtocol*)AdapterCommon->GetProtocolPtrAsVoid();

    LcConsolidatedChannelMask = ((DWORDLONG)Channels2AudioMask(m_wChannelCount)) << m_wChannelNumber;

    BOARD_LOCK();

    // if this audio is not controlled by DHS
    if( APHChkIsAudioDHSControlled(m_wBoardIndex, m_wChannelNumber, FALSE) == FALSE )
    {
        // Make the settings effective on the board
        //
        LcRet =  Lc_pDsp->ILevel_AudioSetDigitalLevel( &LcTarget, LcConsolidatedChannelMask, &LcLevelAudioInfo );
    }
    else
    {
        LcRet = ED_INVALID_PIPE_AUDIO;
    }

    BOARD_UNLOCK();

    // Read back the actual value we've just set
    //
    if ( LcRet == SUCCESS )
    {
		return SUCCESS;
    }

    return STATUS_UNSUCCESSFUL;
}
