// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

// Include files
// *************

#define NTBUFFER_NTDDK_H

#include "pcxerr_e.h"

#include "drvdef.h"

#include "buff_hdl.h"

#include "ntbuffer.h"

#include "drvdbg.h"

#undef NTBUFFER_NTDDK_H


// Local constants, types, macros definitions
// ******************************************

#define DEV_PHYSICAL_MEMORY L"\\Device\\PhysicalMemory"

// TAG_DEBUG_DRIVER : DEB

// We must retain the physical address of buffers
// and whether they are mapped or not in a user space
// --------------------------------------------------
typedef struct _PHYS_MAP_INFO_
{
    PHYSICAL_ADDRESS    pmPhysAddr  ;   // the physical address of the buffer
    BOOLEAN             pmIsMapped  ;   // Is buffer mapped in a user space ?
    BOOLEAN             pmIsReserved;   // Is buffer reserved by a process ?
    HANDLE              pmProcess   ;   // The process where the buffer is mapped into
    DWORD               pmSize      ;   // size of buffer
    PVOID               pmVirtAddr  ;   // the mapped virtual address
} PHYS_MAP_INFO, *PPHYS_MAP_INFO ;

// TAG_DEBUG_DRIVER : FIN

// Static variables
// ****************

STATIC PHYS_MAP_INFO TbPhysMapInfo[MAX_BUFFER];


// Static functions
// ****************

// ****************************************************************************
// Static functions' bodies
// ****************************************************************************

// ****************************************************************************
// STATIC PVOID             NTBProceedMap()
// ********************************
//
// Input parameters :
// ******************
//
//      DWORD    PmBufferIndex   :   the index of the buffer
//
// Return Value :
// **************
//
//  the virtual address mapped into user address space
//
// ****************************************************************************
//
//    This routine maps the DMA buffer into the caller's
// address space.  Must pass a PHYSICAL address, not a
// kernel virtual address.  Generally follows example given
// by MAPMEM sample code in DDK.
//
// ****************************************************************************
STATIC PVOID NTBProceedMap(
    IN  DWORD       PmBufferIndex,
    OUT LPDWORD     PmBufferSize )
{
    UNICODE_STRING      LcUniNameString ;
    OBJECT_ATTRIBUTES   LcAttrib        ;
    HANDLE              LcHdlPhysMem    = 0             ;
    HANDLE              LcCaller        ;
    PVOID               LcMemObject     = 0;
    PHYSICAL_ADDRESS    LcPhysicalAddress, LcViewBase   ;
    PVOID               LcUserAddress   = NULL          ;
    PPHYS_MAP_INFO      LcPhysMapInfo   ;
    SIZE_T              LcLength        ;

    // Let's check whether the buffer is already mapped or not
    // for the current process
    // -------------------------------------------------------
    LcPhysMapInfo = &(TbPhysMapInfo[PmBufferIndex]);

    *PmBufferSize = LcPhysMapInfo->pmSize;

    LcLength = LcPhysMapInfo->pmSize;

    LcCaller = IoGetCurrentProcess();

    DOUT(DBG_BUFFER, ("Process=%x IsMapped ?%x", LcCaller, LcPhysMapInfo->pmIsMapped));
    if ( LcPhysMapInfo->pmIsMapped )
    {
        DOUT(DBG_BUFFER, ("   by %x @=%x", LcPhysMapInfo->pmProcess, LcPhysMapInfo));
    }

    if ( LcPhysMapInfo->pmIsMapped )
    {
        if ( LcPhysMapInfo->pmProcess == LcCaller )
        {
            DOUT(DBG_BUFFER, ("Virt @= %x", LcPhysMapInfo->pmVirtAddr));
            return( LcPhysMapInfo->pmVirtAddr );
        }
        else
        {
            DOUT(DBG_BUFFER, ("NULL"));
            return( NULL );
        }
    }

    LcPhysicalAddress = LcPhysMapInfo->pmPhysAddr ;

    RtlInitUnicodeString( &LcUniNameString, DEV_PHYSICAL_MEMORY );

    InitializeObjectAttributes(
        &LcAttrib,
        &LcUniNameString,
        OBJ_CASE_INSENSITIVE,
        (HANDLE) NULL,
        (PSECURITY_DESCRIPTOR) NULL);

    if ( !NT_SUCCESS(ZwOpenSection(
                            &LcHdlPhysMem       ,
                            SECTION_ALL_ACCESS  ,
                            &LcAttrib           )) )
    {
        DOUT(DBG_BUFFER, ("Open section failed"));
        return NULL ;
    }

    DOUT(DBG_BUFFER, ("ZwMap: IRQL=%d", KeGetCurrentIrql() ));

    if ( !NT_SUCCESS(ObReferenceObjectByHandle(
                            LcHdlPhysMem        ,
                            SECTION_ALL_ACCESS  ,
                            (POBJECT_TYPE) NULL ,
                            KernelMode          ,
                            &LcMemObject        ,
                            (POBJECT_HANDLE_INFORMATION) NULL)) )
    {
        DOUT(DBG_BUFFER, ("Reference failed"));
        goto close;
    }

    LcViewBase = LcPhysicalAddress ;

    if ( !NT_SUCCESS(ZwMapViewOfSection(
                            LcHdlPhysMem,
                            (HANDLE) -1,    // map into caller's address space
                            &LcUserAddress,
                            0L,
                            LcLength,
                            &LcViewBase,
                            &LcLength,
                            ViewShare,
                            0,
                            PAGE_READWRITE )))
    {
        DOUT(DBG_BUFFER, ("Map failed;  Virt @= %x", LcPhysMapInfo->pmVirtAddr ));
        goto close;
    }

    (ULONG_PTR) LcUserAddress += (ULONG_PTR)(LcPhysicalAddress.QuadPart - LcViewBase.QuadPart);

    DOUT(DBG_BUFFER, ("Register=%x", LcCaller ));

    LcPhysMapInfo->pmProcess    = LcCaller      ;
    LcPhysMapInfo->pmVirtAddr   = LcUserAddress ;
    LcPhysMapInfo->pmIsMapped   = TRUE          ;

close:

    ZwClose( LcHdlPhysMem );

    return( LcUserAddress );

}

// ****************************************************************************
// STATIC VOID             NTBProceedUnMap()
// *****************************************
//
// Input parameters :
// ******************
//
//      DWORD    PmBufferIndex   :   the index of the buffer to unmap
//
// ****************************************************************************
//
//    This routine unmaps the DMA buffer from the caller's
// address space.
//
// ****************************************************************************
STATIC VOID NTBProceedUnMap(
    IN DWORD     PmBufferIndex )
{
    PPHYS_MAP_INFO          LcPhysMapInfo   ;

    LcPhysMapInfo = &(TbPhysMapInfo[PmBufferIndex]) ;

    if ( !LcPhysMapInfo->pmIsReserved ) return ;

    if ( !LcPhysMapInfo->pmIsMapped )
    {
        LcPhysMapInfo->pmIsReserved = FALSE ;
        return ;
    }

    // The unmapping must be done further on when running
    // at IRQL_PASSIVE_LEVEL
    // --
    // ## FS (21/12/1998) -- FA #237 Memory leak due to 
    // unmapping failure
    //
    // ## MBR (16/02/2005) -- only if it was the same process !
    // otherwise memory of calling process can get corrupted
    //
	if(LcPhysMapInfo->pmProcess == IoGetCurrentProcess())
	{
		ZwUnmapViewOfSection(
                            (HANDLE) -1,    // same value as ZwMap call
                             LcPhysMapInfo->pmVirtAddr  );
	}
	else
	{
		DOUT(DBG_BUFFER, ("DO NOT CALL ZwUnmapViewOfSection for address %p\n", LcPhysMapInfo->pmVirtAddr));
	}

    // resets the entry
    // NB: a buffer is not expected to be shared
    // by processes
    // -----------------------------------------
    // LOG_CSTR("Clearing @="); LOG_NSTL(LcPhysMapInfo, 16 );

    LcPhysMapInfo->pmIsMapped   =  FALSE           ;
    LcPhysMapInfo->pmProcess    =  NULL            ;
    LcPhysMapInfo->pmIsReserved =  FALSE           ;
}

// ****************************************************************************
// Exported functions' bodies
// ****************************************************************************


// ****************************************************************************
// VOID             NTBInit()
// **************************************
//
// Input parameters :
// ******************
//
// Output parameters :
// ******************
//
// ****************************************************************************
//
//    This routine proceeds to all initializations needed
//
// ****************************************************************************
VOID NTBInit(
    VOID )
{
    BZERO2( TbPhysMapInfo, PHYS_MAP_INFO, MAX_BUFFER);
}


// ****************************************************************************
// PVOID             NTBGetPhysicalAddress()
// *****************************************
//
// Input parameters :
// ******************
//
//      WORD            PmBufferIndex :   the index of the buffer
//
// Output parameters :
// ******************
//
// Return Value :
// **************
//
//  the physical address of this buffer
//
// ****************************************************************************
//
//    This routine gives the physical address to be used in a DMA transfer
//
// ****************************************************************************
DWORD NTBGetBufferPhysicalAddress(
    IN  DWORD            PmBufferIndex   )
{
    if ( TbPhysMapInfo[PmBufferIndex].pmIsReserved == FALSE )
    {
        DOUT(DBG_BUFFER, ("!! UnReserved buffer !!"));
        return( 0 );
    }
    else
    {
        return( TbPhysMapInfo[PmBufferIndex].pmPhysAddr.LowPart );
    }
}


DWORD NTBGetBufferSize( IN DWORD PmBufferIndex )
{
    return( TbPhysMapInfo[PmBufferIndex].pmSize );
}


//******************************************************************************
// WORD NTBGetBufferInfoForDebug(...)
//
// INPUT PARAMETERS     :
//***********************
//
//  WORD PmIndex : the index of the buffer in the array TbPhysMapInfo.
//
// OUTPUT PARAMETERS    :
//***********************
//
//  PBYTE *PmPPPhysMapInfo : the address the record describing the buffer.
//  PWORD PmPSize          : the size of the record.
//
// RETURN VALUE         :
//***********************
//
//  An error code.
//
//******************************************************************************
// Return the record describing a buffer.
//******************************************************************************
//
EXTERN WORD NTBGetPhysMapInfoForDebug(
    IN  WORD    PmIndex,
    OUT PBYTE   *PmPPPhysMapInfo,
    OUT PWORD   PmPSize )
{
    WORD LcRet = SUCCESS;

    if ( PmIndex >= MAX_BUFFER )
    {
        LcRet = ED_INVALID_ADDRESS;
    }
    else
    {
        *PmPPPhysMapInfo = (PBYTE) ( TbPhysMapInfo + PmIndex );
        *PmPSize         = sizeof( PHYS_MAP_INFO );
    }

    return( LcRet );
}


// ****************************************************************************
// PVOID             NTBMapBuffer()
// ********************************
//
// Input parameters :
// ******************
//
//      WORD            PmBufferIndex :   the index of the buffer
//
// Output parameters :
// ******************
//
//      LPADD_DESC_INFO PmAddrDescPtr :   pointer onto the address
//                                      descriptor
//
// Return Value :
// **************
//
//  the virtual address mapped into user address space if already done
//
// ****************************************************************************
//
//    This routine stores a request for further mapping of a buffer
//
// ****************************************************************************
PVOID NTBMapBuffer(
    IN  DWORD           PmBufferIndex   ,
    OUT LPADD_DESC_INFO PmAddrDescPtr   )
{
    PPHYS_MAP_INFO      LcPhysMapInfo   ;

    LcPhysMapInfo = &(TbPhysMapInfo[PmBufferIndex]);

    DOUT(DBG_BUFFER, ("NTBMapBuffer"));

    if ( ! LcPhysMapInfo->pmIsReserved )
    {
        // Init the entry
        // --------------
        LcPhysMapInfo->pmIsMapped   =  FALSE           ;
        LcPhysMapInfo->pmVirtAddr   =  NULL            ;
        LcPhysMapInfo->pmProcess    =  NULL            ;
        LcPhysMapInfo->pmIsReserved =  TRUE            ;
    }

    // Store a request in the reply block
    // ----------------------------------
    PmAddrDescPtr->adAddr1 = 0L ;
    PmAddrDescPtr->adAddr2 = POST_MAP_BUFFER ;
    PmAddrDescPtr->adAddr3 = PmBufferIndex ;

    return( NULL );
}

// ****************************************************************************
// VOID             NTBUnMapBuffer()
// ********************************
//
// Input parameters :
// ******************
//
//      WORD    PmBufferIndex   :   the index of the buffer to unmap
//
// Output parameters :
// ******************
//
//      LPADD_DESC_INFO PmAddrDescPtr :   pointer onto the address
//                                      descriptor
//
// ****************************************************************************
//
//    This routine stores a request for further unmapping of a buffer
//
// ****************************************************************************
VOID NTBUnMapBuffer(
    IN  DWORD           PmBufferIndex   ,
    OUT LPADD_DESC_INFO PmAddrDescPtr   )
{
    PPHYS_MAP_INFO      LcPhysMapInfo   ;

    LcPhysMapInfo = &(TbPhysMapInfo[PmBufferIndex]) ;

    // Do nothing if nothing to do ;-)
    // ---------------------------------
    if ( ! LcPhysMapInfo->pmIsReserved )
    {
        if ( PmAddrDescPtr != NULL )
        {
            // Erase pending map request if any
            // --------------------------------
            PmAddrDescPtr->adAddr1 = 0L ;
            PmAddrDescPtr->adAddr2 = POST_NO_OPERATION ;
            PmAddrDescPtr->adAddr3 = 0L ;
        }

        return ;
    }

    if ( ! LcPhysMapInfo->pmIsMapped )
    {
        LcPhysMapInfo->pmIsReserved = FALSE ;

        if ( PmAddrDescPtr != NULL )
        {
            // Erase pending map request if any
            // --------------------------------
            PmAddrDescPtr->adAddr1 = 0L ;
            PmAddrDescPtr->adAddr2 = POST_NO_OPERATION ;
            PmAddrDescPtr->adAddr3 = 0L ;
        }

        return ;
    }
    else
    {
        if ( PmAddrDescPtr != NULL )
        {
            // Otherwise store a request for further
            // unmapping in reply block
            // ---------------------------------
            PmAddrDescPtr->adAddr1 = 0L ;
            PmAddrDescPtr->adAddr2 = POST_UNMAP_BUFFER ;
            PmAddrDescPtr->adAddr3 = PmBufferIndex ;
        }
    }

    DOUT(DBG_BUFFER, ("BufUnMap d2"));
}

// ****************************************************************************
// NTSTATUS         NTBAllocateAllBuffers()
// **********************************
//
// Input parameters :
// ******************
//
//      PGENERAL_INFO   PmGeneralInfo   :   the list of boards and resources
//                                       to manage
//
// Return Value :
// **************
//
//      STATUS_SUCCESS if allocation has been completed successfully,
//      STATUS_NO_MEMORY otherwise
//
// ****************************************************************************
//
// Try to reserve memory for pseudo-DMA transfer buffers
//
// ****************************************************************************
NTSTATUS NTBAllocateAllBuffers(
    IN  PGENERAL_INFO       PmGeneralInfo   )
{
    PHYSICAL_ADDRESS        LcPhysicalLimit ;
    ULONG                   i               ;
    ULONG                   LcNumBuffersTotal;
    ULONG                   LcNumBigBuffers ;
    WORD                    LcAllocated     = 0 ;
    PVOID                   LcBufferAddress ;
    PBUFFER_INFO            LcBufferInfo    = PmGeneralInfo->PBufferInfo    ;
    PGEN_BUFFER_INFO        LcGeneralBuffer = PmGeneralInfo->PGeneralBuffer ;
    PPHYS_MAP_INFO          LcPhysMapInfo  = &(TbPhysMapInfo[0])          ;
    PHANDLE_BUFFER          LcPtr           ;
    WORD                    LcRet           ;
    DWORD                   BuffSize        ;

    DOUT(DBG_BUFFER, ("Entering Allocate Buffers"));

    LcPhysicalLimit.HighPart = 0xFFFFFFFF ;
    LcPhysicalLimit.LowPart  = 0xFFFFFFFF ;

    LcNumBuffersTotal = LcGeneralBuffer->gbBuffTotalNb;
    LcNumBigBuffers = LcGeneralBuffer->gbBigBuffNb;

    BuffSize = LcGeneralBuffer->gbBuffSize;

    DOUT(DBG_BUFFER, ("Alloc %d buffers of size = %x (%d Ko)", LcNumBuffersTotal, BuffSize, BuffSize/1024));
    DOUT(DBG_BUFFER, ("MmAllocate: IRQL= %d", KeGetCurrentIrql()));

    // Allocate buffers for the DMA data.  According to
    // DDK, we can only do this while initializing driver.
    // --------------------------------------------------
    for ( i = 0 ; i < LcNumBuffersTotal ; i++ )
    {
        // the last buffers are the big ones
        if( i >= (LcNumBuffersTotal - LcNumBigBuffers) )
        {
            BuffSize = LcGeneralBuffer->gbBigBuffSize;
        }

        // Allocate buffers, one by one in contiguous memory
        // in order to get a chance to allocate them all
        // --------------------------------------------------
        LcBufferAddress = MmAllocateContiguousMemory(
                                                    BuffSize        ,
                                                    LcPhysicalLimit );
        if ( LcBufferAddress != 0 )
        {
            // Set the buffer info entry accordingly
            // -------------------------------------
            LcBufferInfo->biCarac    = 1 ;

            // Virtual address (in kernel space)
            LcBufferInfo->biAddress1 = LcBufferAddress;

            // Physical address
            LcPhysMapInfo->pmPhysAddr = MmGetPhysicalAddress(LcBufferAddress);
            LcPhysMapInfo->pmIsMapped = FALSE ;
            LcPhysMapInfo->pmSize     = BuffSize;

            DOUT(DBG_BUFFER, ("Size = %d  Virtual Address = %p  Physical Address = %x", BuffSize, LcBufferInfo->biAddress1, LcPhysMapInfo->pmPhysAddr.LowPart));

            LcBufferInfo++      ;
            LcPhysMapInfo++    ;
            LcAllocated++       ;
        }
        else
        {
            DOUT(DBG_BUFFER, ("MmAllocateContiguousMemory failed"));
            break ;
        }
    }

    DOUT(DBG_BUFFER, ("Actually allocated: %d", LcAllocated));

    LcGeneralBuffer->gbBuffAvailNb = LcAllocated ;
    LcGeneralBuffer->gbBuffTotalNb = LcAllocated ;

    // Clear all other physical map entries
    // -------------------------------------
    if (LcAllocated < MAX_BUFFER)
    {
        ULONG   LcSize ;

        LcSize = (ULONG) MAX_BUFFER - LcAllocated ;
        LcSize *= sizeof(PHYS_MAP_INFO) ;

        RtlZeroMemory(
                    &(TbPhysMapInfo[LcAllocated])   ,
                    LcSize                          );
    }

    if (LcAllocated != 0)
    {
        DOUT(DBG_BUFFER, ("Alloc success"));
        return STATUS_SUCCESS ;
    }

    return STATUS_NO_MEMORY ;
}


// ****************************************************************************
// VOID NTBFreeAllBuffers()
// *********************************
//
// Input parameters :
// ******************
//
//      PGENERAL_INFO   PmGeneralInfo   :   the list of boards and resources
//                                       to manage
//
// ****************************************************************************
//
// Try to reserve memory for pseudo-DMA transfer buffers
//
// ****************************************************************************
VOID NTBFreeAllBuffers(
    IN  PGENERAL_INFO       PmGeneralInfo   )
{
    NTSTATUS                LcNtStatus      = STATUS_SUCCESS;
    ULONG                   i               ;
    ULONG                   LcNumBuffers    ;
    PBUFFER_INFO            LcBufferInfo    = PmGeneralInfo->PBufferInfo    ;
    PGEN_BUFFER_INFO        LcGeneralBuffer = PmGeneralInfo->PGeneralBuffer ;
    PPHYS_MAP_INFO          LcPhysMapInfo   = &(TbPhysMapInfo[0])           ;

    DOUT(DBG_BUFFER, ("NTBFree: IRQL= %d", KeGetCurrentIrql()));

    if (KeGetCurrentIrql() > PASSIVE_LEVEL)
    {
        DOUT(DBG_BUFFER, ("Cannot free, IRQL > PASSIVE_LEVEL"));
    }

    LcNumBuffers = MAX_BUFFER ;

    // Release all buffers for the DMA data.
    // -------------------------------------
    for ( i = 0 ; i < LcNumBuffers ; i++ )
    {
        if ( LcBufferInfo->biAddress1 == NULL ) continue ;

        // actually release physical memory
        // --------------------------------
        MmFreeContiguousMemory( LcBufferInfo->biAddress1 );

        LcPhysMapInfo->pmPhysAddr.QuadPart = 0;

        // Reset the entry
        // ---------------
        LcBufferInfo->biCarac    = 0 ;
        LcBufferInfo->biAddress1 = NULL ;

        LcBufferInfo ++ ;
        LcPhysMapInfo ++;
    }
}


// ****************************************************************************
// DWORD            NTBProceedMappings()
// **************************************
//
// Input parameters :
// ******************
//
//  PPOST_MAPPING_INFO  PmMapInfos  :   pointers onto mapping requests
//
// Output parameters :
// ******************
//
//  DWORD               PmReplyBlk  :   the reply block (for mappings)
//
// ****************************************************************************
//
//    This routine proceeds to all (un)mapping requests posted
// by a driver command and eventually sets the reply block accordingly
//
// For each address descriptor:
//  In:
//          adAddr1 = the total number of pending requests (first descriptor)
//          adAddr2 = pending operation for buffer
//          adAddr3 = buffer index in driver tabs
//  Out:
//          adAddr1 = user virtual address
//          adAddr2 = adAddr3 = 0
//
// ****************************************************************************
DWORD NTBProceedMappings(
    IN  DWORD               PmNbBuffers     ,
    OUT LPADD_DESC_INFO     PmAddrDescrTab  )
{
    DWORD           i ;
    DWORD           LcNbBuffers ;
    PVOID           LcUserAddress ;
    DWORD           LcBufferSize  ;

    // If not specified, read the buffer number from the
    // reply block
    // -------------------------------------------------
    LcNbBuffers = PmNbBuffers ;

    if ( LcNbBuffers == 0 )
    {
        LcNbBuffers = PmAddrDescrTab[0].adAddr1 ;
    }

    // Scan the MapInfos table and lookup for posted request
    // -----------------------------------------------------
    for ( i = 0 ; ( i < MAX_BUFFER ) && ( i < LcNbBuffers ) ; i++ )
    {

        // Dispatch requests onto the right
        // procedure, and fill in reply block
        // if necessary
        // -----------------------------------
        switch ( PmAddrDescrTab[i].adAddr2 )
        {
        case POST_NO_OPERATION :

            break ;

        case POST_MAP_BUFFER :

            LcBufferSize = 0;

            LcUserAddress = NTBProceedMap( PmAddrDescrTab[i].adAddr3, &LcBufferSize );

            // stop mapping buffers if not more possible
            if (LcUserAddress == NULL)
                return i;

            // Fill in the reply block
            // -----------------------
            PmAddrDescrTab[i].adAddr1 = (DWORD) LcUserAddress ;
            PmAddrDescrTab[i].adAddr2 = 0 ;
            PmAddrDescrTab[i].adAddr3 = LcBufferSize ;  // return the size mapped for this buffer in adAddr3

            break ;

        case POST_UNMAP_BUFFER :

            NTBProceedUnMap( PmAddrDescrTab[i].adAddr3 );

            // nothing to do with reply block
            // ------------------------------

            break ;

        default :
            DOUT(DBG_BUFFER, ("Unhandled request"));
        }
    }

    // return buffers mapped/unmapped
    return i;
}

