// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//  Portions Copyright (c) 1998-1999 Intel Corporation
//  Portions Copyright (c) 1998-1999 Microsoft Corporation. 
//
// **************************************************************************

#include "shared.h"

#include "xx_protocol.h"

#include "minwave.h"
#include "ichwave.h"

#include "pcxerr_e.h"

#include "lxeswdm.h"
#include "es_cmds.h"

#define MIN( a, b )     ( (a) < (b) ? (a) : (b) )


// datarange a-la SoundBlaster
/*****************************************************************************
 * PinDataRangesStreamStatic
 *****************************************************************************
 * Structures indicating range of valid format values for streaming pins.
 */
static  KSDATARANGE_AUDIO PinDataRangesStreamStatic[64];
#define DR_IDX_INVALID      64
/*****************************************************************************
 * PinDataRangePointersStreamStatic
 *****************************************************************************
 * List of pointers to structures indicating range of valid format values
 * for streaming pins.
 */
static  PKSDATARANGE PinDataRangePointersStreamStatic[64];

static  BOOL PinDataRangesInit = FALSE;


/*****************************************************************************
 * PinDataRangePointerAnalogStream
 *****************************************************************************
 * This structure pointers to the data range structures for the wave pins.
 */

static KSDATARANGE PinDataRangesAnalogBridge[] =
{
   {
      sizeof(KSDATARANGE),
      0,						// ignored
      0,						// ignored
      0,						// reserved
      STATICGUIDOF(KSDATAFORMAT_TYPE_AUDIO),
      STATICGUIDOF(KSDATAFORMAT_SUBTYPE_ANALOG),
      STATICGUIDOF(KSDATAFORMAT_SPECIFIER_NONE)
   }
};

static PKSDATARANGE PinDataRangePointersAnalogBridge[] =
{
    (PKSDATARANGE)&PinDataRangesAnalogBridge[0]
};


/*****************************************************************************
 * Wave Miniport Topology
 *========================
 *
 *                              +-----------+
 *                              |           |
 *    Capture (PIN_WAVEIN)  <---|3 --ADC-- 2|<=== (PIN_WAVEIN_BRIDGE)
 *                              |           |
 *     Render (PIN_WAVEOUT) --->|0 --DAC-- 1|===> (PIN_WAVEOUT_BRIDGE)
 *                              |           |
 *                              |           |
 *                              +-----------+
 *
 */

#pragma code_seg("PAGE")
/*****************************************************************************
 * CreateMiniportWaveICH
 *****************************************************************************
 * Creates a ICH wave miniport object for the ICH adapter.
 * This uses a macro from STDUNK.H to do all the work.
 */
NTSTATUS CreateMiniportWaveICH
(
    OUT PUNKNOWN   *Unknown,
    IN  REFCLSID,
    IN  PUNKNOWN    UnknownOuter    OPTIONAL,
    IN  POOL_TYPE   PoolType
)
{
    PAGED_CODE ();

    ASSERT (Unknown);

    DOUT (DBG_PRINT, ("[CreateMiniportWaveICH]"));

    STD_CREATE_BODY_WITH_TAG_(CMiniportWaveICH,Unknown,UnknownOuter,PoolType, PCX_MINWAVE_TAG,
                     PMINIPORTWAVEICH);
}


/*****************************************************************************
 * CMiniportWaveICH::NonDelegatingQueryInterface
 *****************************************************************************
 * Obtains an interface.  This function works just like a COM QueryInterface
 * call and is used if the object is not being aggregated.
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICH::NonDelegatingQueryInterface
(
    IN  REFIID  Interface,
    OUT PVOID  *Object
)
{
    PAGED_CODE ();

    ASSERT (Object);

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::NonDelegatingQueryInterface]"));

    // Is it IID_IUnknown?
    if (IsEqualGUIDAligned (Interface, IID_IUnknown))
    {
        *Object = (PVOID)(PUNKNOWN)(PMINIPORTWAVEPCI)this;
    }
    // or IID_IMiniport ...
    else if (IsEqualGUIDAligned (Interface, IID_IMiniport))
    {
        *Object = (PVOID)(PMINIPORT)this;
    }
    // or IID_IMiniportWavePci ...
    else if (IsEqualGUIDAligned (Interface, IID_IMiniportWavePci))
    {
        *Object = (PVOID)(PMINIPORTWAVEPCI)this;
    }
    // or IID_IPowerNotify ...
    else if (IsEqualGUIDAligned (Interface, IID_IPowerNotify))
    {
        *Object = (PVOID)(PPOWERNOTIFY)this;
    }
    // or maybe our IID_IMiniportWaveICH ...
    else if (IsEqualGUIDAligned (Interface, IID_IMiniportWaveICH))
    {
        *Object = (PVOID)(PMINIPORTWAVEICH)this;
    }
    else
    {
        // nothing found, must be an unknown interface.
        *Object = NULL;
        return STATUS_INVALID_PARAMETER;
    }

    //
    // We reference the interface for the caller.
    //
    ((PUNKNOWN)(*Object))->AddRef();
    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICH::~CMiniportWaveICH
 *****************************************************************************
 * Destructor.
 */
CMiniportWaveICH::~CMiniportWaveICH ()
{
    PAGED_CODE ();

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::~CMiniportWaveICH]"));

    //
    // Release the DMA channel.
    //
    if (m_DmaChannel)
    {
        m_DmaChannel->Release ();
        m_DmaChannel = NULL;
    }

    //
    // Release the topology.
    //
    if (m_pTopology)
    {
        m_pTopology->Release ();
        m_pTopology = NULL;
    }

    //
    // Release the interrupt sync.
    //
    if (m_InterruptSync)
    {
        DOUT (DBG_PRINT, ("### CMiniportWaveICH  m_InterruptSync->Release()"));
        m_InterruptSync->Release ();
        m_InterruptSync = NULL;
    }

    //
    // Release the service group.
    //
    if (m_pServiceGroupWave)
    {
        m_pServiceGroupWave->Release ();
        m_pServiceGroupWave = NULL;
    }

    //
    // Release adapter common object.
    //
    if (m_AdapterCommon)
    {
        m_AdapterCommon->Release ();
        m_AdapterCommon = NULL;
    }

    //
    // Release the port.
    //
    if (m_Port)
    {
        m_Port->Release ();
        m_Port = NULL;
    }

    if (FilterDescriptor)
    {
        DDK_FREE_WITH_TAG (FilterDescriptor,PCX_MINWAVE_TAG);
        FilterDescriptor = NULL;
    }

    if (ConnectionDescriptors)
    {
        DDK_FREE_WITH_TAG (ConnectionDescriptors,PCX_MINWAVE_TAG);
        ConnectionDescriptors = NULL;
    }

    if (NodeDescriptors)
    {
        DDK_FREE_WITH_TAG (NodeDescriptors,PCX_MINWAVE_TAG);
        NodeDescriptors = NULL;
    }

    if (PinDescriptors)
    {
        DDK_FREE_WITH_TAG (PinDescriptors,PCX_MINWAVE_TAG);
        PinDescriptors = NULL;
    }

    if (pstNodeTrans)
    {
        DDK_FREE_WITH_TAG (pstNodeTrans,PCX_MINWAVE_TAG);
        pstNodeTrans = NULL;
    }

    if (pstPinTrans)
    {
        DDK_FREE_WITH_TAG (pstPinTrans,PCX_MINWAVE_TAG);
        pstPinTrans = NULL;
    }
}


/*****************************************************************************
 * CMiniportWaveICH::MyVery1stInit
 *****************************************************************************
 * Initializes the miniport indexes
 */
STDMETHODIMP_(void) CMiniportWaveICH::MyVery1stInit
(
    IN      USHORT      PmDeviceIndex,
    IN      USHORT      PmBoardIndex,
    IN      USHORT      PmChannelFirst,
    IN      USHORT      PmChannelCount,
    IN      LPGUID      PmDeviceNameGuid,
    IN      BOOL        PmConnectInterrupt
)
{
    PAGED_CODE ();

    m_sDeviceNumber     = PmDeviceIndex;
    m_sBoardIndex       = PmBoardIndex;
    m_wChannelNumber    = PmChannelFirst ;
    m_wChannelCount     = PmChannelCount ;
    m_guidDevName       = *PmDeviceNameGuid;
    m_bConnectInterrupt = PmConnectInterrupt;
}

/*****************************************************************************
 * CMiniportWaveICH::Init
 *****************************************************************************
 * Initializes the miniport.
 * Initializes variables and modifies the wave topology if needed.
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICH::Init
(
    IN  PUNKNOWN       UnknownAdapter,
    IN  PRESOURCELIST  ResourceList,
    IN  PPORTWAVEPCI   Port_,
    OUT PSERVICEGROUP *ServiceGroup_
)
{
    PAGED_CODE ();

    ASSERT (UnknownAdapter);
    ASSERT (ResourceList);
    ASSERT (Port_);

    WORD LcBoardFamily = 0 ;

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::Init]"));

    // Initialize any fields related to memory allocation
    // and that will require clean up on our detruction
    //
    m_DmaChannel = NULL;
    m_InterruptSync = NULL;
    m_AdapterCommon = NULL;
    m_Port = NULL;
    m_pTopology = NULL;

    //
    // AddRef() is required because we are keeping this pointer.
    //
    m_Port = Port_;
    m_Port->AddRef ();

    //
    // No miniport service group
    //
    *ServiceGroup_ = NULL;

    //
    // Set initial device power state
    //
    m_PowerState = PowerDeviceD0;

    NTSTATUS ntStatus = UnknownAdapter->
        QueryInterface (IID_IAdapterCommon, (PVOID *)&m_AdapterCommon);


    ULONG l_ulMaxWaveInOutNumber =  m_AdapterCommon->GetMaxWaveInOutNumber();
    ULONG l_ulWaveInNumber =        m_AdapterCommon->GetWaveInNumber();
    ULONG l_ulWaveOutNumber =       m_AdapterCommon->GetWaveOutNumber();

    if (NT_SUCCESS (ntStatus))
    {
        ntStatus = BuildDataRangeInformation ();
    }

    m_PinDataRangesIndex = m_wChannelCount - 1;
    ASSERT(m_PinDataRangesIndex < DR_IDX_INVALID);

	// Get protocol From AdapterCommon :
	m_pDsp = (CProtocol*)m_AdapterCommon->GetProtocolPtrAsVoid();

	DOUT (DBG_PROT_INIT, ("CMiniportWaveICH::Init got protocol from AdapterCommon : %lx\n", m_pDsp));

     // build the topology (means register pins, nodes, connections).
    if (NT_SUCCESS (ntStatus))
    {
        ntStatus = BuildTopology (m_sDeviceNumber,
                                  l_ulWaveInNumber,
                                  l_ulWaveOutNumber,
                                  l_ulMaxWaveInOutNumber);
    }

    //
    // Process the resources.
    //
    if (NT_SUCCESS (ntStatus))
    {
        ntStatus = ProcessResources (ResourceList);
    }

    //
    // Create a service group (a DPC abstraction/helper) to help with
    // interrupts.
    //
    if (NT_SUCCESS (ntStatus))
    {
        ntStatus = PcNewServiceGroup (&m_pServiceGroupWave, NULL);
    }
    if (NT_SUCCESS (ntStatus))
    {
        *ServiceGroup_ = m_pServiceGroupWave;
        m_pServiceGroupWave->AddRef ();
    }

    //
    // If we fail we get destroyed anyway (that's where we clean up).
    //
    return ntStatus;
}


NTSTATUS CMiniportWaveICH::BuildDataRangeInformation (void)
{
    PAGED_CODE ();

    NTSTATUS    ntStatus;
    int nChannels;
    int nLoop;

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::BuildDataRangeInformation]"));

    if( PinDataRangesInit )
    {
        return STATUS_SUCCESS;
    }

	CPIOCommands *pio = (CPIOCommands*)m_AdapterCommon->GetIOCommandsPtrAsVoid();
	ULONG es_max_rate;
	ULONG es_min_rate;
	switch( pio->PIOGetFrequencyRatio() )
	{
	case FREQ_RATIO_QUAD_MODE :
		es_max_rate = 4 * ES_MAX_RATE;	// 192000
		es_min_rate = 4 * ES_MIN_RATE;	// 176400
		break;
	case FREQ_RATIO_DUAL_MODE :
		es_max_rate = 2 * ES_MAX_RATE;	// 96000
		es_min_rate = 2 * ES_MIN_RATE;	// 88200
		break;
	case FREQ_RATIO_SINGLE_MODE :
	default:
		es_max_rate = ES_MAX_RATE;		// 48000
		es_min_rate = ES_MIN_RATE;		// 44100
		break;
	}

    for (nLoop = 0; nLoop < DR_IDX_INVALID; nLoop++)
    {
        nChannels = nLoop + 1;

        // Add it to the PinDataRange
        PinDataRangesStreamStatic[nLoop].DataRange.FormatSize = sizeof(KSDATARANGE_AUDIO);
        PinDataRangesStreamStatic[nLoop].DataRange.Flags      = 0;
        PinDataRangesStreamStatic[nLoop].DataRange.SampleSize = 0; // was nChannels * 2;  
        PinDataRangesStreamStatic[nLoop].DataRange.Reserved   = 0;
        PinDataRangesStreamStatic[nLoop].DataRange.MajorFormat = KSDATAFORMAT_TYPE_AUDIO;
        PinDataRangesStreamStatic[nLoop].DataRange.SubFormat   = KSDATAFORMAT_SUBTYPE_PCM;
        PinDataRangesStreamStatic[nLoop].DataRange.Specifier   = KSDATAFORMAT_SPECIFIER_WAVEFORMATEX;
        PinDataRangesStreamStatic[nLoop].MaximumChannels = nChannels;
        PinDataRangesStreamStatic[nLoop].MinimumBitsPerSample = ES_MIN_BITWIDTH;    // 16
        PinDataRangesStreamStatic[nLoop].MaximumBitsPerSample = ES_MAX_BITWIDTH;    // 24
        PinDataRangesStreamStatic[nLoop].MinimumSampleFrequency = es_min_rate;      // Minimum rate.
        PinDataRangesStreamStatic[nLoop].MaximumSampleFrequency = es_max_rate;      // Maximum rate.

        // Add it to the PinDataRangePointer
        PinDataRangePointersStreamStatic[nLoop] = (PKSDATARANGE)&PinDataRangesStreamStatic[nLoop];
    }
    PinDataRangesInit = TRUE;;

    return STATUS_SUCCESS;
}


//=============================================================================
// MBR (2005/08/26) add ComponentId handler from MSVAD driver
//=============================================================================


NTSTATUS
CMiniportWaveICH::PropertyHandlerComponentId
(
    IN PPCPROPERTY_REQUEST      PropertyRequest
)
{
    PAGED_CODE();

    DOUT (DBG_PRINT, ("[PropertyHandlerComponentId] (node = %d)\n", PropertyRequest->Node));

    NTSTATUS ntStatus = STATUS_INVALID_DEVICE_REQUEST;

    if (PropertyRequest->Verb & KSPROPERTY_TYPE_BASICSUPPORT)
    {
        if (PropertyRequest->ValueSize >= (sizeof(KSPROPERTY_DESCRIPTION)) )
        {
            // if return buffer can hold a KSPROPERTY_DESCRIPTION, return it
            //
            PKSPROPERTY_DESCRIPTION PropDesc = PKSPROPERTY_DESCRIPTION(PropertyRequest->Value);

            PropDesc->AccessFlags       = KSPROPERTY_TYPE_BASICSUPPORT | KSPROPERTY_TYPE_GET;
            PropDesc->DescriptionSize   = sizeof(KSPROPERTY_DESCRIPTION);
            PropDesc->PropTypeSet.Set   = KSPROPTYPESETID_General;
            PropDesc->PropTypeSet.Id    = VT_I4;
            PropDesc->PropTypeSet.Flags = 0;
            PropDesc->MembersListCount  = 0;
            PropDesc->Reserved          = 0;

            PropertyRequest->ValueSize = sizeof(KSPROPERTY_DESCRIPTION);
            ntStatus = STATUS_SUCCESS;
        }
        else if (PropertyRequest->ValueSize >= sizeof(ULONG))
        {
            // if return buffer can hold a ULONG, return the access flags
            //
            *(PULONG(PropertyRequest->Value)) = KSPROPERTY_TYPE_BASICSUPPORT | KSPROPERTY_TYPE_GET;

            PropertyRequest->ValueSize = sizeof(ULONG);
            ntStatus = STATUS_SUCCESS;                    
        }
        else if (0 == PropertyRequest->ValueSize)
        {
            // Send the caller required value size.
            PropertyRequest->ValueSize = sizeof(ULONG);
            ntStatus = STATUS_BUFFER_OVERFLOW;
        }
        else
        {
            PropertyRequest->ValueSize = 0;
            ntStatus = STATUS_BUFFER_TOO_SMALL;
        }
    }
    else
    {
        // If the caller is asking for ValueSize.
        //
        if (0 == PropertyRequest->ValueSize) 
        {
            PropertyRequest->ValueSize = sizeof(KSCOMPONENTID);
            ntStatus = STATUS_BUFFER_OVERFLOW;
        }
        else if(PropertyRequest->ValueSize < sizeof(KSCOMPONENTID))
        {
            PropertyRequest->ValueSize = 0;
            ntStatus = STATUS_BUFFER_TOO_SMALL;
        }
        else if (PropertyRequest->Verb & KSPROPERTY_TYPE_GET)
        {
            PKSCOMPONENTID pComponentId = (PKSCOMPONENTID)PropertyRequest->Value;

            //INIT_MMREG_MID(&pComponentId->Manufacturer, MM_DIGIGRAM);   // in mmreg.h
            INIT_MMREG_MID(&pComponentId->Manufacturer, MM_MICROSOFT);  // Defect 4296 : utiliser Microsoft ID a cause du WaveHR
            INIT_MMREG_PID(&pComponentId->Product, MM_WAVE_MAPPER); // use MM_PID_UNMAPPED ??
            pComponentId->Name      = this->m_guidDevName;//GUID_NULL; // on XP you can replace here the devices friendly name defined in the *.inf file !!!
            pComponentId->Component = GUID_NULL; // Not used for extended caps.
            pComponentId->Version   = PCX_VERSION_NUMBER;
            pComponentId->Revision  = PCX_VERSION_RELEASE;

            PropertyRequest->ValueSize = sizeof(KSCOMPONENTID);
            ntStatus = STATUS_SUCCESS;
        }
        else
        {
            DOUT (DBG_ERROR, ("[PropertyHandlerComponentId - Invalid parameter]\n"));
            ntStatus = STATUS_INVALID_PARAMETER;
        }
    }

    return ntStatus;
} // PropertyHandlerComponentId


#define CB_EXTENSIBLE (sizeof(WAVEFORMATEXTENSIBLE) - sizeof(WAVEFORMATEX))

//=============================================================================
NTSTATUS
CMiniportWaveICH::PropertyHandlerProposedFormat
(
    IN PPCPROPERTY_REQUEST      PropertyRequest
)
{
    PAGED_CODE();

    DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] board %d device %d PinDataRange %d (this=%p)\n", m_sBoardIndex, m_sDeviceNumber, m_PinDataRangesIndex, this));

    NTSTATUS ntStatus = STATUS_INVALID_DEVICE_REQUEST;

    if (PropertyRequest->Verb & KSPROPERTY_TYPE_BASICSUPPORT)
    {
        if (PropertyRequest->ValueSize >= (sizeof(KSPROPERTY_DESCRIPTION)) )
        {
            // if return buffer can hold a KSPROPERTY_DESCRIPTION, return it
            //
            PKSPROPERTY_DESCRIPTION PropDesc = PKSPROPERTY_DESCRIPTION(PropertyRequest->Value);

            PropDesc->AccessFlags       = KSPROPERTY_TYPE_BASICSUPPORT | KSPROPERTY_TYPE_GET;
            PropDesc->DescriptionSize   = sizeof(KSPROPERTY_DESCRIPTION);
            PropDesc->PropTypeSet.Set   = GUID_NULL; //KSPROPTYPESETID_General;
            PropDesc->PropTypeSet.Id    = 0; //VT_I4;
            PropDesc->PropTypeSet.Flags = 0;
            PropDesc->MembersListCount  = 0;
            PropDesc->Reserved          = 0;

            PropertyRequest->ValueSize = sizeof(KSPROPERTY_DESCRIPTION);
            ntStatus = STATUS_SUCCESS;
        }
        else if (PropertyRequest->ValueSize >= sizeof(ULONG))
        {
            // if return buffer can hold a ULONG, return the access flags
            //
            *(PULONG(PropertyRequest->Value)) = KSPROPERTY_TYPE_BASICSUPPORT | KSPROPERTY_TYPE_GET;

            PropertyRequest->ValueSize = sizeof(ULONG);
            ntStatus = STATUS_SUCCESS;                    
        }
        else if (0 == PropertyRequest->ValueSize)
        {
            // Send the caller required value size.
            PropertyRequest->ValueSize = sizeof(ULONG);
            ntStatus = STATUS_BUFFER_OVERFLOW;
        }
        else
        {
            PropertyRequest->ValueSize = 0;
            ntStatus = STATUS_BUFFER_TOO_SMALL;
        }
    }
    else
    {
        ULONG cbMinSize = sizeof(KSDATAFORMAT_WAVEFORMATEX);

        if (PropertyRequest->ValueSize == 0)
        {
            PropertyRequest->ValueSize = cbMinSize;
            ntStatus = STATUS_BUFFER_OVERFLOW;
			DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] STATUS_BUFFER_OVERFLOW\n"));
        }
        else if (PropertyRequest->ValueSize < cbMinSize)
        {
            ntStatus = STATUS_BUFFER_TOO_SMALL;
			DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] STATUS_BUFFER_TOO_SMALL\n"));
        }
        else
        {
            if (PropertyRequest->Verb & KSPROPERTY_TYPE_SET)
            {
                KSDATAFORMAT_WAVEFORMATEX* pKsFormat = (KSDATAFORMAT_WAVEFORMATEX*)PropertyRequest->Value;

                ntStatus = STATUS_NO_MATCH;

                if ((pKsFormat->DataFormat.MajorFormat == KSDATAFORMAT_TYPE_AUDIO) &&
                    (pKsFormat->DataFormat.SubFormat == KSDATAFORMAT_SUBTYPE_PCM) &&
                    (pKsFormat->DataFormat.Specifier == KSDATAFORMAT_SPECIFIER_WAVEFORMATEX))
                {
                    WAVEFORMATEX* pWfx = (WAVEFORMATEX*)&pKsFormat->WaveFormatEx;
					ULONG	Lc_my_max_sample_freq = PinDataRangesStreamStatic[m_PinDataRangesIndex].MaximumSampleFrequency;
					ULONG	Lc_my_min_sample_freq = PinDataRangesStreamStatic[m_PinDataRangesIndex].MinimumSampleFrequency;
					ULONG	Lc_my_min_bit_width   = PinDataRangesStreamStatic[m_PinDataRangesIndex].MinimumBitsPerSample;
					ULONG	Lc_my_max_bit_width   = PinDataRangesStreamStatic[m_PinDataRangesIndex].MaximumBitsPerSample;

					if( (pWfx->wBitsPerSample & 0x07)					||
						(pWfx->wBitsPerSample > Lc_my_max_bit_width)	||
						(pWfx->wBitsPerSample < Lc_my_min_bit_width)	)
					{
						DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] index : %d WRONG BPS : %d (%d...%d)\n", m_PinDataRangesIndex, pWfx->wBitsPerSample, Lc_my_min_bit_width, Lc_my_max_bit_width));
						return STATUS_NO_MATCH;
					}
					if( (pWfx->nSamplesPerSec < Lc_my_min_sample_freq) ||
						(pWfx->nSamplesPerSec > Lc_my_max_sample_freq) )
					{
						DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] WRONG FREQ : %d\n", pWfx->nSamplesPerSec));
						return STATUS_NO_MATCH;
					}

                    // make sure the WAVEFORMATEX part of the format makes sense
                    if ((pWfx->nBlockAlign == ((pWfx->nChannels * pWfx->wBitsPerSample) / 8)) &&
                        (pWfx->nAvgBytesPerSec == (pWfx->nSamplesPerSec * pWfx->nBlockAlign)))
                    {
                        if ((pWfx->wFormatTag == WAVE_FORMAT_PCM) && (pWfx->cbSize == 0))
                        {
                            if( pWfx->nChannels <= m_wChannelCount )
                            {
                                ntStatus = STATUS_SUCCESS;
								DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] freq=%d bit=%d chan=%d\n", pWfx->nSamplesPerSec, pWfx->wBitsPerSample, pWfx->nChannels));
                            }
							else
								DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] ERROR Channels=%d\n", pWfx->nChannels));
                        }
                        else
                        if( (pWfx->wFormatTag == WAVE_FORMAT_EXTENSIBLE) && (pWfx->cbSize == CB_EXTENSIBLE) )
                        {
                            WAVEFORMATEXTENSIBLE* pWfxT = (WAVEFORMATEXTENSIBLE*)pWfx;

                            if( ( pWfx->nChannels <= m_wChannelCount ) &&
							   (((pWfx->nChannels != 0) && (pWfxT->dwChannelMask == 0 )) ||
                                ((pWfx->nChannels == 1) && (pWfxT->dwChannelMask == SPEAKER_FRONT_CENTER)) ||
                                ((pWfx->nChannels == 2) && (pWfxT->dwChannelMask == KSAUDIO_SPEAKER_STEREO)) ||
                                ((pWfx->nChannels == 4) && (pWfxT->dwChannelMask == KSAUDIO_SPEAKER_QUAD)) ||
                                ((pWfx->nChannels == 6) && (pWfxT->dwChannelMask == KSAUDIO_SPEAKER_5POINT1)) ||
                                ((pWfx->nChannels == 8) && (pWfxT->dwChannelMask == KSAUDIO_SPEAKER_7POINT1_SURROUND))))
                            {
                                ntStatus = STATUS_SUCCESS;
								DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] freq=%d bit=%d chan=%d mask=%x\n", pWfx->nSamplesPerSec, pWfx->wBitsPerSample, pWfx->nChannels, pWfxT->dwChannelMask));
                            }
							else
								DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] ERROR Channels=%d mask=%x\n", pWfx->nChannels, pWfxT->dwChannelMask));
                        }
						else
								DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] ERROR FormatTag\n"));
                    }
					else
								DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] ERROR WAVEFORMATEX\n"));
                }
				else
								DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] ERROR DATAFORMAT\n"));
            }
            else
            {
                ntStatus = STATUS_INVALID_PARAMETER;
				DOUT (DBG_PRINT, ("[PropertyHandlerProposedFormat] STATUS_INVALID_PARAMETER\n"));
            }
        }
    }

    return ntStatus;
} // PropertyHandlerProposedFormat


//=============================================================================
static NTSTATUS
PropertyHandler_WaveFilter
( 
    IN PPCPROPERTY_REQUEST      PropertyRequest 
)
{
    PAGED_CODE();

    NTSTATUS ntStatus = STATUS_INVALID_DEVICE_REQUEST;
    if(PropertyRequest && PropertyRequest->MajorTarget)
    {
        IMiniportWaveICH* pIMiniportWaveICH;

        ntStatus = PropertyRequest->MajorTarget->QueryInterface(IID_IMiniportWaveICH, (PVOID *)&pIMiniportWaveICH);

        if(NT_SUCCESS (ntStatus) && pIMiniportWaveICH)
        {
            switch (PropertyRequest->PropertyItem->Id)
            {
                case KSPROPERTY_GENERAL_COMPONENTID:
                    ntStatus = pIMiniportWaveICH->PropertyHandlerComponentId( PropertyRequest );
                    break;
        
                case KSPROPERTY_PIN_PROPOSEDATAFORMAT:
                    ntStatus = pIMiniportWaveICH->PropertyHandlerProposedFormat( PropertyRequest );
                    break;

                default:
                    DOUT (DBG_ERROR, ("[PropertyHandler_WaveFilter: Invalid Device Request]\n"));
            }

            pIMiniportWaveICH->Release();
        }
    }

    return ntStatus;
} // PropertyHandler_WaveFilter

//=============================================================================
static
PCPROPERTY_ITEM PropertiesWaveFilter[] =
{
  {
    &KSPROPSETID_General,
    KSPROPERTY_GENERAL_COMPONENTID,
    KSPROPERTY_TYPE_GET | KSPROPERTY_TYPE_BASICSUPPORT,
    PropertyHandler_WaveFilter
  },
  {
    &KSPROPSETID_Pin,
    KSPROPERTY_PIN_PROPOSEDATAFORMAT,
    KSPROPERTY_TYPE_SET | KSPROPERTY_TYPE_BASICSUPPORT,
    PropertyHandler_WaveFilter
  }
};

DEFINE_PCAUTOMATION_TABLE_PROP (AutomationWaveFilter, PropertiesWaveFilter);
//=============================================================================



/*****************************************************************************
 * PropertiesDAC
 *****************************************************************************
 * Properties for the DAC node.
 */
static PCPROPERTY_ITEM PropertiesDAC[] =
{
    { 
        &KSPROPSETID_Audio,
        KSPROPERTY_AUDIO_CHANNEL_CONFIG,
        KSPROPERTY_TYPE_GET|KSPROPERTY_TYPE_GET,
        CMiniportWaveICH::PropertyChannelConfig
    }
};

/*****************************************************************************
 * AutomationVolume
 *****************************************************************************
 * Automation table for volume controls.
 */
DEFINE_PCAUTOMATION_TABLE_PROP (AutomationDAC, PropertiesDAC);



/*****************************************************************************
 * CMiniportWaveICH::BuildTopology
 *****************************************************************************
 * Builds the topology descriptors based on hardware configuration info
 * obtained from the adapter.
 */
NTSTATUS CMiniportWaveICH::BuildTopology
(
    IN      ULONG   PmDeviceIndex,
    IN      ULONG   PmWaveInNumber,
    IN      ULONG   PmWaveOutNumber,
    IN      ULONG   PmMaxWaveInOutNumber
)
{
    PAGED_CODE ();

    NTSTATUS ntStatus = STATUS_SUCCESS;

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::BuildTopology]"));

    // allocate our filter descriptor
    FilterDescriptor = (PPCFILTER_DESCRIPTOR) DDK_ALLOC_WITH_TAG (PagedPool,
                        sizeof(PCFILTER_DESCRIPTOR), PCX_MINWAVE_TAG);
    if (FilterDescriptor)
    {
        // clear out the filter descriptor
        RtlZeroMemory (FilterDescriptor, sizeof(PCFILTER_DESCRIPTOR));

        // build the pin list
        ntStatus = BuildPinDescriptors (PmDeviceIndex,
                                        PmWaveInNumber,
                                        PmWaveOutNumber,
                                        PmMaxWaveInOutNumber);

        if (NT_SUCCESS (ntStatus))
        {
            // build the node list
            ntStatus = BuildNodeDescriptors (PmDeviceIndex,
                                             PmWaveInNumber,
                                             PmWaveOutNumber,
                                             PmMaxWaveInOutNumber);
            if (NT_SUCCESS (ntStatus))
            {
                // build the connection list
                ntStatus = BuildConnectionDescriptors (PmDeviceIndex,
                                                       PmWaveInNumber,
                                                       PmWaveOutNumber,
                                                       PmMaxWaveInOutNumber);
                if (NT_SUCCESS (ntStatus))
                {
                    FilterDescriptor->AutomationTable = &AutomationWaveFilter;
                }
            }
        }
    }
    else
    {
        ntStatus = STATUS_INSUFFICIENT_RESOURCES;
    }

    // that's whatever one of these build... functions returned.
    return ntStatus;
}


/*****************************************************************************
 * SetPinWaveOutSourceDescriptor
 *****************************************************************************
 *
 */
void SetPinWaveOutSourceDescriptor
(
    IN      PPCPIN_DESCRIPTOR   currentPin,
    IN      int                 RangesEntry
)
{
    // MaxGlobalInstanceCount
    currentPin->MaxGlobalInstanceCount = 1;
    // MaxFilterInstanceCount
    currentPin->MaxFilterInstanceCount = 1;
    // MinFilterInstanceCount
    currentPin->MinFilterInstanceCount = 0;
    // AutomationTable
    currentPin->AutomationTable = NULL;
    // KsPinDescriptor
    currentPin->KsPinDescriptor.InterfacesCount = 0;
    currentPin->KsPinDescriptor.Interfaces      = NULL;
    currentPin->KsPinDescriptor.MediumsCount    = 0;
    currentPin->KsPinDescriptor.Mediums         = NULL;
    currentPin->KsPinDescriptor.DataRangesCount = 1;	// only one
    currentPin->KsPinDescriptor.DataRanges      = &PinDataRangePointersStreamStatic[RangesEntry];
    currentPin->KsPinDescriptor.DataFlow        = KSPIN_DATAFLOW_IN;
    currentPin->KsPinDescriptor.Communication   = KSPIN_COMMUNICATION_SINK;
    currentPin->KsPinDescriptor.Category        = (GUID *) &KSCATEGORY_AUDIO;
    currentPin->KsPinDescriptor.Name            = NULL;
    currentPin->KsPinDescriptor.Reserved        = 0;
}

/*****************************************************************************
 * SetPinWaveOutDestDescriptor
 *****************************************************************************
 *
 */
void SetPinWaveOutDestDescriptor
(
    IN      PPCPIN_DESCRIPTOR   currentPin,
    IN      int                 RangesEntry
)
{
    // MaxGlobalInstanceCount
    currentPin->MaxGlobalInstanceCount = 0;
    // MaxFilterInstanceCount
    currentPin->MaxFilterInstanceCount = 0;
    // MinFilterInstanceCount
    currentPin->MinFilterInstanceCount = 0;
    // AutomationTable
    currentPin->AutomationTable = NULL;
    // KsPinDescriptor
    currentPin->KsPinDescriptor.InterfacesCount = 0;
    currentPin->KsPinDescriptor.Interfaces      = NULL;
    currentPin->KsPinDescriptor.MediumsCount    = 0;
    currentPin->KsPinDescriptor.Mediums         = NULL;
    currentPin->KsPinDescriptor.DataRangesCount = 1;	// only one
    currentPin->KsPinDescriptor.DataRanges      = &PinDataRangePointersStreamStatic[RangesEntry];
    currentPin->KsPinDescriptor.DataFlow        = KSPIN_DATAFLOW_OUT;
    currentPin->KsPinDescriptor.Communication   = KSPIN_COMMUNICATION_NONE;
    currentPin->KsPinDescriptor.Category        = (GUID *) &KSCATEGORY_AUDIO;
    currentPin->KsPinDescriptor.Name            = NULL;
    currentPin->KsPinDescriptor.Reserved        = 0;
}

/*****************************************************************************
 * SetPinWaveInSourceDescriptor
 *****************************************************************************
 *
 */
void SetPinWaveInSourceDescriptor
(
    IN      PPCPIN_DESCRIPTOR   currentPin,
    IN      int                 RangesEntry
)
{
    // MaxGlobalInstanceCount
    currentPin->MaxGlobalInstanceCount = 0;
    // MaxFilterInstanceCount
    currentPin->MaxFilterInstanceCount = 0;
    // MinFilterInstanceCount
    currentPin->MinFilterInstanceCount = 0;
    // AutomationTable
    currentPin->AutomationTable = NULL;
    // KsPinDescriptor
    currentPin->KsPinDescriptor.InterfacesCount = 0;
    currentPin->KsPinDescriptor.Interfaces      = NULL;
    currentPin->KsPinDescriptor.MediumsCount    = 0;
    currentPin->KsPinDescriptor.Mediums         = NULL;
    currentPin->KsPinDescriptor.DataRangesCount = 1;	// only one
    currentPin->KsPinDescriptor.DataRanges      = &PinDataRangePointersStreamStatic[RangesEntry];
    currentPin->KsPinDescriptor.DataFlow        = KSPIN_DATAFLOW_IN;
    currentPin->KsPinDescriptor.Communication   = KSPIN_COMMUNICATION_NONE;
    currentPin->KsPinDescriptor.Category        = (GUID *) &KSCATEGORY_AUDIO;
    currentPin->KsPinDescriptor.Name            = NULL;
    currentPin->KsPinDescriptor.Reserved        = 0;
}

/*****************************************************************************
 * SetPinWaveInDestDescriptor
 *****************************************************************************
 *
 */
void SetPinWaveInDestDescriptor
(
    IN      PPCPIN_DESCRIPTOR   currentPin,
    IN      int                 RangesEntry
)
{
    // MaxGlobalInstanceCount
    currentPin->MaxGlobalInstanceCount = 1;
    // MaxFilterInstanceCount
    currentPin->MaxFilterInstanceCount = 1;
    // MinFilterInstanceCount
    currentPin->MinFilterInstanceCount = 0;
    // AutomationTable
    currentPin->AutomationTable = NULL;
    // KsPinDescriptor
    currentPin->KsPinDescriptor.InterfacesCount = 0;
    currentPin->KsPinDescriptor.Interfaces      = NULL;
    currentPin->KsPinDescriptor.MediumsCount    = 0;
    currentPin->KsPinDescriptor.Mediums         = NULL;
    currentPin->KsPinDescriptor.DataRangesCount = 1;	// only one
    currentPin->KsPinDescriptor.DataRanges      = &PinDataRangePointersStreamStatic[RangesEntry];
    currentPin->KsPinDescriptor.DataFlow        = KSPIN_DATAFLOW_OUT;
    currentPin->KsPinDescriptor.Communication   = KSPIN_COMMUNICATION_SINK;
    currentPin->KsPinDescriptor.Category        = (GUID *) &PINNAME_CAPTURE;
    currentPin->KsPinDescriptor.Name            = &KSAUDFNAME_RECORDING_CONTROL;
    currentPin->KsPinDescriptor.Reserved        = 0;
}
/*****************************************************************************
 * CMiniportWaveICH::BuildPinDescriptors
 *****************************************************************************
 * Builds the topology pin descriptors.
 */
NTSTATUS CMiniportWaveICH::BuildPinDescriptors
(
    IN      ULONG   PmDeviceIndex,
    IN      ULONG   PmWaveInNumber,
    IN      ULONG   PmWaveOutNumber,
    IN      ULONG   PmMaxWaveInOutNumber
)
{
// Improvement would be to not use a Macro, use (inline) function instead.
//
#define INIT_PIN( pin, lineNumber, pinptr, category, name, index )  \
    pinptr->KsPinDescriptor.Category = (GUID*) category;            \
    pinptr->KsPinDescriptor.Name = (GUID*) name;                    \
    SetPinTranslation (index++, pin , lineNumber );                 \
    pinptr++

    PAGED_CODE ();

    ULONG               Index;
    PPCPIN_DESCRIPTOR   CurrentPin;

    int MemorySize;

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::BuildPinDescriptors]"));

    MemorySize = WAVE_PIN_TOP_ELEMENT * sizeof(tPinTransTable);
    pstPinTrans = (tPinTransTable*) (DDK_ALLOC_WITH_TAG(PagedPool,MemorySize,PCX_MINWAVE_TAG));

    //
    // initialize translation tables
    //
    if( !pstPinTrans)
        return STATUS_INSUFFICIENT_RESOURCES;

    ULONG i;
    for (i = 0; i < WAVE_PIN_TOP_ELEMENT ; i++)
    {

        pstPinTrans[i].PinDef = WAVE_PIN_INVALID;
        pstPinTrans[i].PinNr = -1;
    }

    // allocate our descriptor memory
    MemorySize = (WAVE_PIN_WAVEOUT_NUMBER) * sizeof(PCPIN_DESCRIPTOR);
    MemorySize += (WAVE_PIN_WAVEIN_NUMBER) * sizeof(PCPIN_DESCRIPTOR);

    PinDescriptors = PPCPIN_DESCRIPTOR (DDK_ALLOC_WITH_TAG (PagedPool, MemorySize, PCX_MINWAVE_TAG));
    if (!PinDescriptors)
        return STATUS_INSUFFICIENT_RESOURCES;

    //
    // set default pin descriptor parameters
    //
    RtlZeroMemory (PinDescriptors, MemorySize);

    //
    // modify the individual pin descriptors
    //
    CurrentPin  = PinDescriptors;
    Index       = 0;

	ASSERT( m_PinDataRangesIndex < DR_IDX_INVALID );

    // ************** ALL WAVE OUT **************************

    if ( PmDeviceIndex < PmWaveOutNumber )
    {
        ULONG lineOut =0;

        // add the WAVE_PIN_WAVEOUT pin descriptor (not optional)

        SetPinWaveOutSourceDescriptor(CurrentPin, m_PinDataRangesIndex);

        INIT_PIN (WAVE_PIN_WAVEOUT,
                  lineOut,
                  CurrentPin,
                  &KSCATEGORY_AUDIO,
                  NULL,
                  Index);

        // add the WAVE_PIN_WAVEOUT_BRIDGE pin descriptor (not optional)

        SetPinWaveOutDestDescriptor(CurrentPin, m_PinDataRangesIndex);

        INIT_PIN (WAVE_PIN_WAVEOUT_BRIDGE,
                  lineOut,
                  CurrentPin,
                  &KSCATEGORY_AUDIO,
                  NULL,
                  Index);
    }

    // ************** ALL WAVE IN **************************

    if ( PmDeviceIndex < PmWaveInNumber )
    {
        ULONG lineIn =0;

        // add the WAVE_PIN_WAVEIN_BRIDGE pin descriptor (not optional)

        SetPinWaveInSourceDescriptor(CurrentPin, m_PinDataRangesIndex);

        INIT_PIN (WAVE_PIN_WAVEIN_BRIDGE,
                  lineIn,
                  CurrentPin,
                  &KSCATEGORY_AUDIO,
                  NULL,
                  Index);


        // add the WAVE_PIN_WAVEIN pin descriptor (not optional)

        SetPinWaveInDestDescriptor(CurrentPin, m_PinDataRangesIndex);

        INIT_PIN (WAVE_PIN_WAVEIN,
                  lineIn,
                  CurrentPin,
                  &PINNAME_CAPTURE,
                  &KSAUDFNAME_RECORDING_CONTROL,
                  Index);
    }

    // add the pin descriptor informatin to the filter descriptor
    FilterDescriptor->PinCount = Index;
    FilterDescriptor->PinSize = sizeof (PCPIN_DESCRIPTOR);
    FilterDescriptor->Pins = PinDescriptors;

	(void)PmMaxWaveInOutNumber;

    return STATUS_SUCCESS;

#undef INIT_PIN
}

/*****************************************************************************
 * CMiniportWaveICH::BuildNodeDescriptors
 *****************************************************************************
 * Builds the topology node descriptors.
 */
NTSTATUS CMiniportWaveICH::BuildNodeDescriptors
(
    IN      ULONG   PmDeviceIndex,
    IN      ULONG   PmWaveInNumber,
    IN      ULONG   PmWaveOutNumber,
    IN      ULONG   PmMaxWaveInOutNumber
)
{
// Improvement would be to not use a Macro, use (inline) function instead.
#define INIT_NODE( node, lineNumber, nodeptr, type, name, automation, index )   \
    nodeptr->Type = (GUID*) type;                                   \
    nodeptr->Name = (GUID*) name;                                   \
    nodeptr->AutomationTable = automation;                          \
    SetNodeTranslation (index++, node, lineNumber);                             \
    nodeptr++

    PAGED_CODE ();

    NTSTATUS ntStatus = STATUS_SUCCESS;
    int MemorySize;

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::BuildNodeDescriptors]"));

    MemorySize = WAVE_NODE_TOP_ELEMENT * sizeof(tNodeTransTable);
    pstNodeTrans = (tNodeTransTable*) (DDK_ALLOC_WITH_TAG(PagedPool,MemorySize,PCX_MINWAVE_TAG));

    if( ! pstNodeTrans)
        return STATUS_INSUFFICIENT_RESOURCES;

    //
    // initialize translation tables
    //
    ULONG i;
    for (i = 0; i < WAVE_NODE_TOP_ELEMENT ; i++)
    {
        pstNodeTrans[i].NodeDef = WAVE_NODE_INVALID;
        pstNodeTrans[i].NodeNr = -1;
    }

    // allocate our descriptor memory
    MemorySize = (WAVE_NODE_WAVEOUT_NUMBER) * sizeof(PCNODE_DESCRIPTOR);
    MemorySize += (WAVE_NODE_WAVEIN_NUMBER) * sizeof(PCNODE_DESCRIPTOR);

    NodeDescriptors = PPCNODE_DESCRIPTOR (DDK_ALLOC_WITH_TAG (PagedPool,MemorySize,PCX_MINWAVE_TAG));
    if (NodeDescriptors)
    {
        PPCNODE_DESCRIPTOR  CurrentNode = NodeDescriptors;
        ULONG               Index = 0;

        //
        // set default node descriptor parameters
        //
        RtlZeroMemory (NodeDescriptors, MemorySize);


        // ************** ALL WAVE OUT **************************

        if ( PmDeviceIndex < PmWaveOutNumber )
        {
            ULONG lineOut =0;

            // add the WAVE_NODE_WAVEOUT_DAC node
            INIT_NODE (WAVE_NODE_WAVEOUT_DAC,
                       lineOut,
                       CurrentNode,
                       &KSNODETYPE_DAC,
                       NULL,
                       &AutomationDAC,	// support channel config
                       Index);
        }

        // ************** ALL WAVE IN **************************

        if ( PmDeviceIndex < PmWaveInNumber )
        {
            ULONG lineIn =0;

            // add the WAVE_NODE_WAVEIN_ADC node
            INIT_NODE (WAVE_NODE_WAVEIN_ADC,
                       lineIn,
                       CurrentNode,
                       &KSNODETYPE_ADC,
                       NULL,
                       NULL,
                       Index);
        }

        // add the nodes to the filter descriptor
        FilterDescriptor->NodeCount = Index;
        FilterDescriptor->NodeSize = sizeof(PCNODE_DESCRIPTOR);
        FilterDescriptor->Nodes = NodeDescriptors;
    }
    else
    {
        ntStatus = STATUS_INSUFFICIENT_RESOURCES;
    }

	(void)PmMaxWaveInOutNumber;

    return ntStatus;

#undef INIT_NODE
}

/*****************************************************************************
 * CMiniportWaveICH::BuildConnectionDescriptors
 *****************************************************************************
 *
 */
NTSTATUS CMiniportWaveICH::BuildConnectionDescriptors
(
    IN      ULONG   PmDeviceIndex,
    IN      ULONG   PmWaveInNumber,
    IN      ULONG   PmWaveOutNumber,
    IN      ULONG   PmMaxWaveInOutNumber
)
{
// Improvement would be to not use a Macro, use (inline) function instead.

// for filter pin to node connections
#define INIT_FN_CONN( cptr, fpin, tnode, tpin )         \
    cptr->FromNode = KSFILTER_NODE;                     \
    cptr->FromNodePin = TransPinDefToPinNr (fpin);      \
    cptr->ToNode = TransNodeDefToNodeNr (tnode);        \
    cptr->ToNodePin = tpin;                             \
    cptr++,ConnectionCount++

// for node to node connections
#define INIT_NN_CONN( cptr, fnode, fpin, tnode, tpin )  \
    cptr->FromNode = TransNodeDefToNodeNr (fnode);      \
    cptr->FromNodePin = fpin;                           \
    cptr->ToNode = TransNodeDefToNodeNr (tnode);        \
    cptr->ToNodePin = tpin;                             \
    cptr++,ConnectionCount++



// for node to filter pin connections
#define INIT_NF_CONN( cptr, fnode, fpin, tpin )         \
    cptr->FromNode = TransNodeDefToNodeNr (fnode);      \
    cptr->FromNodePin = fpin;                           \
    cptr->ToNode = KSFILTER_NODE;                       \
    cptr->ToNodePin = TransPinDefToPinNr (tpin);        \
    cptr++,ConnectionCount++

    PAGED_CODE ();

    NTSTATUS    ntStatus            = STATUS_SUCCESS;
    ULONG       ConnectionCount     = 0;

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::BuildConnectionDescriptors]"));

    // allocate our descriptor memory
    ConnectionDescriptors = PPCCONNECTION_DESCRIPTOR (DDK_ALLOC_WITH_TAG (PagedPool,
                            WAVE_MAX_CONNECTIONS * sizeof(PCCONNECTION_DESCRIPTOR),PCX_MINWAVE_TAG));
    if (ConnectionDescriptors)
    {
        PPCCONNECTION_DESCRIPTOR  CurrentConnection = ConnectionDescriptors;

        // ************** ALL WAVE OUT **************************

        if ( PmDeviceIndex < PmWaveOutNumber )
        {
            ULONG lineOut =0 ;

            // build the wave out (coming in) path

            ULONG l_ulPinOffset = lineOut * WAVE_PIN_TOP_ELEMENT;
            ULONG l_ulNodeOffset = lineOut * WAVE_NODE_TOP_ELEMENT;

            // WAVE_PIN_WAVEOUT -> WAVE_NODE_WAVEOUT_DAC
            INIT_FN_CONN (CurrentConnection,
                          WavePins(WAVE_PIN_WAVEOUT + l_ulPinOffset),
                          WaveNodes(WAVE_NODE_WAVEOUT_DAC + l_ulNodeOffset),
                          KSNODEPIN_STANDARD_IN);

            // WAVE_NODE_WAVEOUT_DAC -> WAVE_PIN_WAVEOUT_BRIDGE
            INIT_NF_CONN (CurrentConnection,
                          WaveNodes(WAVE_NODE_WAVEOUT_DAC + l_ulNodeOffset),
                          KSNODEPIN_STANDARD_OUT,
                          WavePins(WAVE_PIN_WAVEOUT_BRIDGE + l_ulPinOffset));
        }

        // ************** ALL WAVE IN **************************

        if ( PmDeviceIndex < PmWaveInNumber )
        {
            ULONG lineIn =0 ;

            ULONG l_ulPinOffset = lineIn * WAVE_PIN_TOP_ELEMENT;
            ULONG l_ulNodeOffset = lineIn * WAVE_NODE_TOP_ELEMENT;

            // WAVE_PIN_WAVEIN_BRIDGE -> WAVE_NODE_WAVEIN_ADC
            INIT_FN_CONN (CurrentConnection,
                          WavePins(WAVE_PIN_WAVEIN_BRIDGE + l_ulPinOffset),
                          WaveNodes(WAVE_NODE_WAVEIN_ADC + l_ulNodeOffset),
                          KSNODEPIN_STANDARD_IN);

            // WAVE_NODE_WAVEIN_ADC -> WAVE_PIN_WAVEIN
            INIT_NF_CONN (CurrentConnection,
                          WaveNodes(WAVE_NODE_WAVEIN_ADC + l_ulNodeOffset),
                          KSNODEPIN_STANDARD_OUT,
                          WavePins(WAVE_PIN_WAVEIN + l_ulPinOffset));
        }

        // add the connections to the filter descriptor
        FilterDescriptor->ConnectionCount = ConnectionCount;
        FilterDescriptor->Connections = ConnectionDescriptors;
    } else
    {
        ntStatus = STATUS_INSUFFICIENT_RESOURCES;
    }


	(void)PmMaxWaveInOutNumber;

    return ntStatus;

#undef INIT_NN_CONN
#undef INIT_FN_CONN
#undef INIT_NF_CONN
}

/*****************************************************************************
 * CMiniportWaveICH::ProcessResources
 *****************************************************************************
 * Processes the resource list, setting up helper objects accordingly.
 * Sets up the Interrupt + Service routine and DMA.
 */
NTSTATUS CMiniportWaveICH::ProcessResources
(
    IN  PRESOURCELIST ResourceList
)
{
    PAGED_CODE ();

    ASSERT (ResourceList);


    DOUT (DBG_PRINT, ("[CMiniportWaveICH::ProcessResources]"));


    ULONG countIRQ = ResourceList->NumberOfInterrupts ();
    if (countIRQ < 1)
    {
        DOUT (DBG_ERROR, ("Unknown configuration for wave miniport!"));
        return STATUS_DEVICE_CONFIGURATION_ERROR;
    }

    NTSTATUS ntStatus = STATUS_SUCCESS;

	if( m_bConnectInterrupt )
	{
        //
        // Hook up the interrupt.
        //
        ntStatus = PcNewInterruptSync(  &m_InterruptSync,           // Save object ptr
                                        NULL,                       // OuterUnknown(optional).
                                        ResourceList,               // He gets IRQ from ResourceList.
                                        0,                          // Resource Index
                                        InterruptSyncModeNormal     // Run all ISRs till return code is SUCCESS
                                     );
        if (!NT_SUCCESS (ntStatus) || !m_InterruptSync)
        {
            DOUT (DBG_ERROR, ("Failed to create an interrupt sync!"));
            return STATUS_INSUFFICIENT_RESOURCES;
        }

        //
        // Register our ISR.
        //
        ntStatus = m_InterruptSync->RegisterServiceRoutine( InterruptServiceRoutine,
                                                            (PVOID)this, FALSE  );
        if (!NT_SUCCESS (ntStatus))
        {
            DOUT (DBG_ERROR, ("Failed to register ISR!"));
            return ntStatus;
        }

        //
        // Connect the interrupt.
        //
        ntStatus = m_InterruptSync->Connect ();
        if (!NT_SUCCESS (ntStatus))
        {
            DOUT (DBG_ERROR, ("Failed to connect the ISR with InterruptSync!"));
            return ntStatus;
        }

	    CPIOCommands *pio = (CPIOCommands*)m_AdapterCommon->GetIOCommandsPtrAsVoid();

        pio->PIOEnableIrq();


        DOUT (DBG_PRINT, ("### CMiniportWaveICH  m_InterruptSync->Connect() OK"));
	}

    ASSERT(m_Port);

    //
    // Create the DMA Channel object.
    //
    ntStatus = m_Port->NewMasterDmaChannel (&m_DmaChannel,      // OutDmaChannel
                                            NULL,             // OuterUnknown (opt)
                                            NonPagedPool,     // Pool Type
                                            NULL,             // ResourceList (opt)
                                            TRUE,             // ScatterGather
                                            TRUE,             // Dma32BitAddresses
                                            FALSE,            // Dma64BitAddresses
                                            FALSE,            // IgnoreCount
                                            (DMA_WIDTH)(-1),  // Not used (was Width32Bits)
                                            (DMA_SPEED)(-1),  // Not used (was MaximumDmaSpeed)
                                            MAX_DSBUFF_SIZE,  // MaximumLength
                                            0);               // DmaPort
    if (!NT_SUCCESS (ntStatus))
    {
        DOUT (DBG_ERROR, ("Failed on NewMasterDmaChannel!"));
        return ntStatus;
    }

    //
    // On failure object is destroyed which cleans up.
    //
    return STATUS_SUCCESS;
}



/*****************************************************************************
 * CMiniportWaveICH::NewStream
 *****************************************************************************
 * Creates a new stream.
 * This function is called when a streaming pin is created.
 * It checks if the channel is already in use, tests the data format, creates
 * and initializes the stream object.
 */
STDMETHODIMP CMiniportWaveICH::NewStream
(
    OUT PMINIPORTWAVEPCISTREAM *Stream,
    IN  PUNKNOWN                OuterUnknown,
    IN  POOL_TYPE               PoolType,
    IN  PPORTWAVEPCISTREAM      PortStream,
    IN  ULONG                   Pin_,
    IN  BOOLEAN                 Capture,
    IN  PKSDATAFORMAT           DataFormat,
    OUT PDMACHANNEL            *DmaChannel_,
    OUT PSERVICEGROUP          *ServiceGroup
)
{
    PAGED_CODE ();

    ASSERT (Stream);
    ASSERT (PortStream);
    ASSERT (DataFormat);
    ASSERT (DmaChannel_);
    ASSERT (ServiceGroup);

    CMiniportWaveICHStream *pWaveICHStream = 0;
    NTSTATUS ntStatus = STATUS_SUCCESS;
    ULONG Entry = 0;

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::NewStream Capture(%d) Device(%d), Board(%d)]", Capture, m_sDeviceNumber, m_sBoardIndex));

    //
    // Validate the channel (pin id).
    //
    if (
           (Pin_ != WAVE_PIN_WAVEOUT)
        && (Pin_ != WAVE_PIN_WAVEIN)
        )
    {
        DOUT (DBG_ERROR, ("NewStream was passed an invalid channel!"));
        return STATUS_INVALID_PARAMETER;
    }

    //
    // Check if the pin is already in use
    //
    if (Streams[Pin_])
    {
        DOUT (DBG_ERROR, ("Pin is already in use!"));
        return STATUS_UNSUCCESSFUL;
    }

    //
    // Check parameters.
    //
    ntStatus = TestDataFormat(DataFormat,(WavePins) Pin_);

    if (!NT_SUCCESS (ntStatus))
    {
        DOUT (DBG_VSR, ("MinWaveICH::NewStream ==> TestDataFormat failed\n"));
        return ntStatus;
    }

	// update protocol pointer, in case the object changed because of a powerchange
	//
	m_pDsp = (CProtocol*)m_AdapterCommon->GetProtocolPtrAsVoid();

	if (m_pDsp->IDiag_IsDspRunning() == FALSE ) 
		// mise a jour potentielle de la carte
		//
		return STATUS_UNSUCCESSFUL;
    
	//
    // Create a new stream.
    //
    ntStatus = CreateMiniportWaveICHStream(&pWaveICHStream,OuterUnknown,PoolType);

    //
    // Return in case of an error.
    //
    if (!NT_SUCCESS (ntStatus))
    {
        DOUT (DBG_ERROR, ("MinWaveICH::NewStream ==> CreateMiniportWaveICHStream failed\n"));
        return ntStatus;
    }

	// Pass protocol pointer along to stream
	//
	pWaveICHStream->m_pDsp = m_pDsp;

    // Initialize the stream.
    //
    ntStatus = pWaveICHStream->Init (this,
                                     PortStream,
                                     Pin_,
                                     Capture,
                                     DataFormat,
                                     Entry,
                                     m_wChannelNumber,
                                     ServiceGroup);

    if (!NT_SUCCESS (ntStatus)) 
		goto _failed_exit ; 

    //
    // Save the pointers.
    //
    *Stream = (PMINIPORTWAVEPCISTREAM)pWaveICHStream;

    // Needed though useless for us ! Sigh...
    //
    *DmaChannel_ = m_DmaChannel;
    //m_DmaChannel->SetBufferSize(MAX_DSBUFF_SIZE);

    return STATUS_SUCCESS;

_failed_exit:

    //
    // Release the stream and clean up.
    //
    DOUT (DBG_ERROR, ("MinWaveICH::NewStream ==> WaveICHStream->Init failed RELEASE !\n"));
    pWaveICHStream->Release ();
    *Stream = NULL;

    return ntStatus;

}


/*****************************************************************************
 * CMiniportWaveICH::GetDescription
 *****************************************************************************
 * Gets the topology.
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICH::GetDescription
(
    OUT PPCFILTER_DESCRIPTOR *OutFilterDescriptor
)
{
    PAGED_CODE ();

    ASSERT (OutFilterDescriptor);

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::GetDescription]"));

#if (DBG)
    // Dump it here. The system requests the topology only once.
    DumpTopology ();
#endif

    *OutFilterDescriptor = FilterDescriptor;


    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICH::DataRangeIntersection
 *****************************************************************************
 * Tests a data range intersection.
 * Cause the AC97 controller does not support mono render or capture, we have
 * to check the max. channel field (unfortunately, there is no MinimumChannel
 * and MaximumChannel field, just a MaximumChannel field).
 * If the MaximumChannel is 2, then we can pass this to the default handler of
 * portcls which always chooses the most (SampleFrequency, Channel, Bits etc.)
 *
 * This DataRangeIntersection function is strictly only for the exposed formats
 * in this sample driver. If you intend to add other formats like AC3 then
 * you have to be make sure that you check the GUIDs and the data range, since
 * portcls only checks the data range for waveformatex.
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICH::DataRangeIntersection
(
    IN      ULONG           PinId,
    IN      PKSDATARANGE    ClientsDataRange,
    IN      PKSDATARANGE    MyDataRange,
    IN      ULONG           OutputBufferLength,
    OUT     PVOID           ResultantFormat,
    OUT     PULONG          ResultantFormatLength
)
{
    PAGED_CODE ();

	WORD	Lc_client_channel_number = 0; 
	DWORD	Lc_client_max_sample_freq =0;
	DWORD	Lc_client_min_sample_freq =0;
	WORD	Lc_client_max_bits_per_spl=0;

	WORD	Lc_my_channel_number = 0; 
	DWORD	Lc_my_max_sample_freq =0;
	DWORD	Lc_my_min_sample_freq =0;
	WORD	Lc_my_min_bit_width   =0;
	WORD	Lc_my_max_bit_width   =0;


	(void)PinId;

	*ResultantFormatLength = 0;
	PWAVEFORMATPCMEX WaveFormat = NULL;

    //
    // This function gets only called if the GUIDS in the KSDATARANGE_AUDIO
    // structure that we attached to the pin are equal with the requested
    // format (see "BuildDataRangeInformation).
    // Additionally, for waveformatex portcls checks that the requested sample
    // frequency range fits into our exposed sample frequency range. Since we
    // only have discrete sample frequencies in the pin's data range, we don't
    // have to check that either.
    // There is one exeption to this rule: portcls clones all WAVEFORMATEX
    // data ranges to DSOUND dataranges, so we might get a data range
    // intersection that has a DSOUND specifier. We don't support that
    // since this is only used for HW acceleration
    //
    if (IsEqualGUIDAligned (ClientsDataRange->Specifier, KSDATAFORMAT_SPECIFIER_DSOUND))
    {
        DOUT (DBG_PRINT, ("[DataRangeIntersection] We don't support DSOUND specifier"));
        return STATUS_NOT_SUPPORTED;
    }
    
    // Start with checking the size of the output buffer.
    // (Query)
    if (!OutputBufferLength) 
    {
        // (Query)
		*ResultantFormatLength = sizeof(KSDATAFORMAT) + sizeof(WAVEFORMATPCMEX);
        return STATUS_BUFFER_OVERFLOW;
    } 
    
    if (OutputBufferLength < (sizeof(KSDATAFORMAT) + sizeof(WAVEFORMATPCMEX))) 
    {
        DOUT (DBG_WARNING, ("[DataRangeIntersection] Buffer too small"));
        return STATUS_BUFFER_TOO_SMALL;
    }

	Lc_client_channel_number	=	(WORD)((PKSDATARANGE_AUDIO)ClientsDataRange)->MaximumChannels;
	Lc_client_max_sample_freq	=	((PKSDATARANGE_AUDIO)ClientsDataRange)->MaximumSampleFrequency;
	Lc_client_min_sample_freq	=	((PKSDATARANGE_AUDIO)ClientsDataRange)->MinimumSampleFrequency;
	Lc_client_max_bits_per_spl	=	(WORD)((PKSDATARANGE_AUDIO)ClientsDataRange)->MaximumBitsPerSample;

	Lc_my_channel_number	=	(WORD)((PKSDATARANGE_AUDIO)MyDataRange)->MaximumChannels;
	Lc_my_max_sample_freq	=	((PKSDATARANGE_AUDIO)MyDataRange)->MaximumSampleFrequency;
	Lc_my_min_sample_freq	=	((PKSDATARANGE_AUDIO)MyDataRange)->MinimumSampleFrequency;
	Lc_my_max_bit_width	    =	(WORD)((PKSDATARANGE_AUDIO)MyDataRange)->MaximumBitsPerSample;
	Lc_my_min_bit_width	    =	(WORD)((PKSDATARANGE_AUDIO)MyDataRange)->MinimumBitsPerSample;

    DOUT (DBG_PRINT, ("[DataRangeIntersection] Pin(%d) Device(%d), Board(%d)\n", PinId, m_sDeviceNumber, m_sBoardIndex));

    DOUT (DBG_PRINT, ("[DataRangeIntersection] CLIENT: Maxfreq %d, Minfreq %d - MaxBps: %d, MaxChan: %d\n",\
		Lc_client_max_sample_freq,\
		Lc_client_min_sample_freq,\
		Lc_client_max_bits_per_spl, \
		Lc_client_channel_number));

    DOUT (DBG_PRINT, ("[DataRangeIntersection]  THIS : Maxfreq %d, Minfreq %d - MaxBps: %d, MaxChan: %d\n",\
		Lc_my_max_sample_freq,\
		Lc_my_min_sample_freq,\
		Lc_my_max_bit_width, \
		Lc_my_channel_number));

	// Fill in the structure the datarange structure.
	// KSDATARANGE and KSDATAFORMAT are the same.
	//
	*(PKSDATAFORMAT)ResultantFormat = *MyDataRange;

	//
	// Modify the size of the data format structure to fit the WAVEFORMATPCMEX
	// structure.
	//
	((PKSDATAFORMAT)ResultantFormat)->FormatSize =	sizeof(KSDATAFORMAT) + sizeof(WAVEFORMATPCMEX);
	
	//
	// Append the WAVEFORMATPCMEX structur.
	//
	WaveFormat = (PWAVEFORMATPCMEX)((PKSDATAFORMAT)ResultantFormat + 1);

	// We want a WAFEFORMATEXTENSIBLE which is equal to WAVEFORMATPCMEX.
	WaveFormat->Format.wFormatTag = WAVE_FORMAT_EXTENSIBLE;
	
	WaveFormat->Format.nChannels =  MIN( Lc_client_channel_number, Lc_my_channel_number);

		
	
    // Check bitpersamples.
    if ( Lc_client_max_bits_per_spl < Lc_my_min_bit_width)
        WaveFormat->Format.wBitsPerSample = Lc_my_min_bit_width;
    else if ( Lc_client_max_bits_per_spl > Lc_my_max_bit_width)
        WaveFormat->Format.wBitsPerSample = Lc_my_max_bit_width;
    else
        WaveFormat->Format.wBitsPerSample = Lc_client_max_bits_per_spl;
		
	// Check if this sample rate is in the requested data range of the client.
    if (	  (Lc_client_max_sample_freq < Lc_my_min_sample_freq)
		   || (Lc_client_min_sample_freq > Lc_my_max_sample_freq)
		)
	{
			return STATUS_NO_MATCH;
	}

	ULONG   ulFrequency = 0;

    m_AdapterCommon->GetCurrentClock(0, 0, &ulFrequency, NULL, NULL, NULL);

	if (   (ulFrequency > Lc_client_max_sample_freq)
		|| (ulFrequency < Lc_client_min_sample_freq))
	{
        if( ulFrequency != 0 )
        {
			return STATUS_NO_MATCH;
        }
	}
	if (   (ulFrequency <= Lc_my_max_sample_freq)
		&& (ulFrequency >= Lc_my_min_sample_freq))
	{
		WaveFormat->Format.nSamplesPerSec = ulFrequency;
	}
	else
	{
		WaveFormat->Format.nSamplesPerSec = MIN( Lc_client_max_sample_freq, Lc_my_max_sample_freq);
	}

	WaveFormat->Format.nBlockAlign = (WaveFormat->Format.wBitsPerSample * WaveFormat->Format.nChannels) / 8;
	WaveFormat->Format.nAvgBytesPerSec = WaveFormat->Format.nSamplesPerSec * WaveFormat->Format.nBlockAlign;
	WaveFormat->Format.cbSize = 22;		// WAVEFORMATPCMEX
	WaveFormat->Samples.wValidBitsPerSample = WaveFormat->Format.wBitsPerSample;

	WaveFormat->dwChannelMask = 0;		// ** unsupported **

	WaveFormat->SubFormat = KSDATAFORMAT_SUBTYPE_PCM; //subtype of the WAVEFORMATEXTENSIBLE.

	// Now overwrite also the sample size in the ksdataformat structure.
	((PKSDATAFORMAT)ResultantFormat)->SampleSize = WaveFormat->Format.nBlockAlign;
	
	*ResultantFormatLength = sizeof(KSDATAFORMAT) + sizeof(WAVEFORMATPCMEX);
	
	DOUT (DBG_PRINT, ("[DataRangeIntersection] RESULTING Frequency: %d, Channels: %d, bps: %d, ChannelMask: %X",
		WaveFormat->Format.nSamplesPerSec, WaveFormat->Format.nChannels,
		WaveFormat->Format.wBitsPerSample, WaveFormat->dwChannelMask));

	// Let portcls do some work ...
    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICH::PropertyChannelConfig
 *****************************************************************************
 * This is the property handler for KSPROPERTY_AUDIO_CHANNEL_CONFIG of the
 * DAC node. It sets the channel configuration (how many channels, how user
 * was setting up the speakers).
 */
NTSTATUS CMiniportWaveICH::PropertyChannelConfig
(
    IN      PPCPROPERTY_REQUEST PropertyRequest
)
{
    PAGED_CODE ();

    ASSERT (PropertyRequest);

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::PropertyChannelConfig]"));

    NTSTATUS        ntStatus = STATUS_INVALID_PARAMETER;
    // The major target is the object pointer to the wave miniport.
    CMiniportWaveICH *that =
        (CMiniportWaveICH *) (PMINIPORTWAVEPCI)PropertyRequest->MajorTarget;

    ASSERT (that);

    // We only have a set defined.
    if (PropertyRequest->Verb & KSPROPERTY_TYPE_GET)
    {
        // validate buffer size.
        if (PropertyRequest->ValueSize < sizeof(LONG))
            return ntStatus;

        // The "Value" is the input buffer with the channel config.
        if (PropertyRequest->Value)
        {
			DOUT (DBG_PRINT, ("PropertyChannelConfig GET KSAUDIO_SPEAKER_DIRECTOUT\n"));

			*(PLONG)PropertyRequest->Value = KSAUDIO_SPEAKER_DIRECTOUT;
			PropertyRequest->ValueSize = sizeof(LONG);
			ntStatus = STATUS_SUCCESS;

            // We can accept different channel configurations, depending
            // on the number of channels we can play.
            /*if (that->AdapterCommon->GetPinConfig (PINC_SURROUND_PRESENT))
            {
                if (that->AdapterCommon->GetPinConfig (PINC_CENTER_LFE_PRESENT))
                {
                    // we accept 5.1
                    if (*(PLONG)PropertyRequest->Value == KSAUDIO_SPEAKER_5POINT1)
                    {
                      //  that->m_dwChannelMask =  *(PLONG)PropertyRequest->Value;
                      //  that->m_wChannels = 6;
                       // that->AdapterCommon->WriteChannelConfigDefault (that->m_dwChannelMask);
                        ntStatus = STATUS_SUCCESS;
                    }
                }
                
                // accept also surround or quad.
                if ((*(PLONG)PropertyRequest->Value == KSAUDIO_SPEAKER_QUAD) ||
                    (*(PLONG)PropertyRequest->Value == KSAUDIO_SPEAKER_SURROUND))
                {
                   // that->m_dwChannelMask =  *(PLONG)PropertyRequest->Value;
                   // that->m_wChannels = 4;
                   // that->AdapterCommon->WriteChannelConfigDefault (that->m_dwChannelMask);
                    ntStatus = STATUS_SUCCESS;
                }
            }
            */
            // accept also stereo speakers.
          /*  if (*(PLONG)PropertyRequest->Value == KSAUDIO_SPEAKER_STEREO)
            {
               // that->m_dwChannelMask =  *(PLONG)PropertyRequest->Value;
               // that->m_wChannels = 2;
               // that->AdapterCommon->WriteChannelConfigDefault (that->m_dwChannelMask);
                ntStatus = STATUS_SUCCESS;
            }*/
        }
    }

	if (PropertyRequest->Verb & KSPROPERTY_TYPE_SET)
    {
        // validate buffer size.
        if (PropertyRequest->ValueSize < sizeof(LONG))
            return ntStatus;

        // The "Value" is the input buffer with the channel config.
        if (PropertyRequest->Value)
        {
            // We can accept different channel configurations, depending
            // accept also stereo speakers.
            if (*(PLONG)PropertyRequest->Value == KSAUDIO_SPEAKER_DIRECTOUT)
            {
               // that->m_dwChannelMask =  *(PLONG)PropertyRequest->Value;
               // that->m_wChannels = 2;
               // that->AdapterCommon->WriteChannelConfigDefault (that->m_dwChannelMask);
                ntStatus = STATUS_SUCCESS;
            }
        }
		DOUT (DBG_PRINT, ("PropertyChannelConfig (mask %x) SET = %x\n", *(PLONG)PropertyRequest->Value, ntStatus));
    }

    return ntStatus;
} // PropertyChannelConfig


/*****************************************************************************
 * CMiniportWaveICH::PowerChangeNotify
 *****************************************************************************
 * This routine gets called as a result of hooking up the IPowerNotify
 * interface. This interface indicates the driver's desire to receive explicit
 * notification of power state changes. The interface provides a single method
 * (or callback) that is called by the miniport's corresponding port driver in
 * response to a power state change. Using wave audio as an example, when the
 * device is requested to go to a sleep state the port driver pauses any
 * active streams and then calls the power notify callback to inform the
 * miniport of the impending power down. The miniport then has an opportunity
 * to save any necessary context before the adapter's PowerChangeState method
 * is called. The process is reversed when the device is powering up. PortCls
 * first calls the adapter's PowerChangeState method to power up the adapter.
 * The port driver then calls the miniport's callback to allow the miniport to
 * restore its context. Finally, the port driver unpauses any previously paused
 * active audio streams.
 */
STDMETHODIMP_(void) CMiniportWaveICH::PowerChangeNotify
(
    IN  POWER_STATE NewState
)
{
    PAGED_CODE ();
    NTSTATUS ntStatus = STATUS_SUCCESS;

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::PowerChangeNotify]"));

    //
    // Check to see if this is the current power state.
    //
    if (NewState.DeviceState == m_PowerState)
    {
        return;
    }

    //
    // Check the new device state.
    //
    if ((NewState.DeviceState < PowerDeviceD0) ||
        (NewState.DeviceState > PowerDeviceD3))
    {
        DOUT (DBG_ERROR, ("Unknown device state: D%d.",
             (ULONG)NewState.DeviceState - (ULONG)PowerDeviceD0));
        return;
    }

    DOUT (DBG_POWER, ("Changing state to D%d.", (ULONG)NewState.DeviceState -
                    (ULONG)PowerDeviceD0));

    //
    // In case we return to D0 power state from a D3 state, restore the
    // interrupt connection.
    //
    if (NewState.DeviceState == PowerDeviceD0)
    {
		if(m_InterruptSync)
		{
            ntStatus = m_InterruptSync->Connect ();
            if (!NT_SUCCESS (ntStatus))
            {
                DOUT (DBG_ERROR, ("Failed to connect the ISR with InterruptSync!"));
                // We can do nothing else than just continue ...
            }
            DOUT (DBG_PRINT, ("### CMiniportWaveICH  m_InterruptSync->Connect()"));
        }
    }

    //
    // Call the stream routine which takes care of the DMA engine.
    // That's all we have to do.
    //
    if (Streams[WAVE_PIN_WAVEIN])
    {
        ntStatus = Streams[WAVE_PIN_WAVEIN]->PowerChangeNotify (NewState);
        if (!NT_SUCCESS (ntStatus))
        {
            DOUT (DBG_ERROR, ("PowerChangeNotify D%d for WAVEIN failed",
                          (ULONG)NewState.DeviceState - (ULONG)PowerDeviceD0));
        }
    }

    if (Streams[WAVE_PIN_WAVEOUT])
    {
        ntStatus = Streams[WAVE_PIN_WAVEOUT]->PowerChangeNotify (NewState);
        if (!NT_SUCCESS (ntStatus))
        {
            DOUT (DBG_ERROR, ("PowerChangeNotify D%d for WAVEOUT failed",
                          (ULONG)NewState.DeviceState - (ULONG)PowerDeviceD0));
        }
    }

    //
    // In case we go to any sleep state we disconnect the interrupt service
    // reoutine from the interrupt.
    // Normally this is not required to do, but for some reason this fixes
    // a problem where we won't have any interrupts on specific motherboards
    // after resume.
    //
    if (NewState.DeviceState != PowerDeviceD0)
    {
		if(m_InterruptSync)
		{
			m_InterruptSync->Disconnect();
            DOUT (DBG_PRINT, ("### CMiniportWaveICH  m_InterruptSync->Disconnect()"));
        }
    }

    //
    // Save the new state.  This local value is used to determine when to
    // cache property accesses and when to permit the driver from accessing
    // the hardware.
    //
    m_PowerState = NewState.DeviceState;
    DOUT (DBG_POWER, ("MINWAVE Entering D%d", (ULONG)m_PowerState - (ULONG)PowerDeviceD0));
}


/*****************************************************************************
 * CMiniportWaveICH::GetPhysicalConnectionPins
 *****************************************************************************
 * Returns the system pin IDs of the bridge pins that are connected with the
 * wave miniport.
 * If one pin is not used, the value is -1, that could only happen for MinInDest.
 */
STDMETHODIMP CMiniportWaveICH::GetPhysicalConnectionPins
(
    IN ULONG  Index,
    OUT PULONG  Wave
)
{
    PAGED_CODE ();

    ASSERT (Wave);


    // set the pin IDs.
    *Wave = TransPinDefToPinNr (WavePins(Index));

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::GetPhysicalConnectionPins %d : %d]",Index, *Wave ));

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICH::SetTopologyRef
 *****************************************************************************
 * Set the reference to topology manager object
 */
void CMiniportWaveICH::SetTopologyRef
(
    IN      PMINIPORTTOPOLOGYICH    PmTopologyRef
)
{
    PAGED_CODE ();

    DOUT (DBG_PRINT, ("[CMiniportWaveICH::SetTopologyRef]"));

    m_pTopology = PmTopologyRef ;
    m_pTopology->AddRef();
}

#if (DBG)
/*****************************************************************************
 * CMiniportWaveICH::DumpTopology
 *****************************************************************************
 * Dumps the topology for debugging.
 * See the defines at the beginning of this file?
 */
void CMiniportWaveICH::DumpTopology (void)
{
    PAGED_CODE ();

//    {
        // dump the pins
        DOUT (DBG_PINS, ("WAVE MINIPORT PINS"));
        for(ULONG index = 0; index < FilterDescriptor->PinCount; index++)
        {
            DOUT (DBG_PINS, ("  %2d", index));
        }

        // dump the nodes
        DOUT (DBG_NODES, ("WAVE MINIPORT NODES"));
        for(index = 0; index < FilterDescriptor->NodeCount; index++)
        {
            DOUT (DBG_NODES, ("  %2d", index));
        }

        // dump the connections
        DOUT (DBG_CONNS, ("TOPOLOGY MINIPORT CONNECTIONS"));
        for(index = 0; index < FilterDescriptor->ConnectionCount; index++)
        {
            ULONG FromNode = ConnectionDescriptors[index].FromNode;
            ULONG FromNodePin = ConnectionDescriptors[index].FromNodePin;
            ULONG ToNode = ConnectionDescriptors[index].ToNode;
            ULONG ToNodePin = ConnectionDescriptors[index].ToNodePin;

            DOUT (DBG_CONNS, ("  %2d (%d,%d)->(%d,%d)", index,
                FromNode,
                FromNodePin,
                ToNode,
                ToNodePin));
        }
//  }
}
#endif


/*****************************************************************************
 * Non paged code begins here
 *****************************************************************************
 */

#pragma code_seg()

#define BOARD_LOCK()\
    KIRQL   LcOldBoardIrql; \
    \
    DOUT (DBG_SYSINFO, ("[Miniport Wave BOARD_LOCK]"));\
    m_AdapterCommon->AcquireBoardLock(&LcOldBoardIrql);\
    DOUT (DBG_SYSINFO, ("[Wave BOARD_LOCKED]"))

#define BOARD_LOCK_AGAIN()\
    DOUT (DBG_SYSINFO, ("[Miniport Wave BOARD_LOCK_AGAIN]"));\
    m_AdapterCommon->AcquireBoardLock(&LcOldBoardIrql);\
    DOUT (DBG_SYSINFO, ("[Wave BOARD_LOCK_AGAIN]"))

#define BOARD_UNLOCK()\
    DOUT (DBG_SYSINFO, ("[Miniport Wave BOARD_UNLOCK]"));\
    m_AdapterCommon->ReleaseBoardLock(&LcOldBoardIrql)

#define BOARD_LOCK_DPC()\
    DOUT (DBG_SYSINFO, ("[BOARD_LOCK_DPC]"));\
    m_AdapterCommon->AcquireBoardLockAtDpcLevel();\
    DOUT (DBG_SYSINFO, ("[BOARD_LOCKED_DPC]"))

#define BOARD_UNLOCK_DPC()\
    DOUT (DBG_SYSINFO, ("[BOARD_UNLOCK_DPC]"));\
    m_AdapterCommon->ReleaseBoardLockAtDpcLevel()

/*****************************************************************************
 * CMiniportWaveICH::Service
 *****************************************************************************
 * Processing routine for dealing with miniport interrupts.  This routine is
 * called at DISPATCH_LEVEL.
 */
STDMETHODIMP_(void) CMiniportWaveICH::Service (void)
{
    DOUT (DBG_SYSINFO, ("[CMiniportWaveICH::Service]"));

	// LARGE_INTEGER LcPerfFreq;
	// LARGE_INTEGER LcTime1 = KeQueryPerformanceCounter(&LcPerfFreq);
	// DOUT(DBG_PRINT, ("CMiniportWaveICH::Service(APHTraitIt) BEGIN\n"));

    BOARD_LOCK_DPC();

    // Handle any exception on the board
    // This call cannot manage sound transfers since
    // the Api_Hdl data structures are not used, and stay
    // zero-initialized
    //
    TraitHardItBoard(m_sBoardIndex);

    BOARD_UNLOCK_DPC();

	//LARGE_INTEGER LcTime2 = KeQueryPerformanceCounter(NULL);
	//LONGLONG LcTime = LcTime2.QuadPart - LcTime1.QuadPart;
	//if(LcTime)
	//{
	//	LcTime *= 1000000;
	//	LcTime /= LcPerfFreq.QuadPart;
	//	DOUT(DBG_PRINT, ("CMiniportWaveICH::Service(APHTraitIt) = %I64d us\n", LcTime));
	//}
}


NTSTATUS CMiniportWaveICH::NotifyStream( BOOL PmIsRecord, DWORD PmAudio, BOOL PmDirectCall )
{
    if(PmIsRecord == 0)
    {
		if(Streams[WAVE_PIN_WAVEOUT])
		{
			DOUT(DBG_SYSINFO, ("NotifyStream WAVE_PIN_WAVEOUT (PmAudio = %x)\n", PmAudio));

			if(PmDirectCall)
			{
				// direct Service
				Streams[WAVE_PIN_WAVEOUT]->Service();
			}
			else
			{
				// request deferred Service
                if( (m_Port) && (Streams[WAVE_PIN_WAVEOUT]->m_ServiceGroup) )
                {
				    m_Port->Notify(Streams[WAVE_PIN_WAVEOUT]->m_ServiceGroup);
                }
			}
		}
    }
	else
	{
		if(Streams[WAVE_PIN_WAVEIN])
		{
			DOUT(DBG_SYSINFO, ("NotifyStream WAVE_PIN_WAVEIN (PmAudio = %x)\n", PmAudio));

			if(PmDirectCall)
			{
				// direct Service
				Streams[WAVE_PIN_WAVEIN]->Service();
			}
			else
			{
				// request deferred Service
                if( (m_Port) && (Streams[WAVE_PIN_WAVEIN]->m_ServiceGroup) )
                {
				    m_Port->Notify(Streams[WAVE_PIN_WAVEIN]->m_ServiceGroup);
                }
			}
		}
	}
	return STATUS_SUCCESS;
}



VOID  DSNDNotifyStream(	IN LPVOID PmParentIchWave,
						IN BOOL   PmIsRecord,
						IN DWORD  PmAudio,
						IN BOOL   PmDirectCall)
{
    CMiniportWaveICH* that = (CMiniportWaveICH*)PmParentIchWave;

    ASSERT(that);

    if(that)
    {
        that->NotifyStream( PmIsRecord, PmAudio, PmDirectCall );
    }
}


/*****************************************************************************
 * InterruptServiceRoutine
 *****************************************************************************
 * The task of the ISR is to clear an interrupt from this device so we don't
 * get an interrupt storm and schedule a DPC which actually does the
 * real work.
 */
NTSTATUS CMiniportWaveICH::InterruptServiceRoutine
(
    IN  PINTERRUPTSYNC  InterruptSync,
    IN  PVOID           DynamicContext
)
{
    NTSTATUS    ntStatus;
	BOOLEAN     PmTimerOccured = FALSE;
	BOOLEAN     PmAsyncPending = FALSE;
	BOOLEAN     PmAsyncESCMD = FALSE;

    ASSERT (InterruptSync);
    ASSERT (DynamicContext);

    //
    // Get our context which is a pointer to class CMiniportWaveICH.
    //
    CMiniportWaveICH *that = (CMiniportWaveICH *)DynamicContext;

    // Defect #4297 :
    // update protocol pointer, in case the object changed because of a powerchange
	//
	CProtocol *pDsp = (CProtocol*)that->m_AdapterCommon->GetProtocolPtrAsVoid();

    //
    // ACK the ISR.
    //
    if ( pDsp->IEvent_AcknowledgeIRQ(&PmTimerOccured, &PmAsyncPending, &PmAsyncESCMD) ) {

			ntStatus = STATUS_SUCCESS;

			if(PmTimerOccured)
			{
				//if ((that->m_Port) && (that->Streams[WAVE_PIN_WAVEOUT]))
				//{
				//	that->m_Port->Notify (that->Streams[WAVE_PIN_WAVEOUT]->m_ServiceGroup);
				//}
				//if ((that->m_Port) && (that->Streams[WAVE_PIN_WAVEIN]))
				//{
				//	that->m_Port->Notify (that->Streams[WAVE_PIN_WAVEIN]->m_ServiceGroup);
				//}

				APHNotifyAllDSoundPipes( that->m_sBoardIndex );
			}

			if(PmAsyncPending)
			{
				//
				// Request DPC service.
				//
				if ((that->m_Port) && (that->m_pServiceGroupWave))
				{
					that->m_Port->Notify (that->m_pServiceGroupWave);
				}
			}

            if( PmAsyncESCMD )
            {
                ESCMDSInterruptService( that->m_sBoardIndex, FALSE );
            }
    }
	else
	{
		ntStatus = STATUS_UNSUCCESSFUL;
	}

    return ntStatus;
}

/*****************************************************************************
 * CMiniportWaveICH::TestDataFormat
 *****************************************************************************
 * Checks if the passed data format is known to the driver and verifies that
 * the number of channels, the width of one sample match to the AC97
 * specification.
 */
NTSTATUS CMiniportWaveICH::TestDataFormat
(
    IN  PKSDATAFORMAT Format,
    IN  WavePins      /*Pin*/
)
{
    ASSERT (Format);

    DOUT (DBG_PRINT, ("[TestDataFormat]"));

    //
    // KSDATAFORMAT contains three GUIDs to support extensible format.  The
    // first two GUIDs identify the type of data.  The third indicates the
    // type of specifier used to indicate format specifics.  We are only
    // supporting PCM audio formats that use WAVEFORMATEX.
    //
    if (
        !IsEqualGUIDAligned (Format->MajorFormat, KSDATAFORMAT_TYPE_AUDIO) ||
        !IsEqualGUIDAligned (Format->SubFormat, KSDATAFORMAT_SUBTYPE_PCM)
        )
    {
        DOUT (DBG_ERROR, ("Invalid format type!"));
        return STATUS_NOT_SUPPORTED;
    }

    //
    // A WAVEFORMATEX structure should appear after the generic KSDATAFORMAT
    // if the GUIDs turn out as we expect.
    //
    if (!IsEqualGUIDAligned (Format->Specifier, KSDATAFORMAT_SPECIFIER_WAVEFORMATEX))
    {
        DOUT (DBG_ERROR, ("Invalid/Unsupported specifier!"));
        return STATUS_NOT_SUPPORTED;
    }

    //
    // If the size doesn't match, then something is screwed up.
    //
    if (Format->FormatSize < sizeof(KSDATAFORMAT_WAVEFORMATEX))
    {
        DOUT (DBG_ERROR, ("Invalid FormatSize!"));
        return STATUS_NOT_SUPPORTED;
    }

    
    
    PWAVEFORMATEXTENSIBLE waveFormat = (PWAVEFORMATEXTENSIBLE)(Format + 1);
    
    //
    // Print the information.
    //
        DOUT (DBG_PRINT,
             ("TestDataFormat: Frequency: %d, Channels: %d, bps: %d\n",
              waveFormat->Format.nSamplesPerSec,
              waveFormat->Format.nChannels,
              waveFormat->Format.wBitsPerSample));
	if(waveFormat->Format.wFormatTag == WAVE_FORMAT_EXTENSIBLE)	{
		DOUT (DBG_PRINT,
			 ("TestDataFormat: WAVEFORMATEXTENSIBLE : wValidBitsPerSample: %d, ChannelMask: %x\n",
			  waveFormat->Samples.wValidBitsPerSample,
			  waveFormat->dwChannelMask));
	}


	ASSERT( m_PinDataRangesIndex < DR_IDX_INVALID );

	ULONG	Lc_my_max_sample_freq = PinDataRangesStreamStatic[m_PinDataRangesIndex].MaximumSampleFrequency;
	ULONG	Lc_my_min_sample_freq = PinDataRangesStreamStatic[m_PinDataRangesIndex].MinimumSampleFrequency;
	ULONG	Lc_my_min_bit_width   = PinDataRangesStreamStatic[m_PinDataRangesIndex].MinimumBitsPerSample;
	ULONG	Lc_my_max_bit_width   = PinDataRangesStreamStatic[m_PinDataRangesIndex].MaximumBitsPerSample;

    //
    // We only support PCM, 24/16/8-bit.
    //
    if (   (    ( waveFormat->Format.wFormatTag != WAVE_FORMAT_PCM )
             && ( waveFormat->Format.wFormatTag != WAVE_FORMAT_EXTENSIBLE) )
        || (waveFormat->Format.wBitsPerSample & 0x07)             
        || (waveFormat->Format.wBitsPerSample > Lc_my_max_bit_width)
        || (waveFormat->Format.wBitsPerSample < Lc_my_min_bit_width)
       )
    {
        DOUT (DBG_WARNING,  ("TestDataFormat UNSUPPORTED wBitsPerSample: %d", waveFormat->Format.wBitsPerSample));
        return STATUS_NOT_SUPPORTED;
    }

	// check max channels
	//
	ULONG	Lc_my_max_channels = PinDataRangesStreamStatic[m_PinDataRangesIndex].MaximumChannels;

    if ( Lc_my_max_channels < waveFormat->Format.nChannels )
    {
        DOUT (DBG_WARNING,  ("TestDataFormat UNSUPPORTED nChannels: %d", waveFormat->Format.nChannels));
        return STATUS_NOT_SUPPORTED;
    }

    //
    // Check SampleRate
    //
    DWORD  Rate = waveFormat->Format.nSamplesPerSec;

	if ( ( Rate < Lc_my_min_sample_freq) ||	( Rate > Lc_my_max_sample_freq) )
	{
		DOUT (DBG_PRINT | DBG_PROT_FLOW|DBG_VSR,("TestDataFormat :  %d Hz UNSUPPORTED", waveFormat->Format.nSamplesPerSec ));
		return STATUS_INVALID_PARAMETER;
	}
        DOUT (DBG_PRINT | DBG_PROT_FLOW|DBG_VSR,("TestDataFormat : %d Hz OK ", waveFormat->Format.nSamplesPerSec ));

	return STATUS_SUCCESS;
}

