// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

#include "drvdef.h"
#include "drvdbg.h"
#include "unicode.h"   // UNCAllocateUnicodeString
#include "log.h"

//
// Local variables
//
// **************************************************************************
PVOID   PseudoHandle = NULL ;

//
// Prototypes for local functions
//
// **************************************************************************


STATIC VOID LogFilePrintDate( VOID );
STATIC VOID LogFilePrintLine( PCHAR );


STATIC VOID WNTLogFileOpen( PUNICODE_STRING, BOOLEAN );
STATIC VOID WNTLogFileClose( );
STATIC VOID WNTLogFilePrintDate( );
STATIC VOID WNTLogFilePrintLine( PCHAR );

//
// Code implementing local functions
//
// **************************************************************************


STATIC VOID LogFilePrintDate( VOID )
{
    if ( PseudoHandle )
    {
        WNTLogFilePrintDate();
    }
}

/*****************************************************************************/
/* Function LogFilePrintLine                                              */
/*                                                                           */
/*****************************************************************************/
//
STATIC VOID LogFilePrintLine(
    char  *PmStr )
{
    if ( PseudoHandle )
    {
        WNTLogFilePrintLine( PmStr );
    }
}

//
// Code implementing exported functions
//
// **************************************************************************

VOID LOGFileAppend( 
    PVOID PmVoid )
{
    //
    // Open trace file if not done yet
    //
    // **************************************************************************
    if ( !PseudoHandle )
    {
        WNTLogFileOpen( (PUNICODE_STRING) PmVoid, TRUE );   // APPEND
        PseudoHandle = ULongToPtr(0xface2deb) ;

        LogFilePrintLine( "CONTINUE LOGGING SESSION: " );
        LogFilePrintDate();
    }
}

VOID LOGFileOpen( 
    PVOID PmVoid )
{
    //
    // Open trace file if not done yet
    //
    // **************************************************************************
    if ( !PseudoHandle )
    {
        WNTLogFileOpen( (PUNICODE_STRING) PmVoid, FALSE );  // OVERWRITE
        PseudoHandle = ULongToPtr(0xface2deb) ;
        LogFilePrintLine( "START OF LOGGING SESSION: " )  ;
        LogFilePrintDate();
    }
}

VOID LOGFileClose( VOID )
{
    //
    // Close trace file if not done yet
    //
    // **************************************************************************
    if ( PseudoHandle )
    {
        LogFilePrintLine( "END OF LOGGING SESSION: " );
        LogFilePrintDate();
    }
    WNTLogFileClose();
    PseudoHandle = NULL ;
}

VOID LOGFileTrace(
    PCHAR PmStr, ... )
{
    va_list LcArgList ;
    int LcLen;

    static char LcBufferVideo[SIZE_MAX_STRING];

    if ( (!PseudoHandle) || (!PmStr) ) return ;

    va_start( LcArgList, PmStr );
    VSNPRINTF(LcBufferVideo, SIZE_MAX_STRING, PmStr, LcArgList );
    va_end( LcArgList );

    // be sure that it is zero terminated
    LcBufferVideo[SIZE_MAX_STRING-1] = 0;

    // resolve problem with missing carriage return
    //
    LcLen = STRLEN(LcBufferVideo);
    if((LcLen > 0) && (LcBufferVideo[LcLen-1] == '\n'))
    {
        if(LcLen < (SIZE_MAX_STRING-2))
        {
            LcBufferVideo[LcLen-1] = '\r';
            LcBufferVideo[LcLen]   = '\n';
            LcBufferVideo[LcLen+1] = 0;
        }
    }

    LogFilePrintLine( LcBufferVideo );
}



// debug_nt.c



// Size of registry query table ( 1 entry + 1 NULL-entry )
// -------------------------------------------------------
#define LOG_QUERY_TAB_SIZE          2

// Path to driver specific keys
// ----------------------------
#define PCX_PARAMETERS_KEY      L"\\Parameters"


// Default value returned by registry if a key is not found
// --------------------------------------------------------
static WCHAR DefaultKeyValue[]  = L"!NOT FOUND!";

// Names of subkeys
// ----------------
#define PCX_BINARIES_ROOT_KEY   L"BinariesRootPath"

static HANDLE logFileHandle   = (HANDLE) NULL ;
static PSTR   logFileName     = (PSTR) NULL   ;

// ***************************
// Corps des fonctions locales
// ***************************


//{
/*****************************************************************************/
/* Fonction WNTLogFileOpen                                                   */
/*                                                                           */
/* Open (or create if non existing) a log file for debug purpose             */
/* NOTE: this routine can only be called at IRQL PASSIVE_LEVEL.              */
/*       As a direct consequence, the best moment to open the                */
/*       log file is in the DriverEntry() routine.                           */
/*****************************************************************************/
//
STATIC VOID WNTLogFileOpen(
    PUNICODE_STRING PmRegistryPath,
    BOOLEAN         PmAppend )
{
    OBJECT_ATTRIBUTES   lc_objectAttributes ;
    IO_STATUS_BLOCK     lc_io_status_block  ;
    UNICODE_STRING      lc_pu_path_name     ;
    UNICODE_STRING      lc_pu_file_name     ;
    ANSI_STRING         lc_pa_file_name     ;
    UNICODE_STRING      LcParamPath   ;
    UNICODE_STRING      LcRootString  ;
    RTL_QUERY_REGISTRY_TABLE LcParamTable[LOG_QUERY_TAB_SIZE];
    NTSTATUS            LcNtStatus = STATUS_UNSUCCESSFUL ;

    ACCESS_MASK         LcDesiredAccess;
    ULONG               LcCreateDisposition;
    ULONG               LcCreateOptions;

    if ( KeGetCurrentIrql() != PASSIVE_LEVEL )
        return ;

    if ( logFileHandle || !PmRegistryPath )
        return;

    // Initialize empty strings
    // -------------------------
    lc_pu_file_name.Buffer = NULL ;
    lc_pu_path_name.Buffer = NULL ;
    LcRootString.Buffer = NULL ;
    LcParamPath.Buffer  = NULL ;

    if ( !UNCAllocateUnicodeString(&lc_pu_file_name) )
        goto clean_exit ;

    if ( !UNCAllocateUnicodeString(&lc_pu_path_name) )
        goto clean_exit ;

    /*
     *  Convert LOGFILE string into an UNICODE one, because
     *  WindowsNT low-level file system operations use UNICODE STRING
     *  as parameters.
     *  Then, we build the path: %SystemRoot%\System32\drivers\<logfile>.
     */
    RtlInitAnsiString(&lc_pa_file_name,LOGFILE)                   ;
    RtlAnsiStringToUnicodeString(&lc_pu_file_name,&lc_pa_file_name,FALSE)   ;

    LcParamPath.MaximumLength     = 300 ;
    LcParamPath.Length            = 0                         ;
    LcParamPath.Buffer            = (PWSTR) DDK_ALLOC( sizeof(WCHAR) * LcParamPath.MaximumLength );

    if ( LcParamPath.Buffer == NULL )
    {
        LcNtStatus = STATUS_NO_MEMORY ;
        goto clean_exit ;
    }

    if ( !UNCAllocateUnicodeString( &LcRootString ) )
    {
        LcNtStatus = STATUS_NO_MEMORY ;
        goto clean_exit               ;
    }

    // Points to driver parameters
    // ----------------------------
    RtlAppendUnicodeStringToString( &LcParamPath, PmRegistryPath );
    RtlAppendUnicodeToString( &LcParamPath, PCX_PARAMETERS_KEY );

    // NULL-terminate the ParamTable
    // ------------------------------
    BZERO2( LcParamTable, RTL_QUERY_REGISTRY_TABLE, LOG_QUERY_TAB_SIZE );

    LcParamTable[0].QueryRoutine = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
    LcParamTable[0].Flags        = RTL_QUERY_REGISTRY_DIRECT;
    LcParamTable[0].Name         = PCX_BINARIES_ROOT_KEY;
    LcParamTable[0].EntryContext  = (PVOID)&LcRootString;
    LcParamTable[0].DefaultType   = REG_SZ;
    LcParamTable[0].DefaultData   = DefaultKeyValue;
    LcParamTable[0].DefaultLength = 0;

    LcNtStatus = RtlQueryRegistryValues(
                        RTL_REGISTRY_ABSOLUTE | RTL_REGISTRY_OPTIONAL,
                        LcParamPath.Buffer,
                        LcParamTable,
                        NULL,
                        NULL );

    if ( !NT_SUCCESS( LcNtStatus ) )
    {
        goto clean_exit ;
    }
    RtlAppendUnicodeStringToString(&lc_pu_path_name,&LcRootString)   ;
    RtlAppendUnicodeStringToString(&lc_pu_path_name,&lc_pu_file_name)   ;

    InitializeObjectAttributes(
        &lc_objectAttributes,
        &lc_pu_path_name,
        OBJ_CASE_INSENSITIVE,
        NULL,
        NULL) ;                             // No security descriptor

    if( PmAppend )  // APPEND
    {
        LcDesiredAccess     = FILE_APPEND_DATA | SYNCHRONIZE;
        LcCreateDisposition = FILE_OPEN_IF;
    }
    else            // OVERWRITE
    {
        LcDesiredAccess     = FILE_WRITE_DATA | SYNCHRONIZE;
        LcCreateDisposition = FILE_OVERWRITE_IF;
    }

    LcCreateOptions = FILE_NON_DIRECTORY_FILE | FILE_SYNCHRONOUS_IO_NONALERT;

    LcNtStatus = ZwCreateFile(
        &logFileHandle,                     // FileHandle
        LcDesiredAccess,                    // DesiredAccess
        &lc_objectAttributes,               // ObjectAttributes
        &lc_io_status_block,                // StatusBlock
        NULL,                               // AllocationSize
        FILE_ATTRIBUTE_NORMAL,              // FileAttributes
        0,                                  // ShareAccess
        LcCreateDisposition,                // CreateDispositions
        LcCreateOptions,                    // CreateOptions
        NULL,                               // EaBuffer
        0) ;                                // EaLength

clean_exit:

    if(lc_pu_file_name.Buffer)  DDK_FREE(lc_pu_file_name.Buffer) ;    // Release buffer
    if(lc_pu_path_name.Buffer)  DDK_FREE(lc_pu_path_name.Buffer) ;    // Release buffer
    if(LcRootString.Buffer)     DDK_FREE(LcRootString.Buffer);
    if(LcParamPath.Buffer)      DDK_FREE(LcParamPath.Buffer);

    if ( LcNtStatus != STATUS_SUCCESS )
    {
        logFileHandle = (HANDLE) NULL ;
    }

    return;
}

/*****************************************************************************/
/* Fonction WNTLogFileClose                                                     */
/* Close a log file.                                                         */
/* NOTE: this routine can only be called at IRQL PASSIVE_LEVEL.              */
/*       As a direct consequence, the best moment to close the               */
/*       log file is the DriverEntry() or PcxUnload() routine.               */
/*                                                                           */
/*****************************************************************************/
//
STATIC VOID WNTLogFileClose( VOID )
{
    if ( logFileHandle )
    {
        if ( KeGetCurrentIrql() == PASSIVE_LEVEL )
        {
            (VOID) ZwClose( logFileHandle )   ;
            logFileHandle = (HANDLE) NULL   ;
        }
    }
}


// ***************************************************************************
// *  Affiche un retour chariot                                              *
// ***************************************************************************
//
STATIC VOID WNTLogFilePrintNewLine( VOID )
{
    WNTLogFilePrintLine( "\n" );
}


/*****************************************************************************/
/* Fonction WNTLogFilePrintLine                                              */
/*                                                                           */
/*****************************************************************************/
//
STATIC VOID WNTLogFilePrintLine(
    char  *str )
{
    NTSTATUS            lc_ntstatus         ;
    IO_STATUS_BLOCK     lc_io_status_block  ;

    if ( logFileHandle == (HANDLE) NULL )
    {
        return ;
    }

    DPrint("%s",str) ;

    if ( KeGetCurrentIrql() == PASSIVE_LEVEL )
    {
        lc_ntstatus = ZwWriteFile(
            logFileHandle,
            NULL,                           // Event
            NULL,                           // ApcRoutine
            NULL,                           // ApcContext
            &lc_io_status_block,
            str,
            STRLEN(str),
            NULL,                           // ByteOffset
            NULL) ;                         // Key
    }
}


// ***************************************************************************
// Fonction WNTLogFilePrintDate
//
// ***************************************************************************
//
STATIC VOID WNTLogFilePrintDate( VOID )
{
    LARGE_INTEGER   lc_current_time ;
    TIME_FIELDS     lc_time_fields  ;
    static char     buffer[SIZE_MAX_STRING];

    KeQuerySystemTime( &lc_current_time );
    ExSystemTimeToLocalTime( &lc_current_time, &lc_current_time );
    RtlTimeToTimeFields( &lc_current_time, &lc_time_fields );

    LOGFileTrace( "%d/%d/%d (%d:%02d:%02d)\n",
                  lc_time_fields.Day,
                  lc_time_fields.Month,
                  lc_time_fields.Year,
                  lc_time_fields.Hour,
                  lc_time_fields.Minute,
                  lc_time_fields.Second);
}


