// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

#include "shared.h"
#include "common.h"

#include "drvdef.h"
#include "log.h"
#include "buff_hdl.h"
#include "api_hdl.h"
#include "pcxerr_e.h"

#include "lxeswdm.h"
#include "es_cmds.h"

// Synchronization with VX32 api

extern	BYTE			APH_Boards_Count;		
extern	BOARD_INFO		APH_Board_Info_Array[];
extern	CProtocol*	    APH_Protocol_Array[];
extern	CPIOCommands*	APH_Commands_Array[];

// These macros are only defined below
//
#define BOARD_LOCK()\
    KIRQL   LcOldBoardIrql; \
    AcquireBoardLock(&LcOldBoardIrql);

#define BOARD_LOCK_AGAIN()\
    AcquireBoardLock(&LcOldBoardIrql);

#define BOARD_UNLOCK()\
    ReleaseBoardLock(&LcOldBoardIrql)


/*****************************************************************************
 * Static Members
 *****************************************************************************
 */

#pragma code_seg("PAGE")


LONG CAdapterCommon::m_Instance = 0;         // object instance counter


/*****************************************************************************
 * NewAdapterCommon
 *****************************************************************************
 * Create a new adapter common object.
 */
NTSTATUS NewAdapterCommon
(
    OUT PUNKNOWN   *Unknown,
    IN  REFCLSID,
    IN  PUNKNOWN    UnknownOuter    OPTIONAL,
    IN  POOL_TYPE   PoolType
)
{
    PAGED_CODE ();

    ASSERT (Unknown);

    DOUT (DBG_PRINT, ("[NewAdapterCommon]"));

    STD_CREATE_BODY_WITH_TAG_
    (
        CAdapterCommon,
        Unknown,
        UnknownOuter,
        PoolType,
		PCX_COMMON_TAG,
        PADAPTERCOMMON
    );
}


/*****************************************************************************
 * CAdapterCommon::InitDSP
 *****************************************************************************
 * Initialize the adapter common object -> initialize and probe HW.
 * Pass only checked resources.
 */
STDMETHODIMP_(NTSTATUS) CAdapterCommon::InitDSP
(
    IN  PDEVICE_OBJECT  PmDeviceObject,
    IN  LPGENERAL_INFO  PmGeneralInfo,
    IN  WORD            PmBoardIndex
)
{
    PAGED_CODE ();

    NTSTATUS            LcStatus    = STATUS_SUCCESS;
    WORD                LcCurrentBoard = PmBoardIndex;
    WORD                LcDspType = 0;
    BOOL                LcFirstTime = FALSE;

	(void)PmDeviceObject; // unref'd

    LOGFileTrace("[Init Board configuration]\n");

    m_PIO->PIODefaultConfig( );

    LOGFileTrace( "-> Testing config board\n" );

    if ( ! m_PIO->PIOTestConfig( &LcDspType) )
    {
        LcStatus = STATUS_DEVICE_CONFIGURATION_ERROR;
        goto clean_exit;
    }

    // give access to APH
    APH_Commands_Array[m_BoardIndex] = m_PIO;

    InitEtherConfig();

    LOGFileTrace( "... ok ...");

    // Board is running now
    // ---------------------
    m_ResetBoardDone = TRUE;

	// Init Protocol Object

    if( m_pDSP == NULL )
    {
	    m_pDSP = (CProtocol*) new (NonPagedPool, PCX_COM_NEW_TAG) CLXProtocol( &(APH_Board_Info_Array[PmBoardIndex].biTbDspInfo[0]), m_PIO);
        LcFirstTime = TRUE;
    }

    if (!m_pDSP)
    {
        LcStatus = STATUS_INSUFFICIENT_RESOURCES;
        goto clean_exit;
    }

    LOGFileTrace("Protocol Started\n");

    //WORD LcOptions = m_PIO->PIOGetOptions();
    //LOGFileTrace( "-> Options %s%s%s%s available\n",
    //    LcOptions & OPT_TS_CF_EFFECTS ? "TSCF" : "",
    //    LcOptions & OPT_EQUALIZER_EFFECTS ? " EQ" : "",
    //    LcOptions & OPT_MAXIMIZER_EFFECTS ? " MX" : "",
    //    LcOptions & OPT_MPEG123_FORMATS ? " MPEG" : "");

	// Pass Info pointers along to API
	APH_Protocol_Array[PmBoardIndex]	=	m_pDSP;

	APH_Boards_Count++;

clean_exit:

    if ( LcStatus != STATUS_SUCCESS ) {
        LOGFileTrace("... failed (0x%lx)\n", LcStatus);
    }

    if ( LcStatus == STATUS_SUCCESS ) {
        WORD    LcRet       ;

        BYTE    LcSoftNum   = 1 ;
        BYTE    LcFirmNum   = 1 ;
		DWORD   LcDspVersion;

		// save DSP SoftNum (the right one to avoid failing TestVersion !
		APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0].dsEtat = LcSoftNum ;

        // board is ready
        APH_Board_Info_Array[LcCurrentBoard].biCarac = CARAC_USED;

        // set mix latency mode, zero means PCX classic mode.
        // call this prior to checking the version.
        //
        // m_pDSP->IDiag_SetPCMOnlyGranularity(m_pGeneralInfo->PCMOnlyGranularity);

		LcRet = m_pDSP->IInit_GetVersionAndFeatures( LcFirmNum, m_pGeneralInfo, &LcDspVersion );
        APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0].dsDspVersion = LcDspVersion ;

        if ( LcRet & ERROR_MASK ) {

            LOGFileTrace("-> Error : Wrong version of DSP software (%x)\n", LcDspVersion);
//            LcStatus = STATUS_UNSUCCESSFUL;
        }
//        else {

			// set granularity according to registry
			//
			IBL_RESP_INFO	IBL_Info;
							IBL_Info.iblSamples		=  (WORD) m_pGeneralInfo->PCMOnlyGranularity;
							IBL_Info.iblMin			=  0;
							IBL_Info.iblMax			=  0;
							IBL_Info.iblGranularity	=  0;

			m_pDSP->IDiag_HandleBoardIBL( &IBL_Info );

            LOGFileTrace("-> DSP version : V%02d.%02d #%d\n", (LcDspVersion>>16) & 0xff, (LcDspVersion>>8) & 0xff, LcDspVersion & 0xff );
            LOGFileTrace("-> PCM Granularity = %d\n", m_pDSP->IDiag_GetPCMOnlyGranularity() );
            //
            // Valide interrupt on the board.
            // ------------------------------
            //m_PIO->PIOEnableIrq( );

            if(LcFirstTime)
            {
                RegisterBoardAPINP();
            }
//        }
	}

    return ((NTSTATUS) LcStatus) ;
}



/*****************************************************************************
 * CAdapterCommon::Init
 *****************************************************************************
 * Initialize the adapter common object -> initialize and probe HW.
 * Pass only checked resources.
 */
STDMETHODIMP_(NTSTATUS) CAdapterCommon::Init
(
    IN  PRESOURCELIST   ResourceList,
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  LPGENERAL_INFO  GeneralInfo,
    IN  WORD            BoardIndex
)
{
    PAGED_CODE ();

    ASSERT (ResourceList);
    ASSERT (DeviceObject);
    ASSERT (GeneralInfo);

    NTSTATUS ntStatus = STATUS_SUCCESS;

    PBOARD_INFO LcPtrBoardInfo = NULL;

    DOUT (DBG_BOARD, ("[CAdapterCommon::Init]"));

    if( 1 == InterlockedIncrement(&m_Instance) )
    {
        ntStatus = AllocApiNpResources( DeviceObject );
    }

    //
    // Save the device object 
    //
    m_pDeviceObject = DeviceObject;
    m_pGeneralInfo = GeneralInfo;
    m_BoardIndex = BoardIndex;
    m_ResetBoardDone = FALSE;

    LcPtrBoardInfo = &(m_pGeneralInfo->PBoardInfo[BoardIndex]);

    CPIOCommands* pCPIOCommands;
    IOPCIDevice   oIODevice;
    oIODevice.AddressPLX = LcPtrBoardInfo->portPLX;
    oIODevice.AddressDSP = LcPtrBoardInfo->portDSP;
    oIODevice.SizePLX = LcPtrBoardInfo->portPLXsize;
    oIODevice.SizeDSP = LcPtrBoardInfo->portDSPsize;
    oIODevice.dwBoardVersion = LcPtrBoardInfo->biTypeNoAlias;
    oIODevice.ucBoardFlavor  = LcPtrBoardInfo->biFlavor;

	// Spin lock for board SendMessages
	//
    KeInitializeSpinLock (&m_MsgLock);

    pCPIOCommands = new (NonPagedPool, PCX_IOCMDS_TAG) CESIOCommands(&oIODevice, &m_MsgLock);

    if ( pCPIOCommands == NULL )
    {
        LcPtrBoardInfo->biCarac = CARAC_GHOST;
        ntStatus = STATUS_DEVICE_CONFIGURATION_ERROR;
        goto clean_exit ;
    }

    m_PIO = pCPIOCommands;

    pCPIOCommands->PIOSetBoardIndex(    BoardIndex) ;

    // Let's initialize the number of inputs/outputs.

    // init
    m_ulWaveOutNumber = 32;
    m_ulWaveInNumber = 32;
    m_ulMaxWaveInOutNumber = 32;

    //
    // Initialize the device state.
    //
    m_PowerState = PowerDeviceD0;

clean_exit:
    return ntStatus;
}


/*****************************************************************************
 * CAdapterCommon::~CAdapterCommon
 *****************************************************************************
 * Destructor.
 */
CAdapterCommon::~CAdapterCommon ()
{
    PAGED_CODE ();

    DOUT (DBG_BOARD, ("[CAdapterCommon::~CAdapterCommon]"));

    PDSOUND_DEVICE_EXTENSION LcDeviceExtension = (PDSOUND_DEVICE_EXTENSION) m_pDeviceObject->DeviceExtension;

    KeWaitForSingleObject(  LcDeviceExtension->dsdeSemaphorePtr,
                            Executive,
                            KernelMode,
                            FALSE,
                            NULL );

    DOUT (DBG_PRINT, ("NP88 POWER DOWN board=%d (~CAdapterCommon)", m_BoardIndex));

    // cleanup pipe and voie structures and notify pipes
    //
    APHRemoveAllBoardPipeEntry(m_BoardIndex, FALSE);

    APHNotifyBoardPnpEvent(m_BoardIndex,BOARD_EVENT_BOARD_REMOVED);

    // "disconnect" API
	APH_Boards_Count--;

	BZERO2( &APH_Board_Info_Array[m_BoardIndex], BOARD_INFO, 1);
	APH_Protocol_Array[m_BoardIndex]		= NULL;
    APH_Commands_Array[m_BoardIndex]		= NULL;

    // MTR macosX destroy Protocol
	if (m_pDSP) delete(m_pDSP);
	m_pDSP = NULL;

    if (m_ResetBoardDone == TRUE)
    {
        // Let's avoid any further interrupt
        //
        m_PIO->PIODisableIrq( );

		m_PIO->PIOPowerDown();
    }

    if(m_PIO)
    {
        delete m_PIO; // deletes also the CNPBootStrap object
    }

    PBOARD_INFO LcPtrBoardInfo = &(m_pGeneralInfo->PBoardInfo[m_BoardIndex]);

    // Remove the board for our internal structures
    //
    RtlZeroMemory( LcPtrBoardInfo, sizeof(BOARD_INFO) );

    if( 0 == InterlockedDecrement(&m_Instance) )
    {
        ReleaseApiNpResources( );
    }

    KeReleaseSemaphore( LcDeviceExtension->dsdeSemaphorePtr,
                        0,          // priority increment
                        1,          // semaphore counter increment
                        FALSE );    // immediate execution of next waiting thread
}



/*****************************************************************************
 * CAdapterCommon::NonDelegatingQueryInterface
 *****************************************************************************
 * Obtains an interface.  This function works just like a COM QueryInterface
 * call and is used if the object is not being aggregated.
 * We basically just check any GUID we know and return this object in case we
 * know it.
 */
STDMETHODIMP_(NTSTATUS) CAdapterCommon::NonDelegatingQueryInterface
(
    IN  REFIID  Interface,
    OUT PVOID * Object
)
{
    PAGED_CODE ();

    ASSERT (Object);

    DOUT (DBG_PRINT, ("[CAdapterCommon::NonDelegatingQueryInterface]"));

    // Is it IID_IUnknown?
    if (IsEqualGUIDAligned (Interface, IID_IUnknown))
    {
        *Object = (PVOID)(PUNKNOWN)(PADAPTERCOMMON)this;
    }
    else
    // or IID_IAdapterCommon ...
    if (IsEqualGUIDAligned (Interface, IID_IAdapterCommon))
    {
        *Object = (PVOID)(PADAPTERCOMMON)this;
    }
    else
    // or IID_IAdapterPowerManagement ...
    if (IsEqualGUIDAligned (Interface, IID_IAdapterPowerManagement))
    {
        *Object = (PVOID)(PADAPTERPOWERMANAGEMENT)this;
    }
    else
    {
        // nothing found, must be an unknown interface.
        *Object = NULL;
        return STATUS_INVALID_PARAMETER;
    }

    //
    // We reference the interface for the caller.
    //
    ((PUNKNOWN)*Object)->AddRef ();
    return STATUS_SUCCESS;
}

/*****************************************************************************
 * CAdapterCommon::GetWaveOutNumber
 *****************************************************************************
 */
STDMETHODIMP_(ULONG) CAdapterCommon::GetWaveOutNumber()
{
    PAGED_CODE();

    return m_ulWaveOutNumber;
}

/*****************************************************************************
 * CAdapterCommon::GetWaveInNumber
 *****************************************************************************
 */
STDMETHODIMP_(ULONG) CAdapterCommon::GetWaveInNumber()
{
    PAGED_CODE();

    return m_ulWaveInNumber;
}

/*****************************************************************************
 * CAdapterCommon::GetMaxWaveInOutNumber
 *****************************************************************************
 */
STDMETHODIMP_(ULONG) CAdapterCommon::GetMaxWaveInOutNumber()
{
    PAGED_CODE();

    return m_ulMaxWaveInOutNumber;
}

/*****************************************************************************
 * CAdapterCommon::GetBoardIndex
 *****************************************************************************
 */
STDMETHODIMP_(WORD) CAdapterCommon::GetBoardIndex()
{
    PAGED_CODE();

    return m_BoardIndex;
}

/*****************************************************************************
 * CAdapterCommon::PowerChangeState
 *****************************************************************************
 * Change power state for the device.  We handle the codec, PowerChangeNotify
 * in the wave miniport handles the DMA registers.
 */
STDMETHODIMP_(VOID) CAdapterCommon::PowerChangeState
(
    IN  POWER_STATE NewState
)
{
    PAGED_CODE ();

    DOUT (DBG_PRINT, ("[CAdapterCommon::PowerChangeState]"));
    //
    // Check to see if this is the current power state.
    //
    if (NewState.DeviceState == m_PowerState)
    {
        DOUT (DBG_POWER, ("New device state equals old state."));
        return;
    }

    //
    // Check the new device state.
    //
    if ((NewState.DeviceState < PowerDeviceD0) ||
        (NewState.DeviceState > PowerDeviceD3))
    {
        DOUT (DBG_ERROR, ("Unknown device state: D%d.",
             (ULONG)NewState.DeviceState - (ULONG)PowerDeviceD0));
        return;
    }

    DOUT (DBG_POWER, ("Changing state to D%d.", (ULONG)NewState.DeviceState -
                    (ULONG)PowerDeviceD0));

    if(NewState.DeviceState == PowerDeviceD0)
    {
        m_PIO->PIOEnableSendMessage(TRUE);
    }

    // Switch on new state.
    switch (NewState.DeviceState)
    {
        case PowerDeviceD0:
        {
            // If we are coming from D2 or D3 we have to init the card and load the DSP
            // there might have been a power loss.
            //
            if ((m_PowerState == PowerDeviceD3) || (m_PowerState == PowerDeviceD2))
            {
    PDSOUND_DEVICE_EXTENSION LcDeviceExtension = (PDSOUND_DEVICE_EXTENSION) m_pDeviceObject->DeviceExtension;
    KeWaitForSingleObject(  LcDeviceExtension->dsdeSemaphorePtr,
                            Executive,
                            KernelMode,
                            FALSE,
                            NULL );

				//yes : reload all board firmware
				DOUT (DBG_POWER,("[CAdapterCommon::PowerChangeState PowerDeviceD0]\n"));

                DOUT (DBG_POWER,( "-> Loading FPGA firmware\n" ));

				//download the dsp
				InitDSP( m_pDeviceObject, m_pGeneralInfo, m_BoardIndex);

                // after wake up apply all DHS settings for this board
                // TODO - APHWakeUpDHS(m_BoardIndex);

                m_PIO->PIOEnableIrq();

                // clear potential pending ES command and purge its response data
                ESCMDSInterruptService(m_BoardIndex, TRUE);

                DOUT (DBG_PRINT, ("NP88 POWER UP board=%d", m_BoardIndex));

    KeReleaseSemaphore( LcDeviceExtension->dsdeSemaphorePtr,
                        0,          // priority increment
                        1,          // semaphore counter increment
                        FALSE );    // immediate execution of next waiting thread			}
			}
            else    // m_PowerState == PowerDeviceD1
            {
                m_PIO->PIOEnableIrq();
            }

            break;
		}

        case PowerDeviceD1:
		{
            // nothing to do on this device. Either when powering up or 
			// powering down
			DOUT (DBG_POWER,("[CAdapterCommon::PowerChangeState PowerDeviceD1]\n"));
            break;
		}

        case PowerDeviceD2:
        case PowerDeviceD3:
		{
			//are we powering down and card is not yet powered down :
            if ((m_PowerState == PowerDeviceD0) || (m_PowerState == PowerDeviceD1))
			{
				DOUT (DBG_POWER,("[CAdapterCommon::PowerChangeState PowerDeviceD2-3]\n"));

                PDSOUND_DEVICE_EXTENSION LcDeviceExtension = (PDSOUND_DEVICE_EXTENSION) m_pDeviceObject->DeviceExtension;
                KeWaitForSingleObject(  LcDeviceExtension->dsdeSemaphorePtr,
                                        Executive,
                                        KernelMode,
                                        FALSE,
                                        NULL );

                APHRemoveAllBoardPipeEntry(m_BoardIndex, TRUE); // keep DirectSoundPipes

                APHNotifyBoardPnpEvent(m_BoardIndex,BOARD_EVENT_BOARD_REMOVED);

                // "disconnect" API
	            APH_Boards_Count--;

                APH_Protocol_Array[m_BoardIndex] = NULL;
                APH_Commands_Array[m_BoardIndex] = NULL;

                if( m_ResetBoardDone )
                {
		            m_PIO->PIOPowerDown();

                    m_ResetBoardDone = FALSE;
                }

                // Reset control fields
	            APH_Board_Info_Array[m_BoardIndex].biPipeClockMask[0] = 0;
	            APH_Board_Info_Array[m_BoardIndex].biPipeClockMask[1] = 0;
                APH_Board_Info_Array[m_BoardIndex].biCarac = CARAC_GHOST;

                KeReleaseSemaphore( LcDeviceExtension->dsdeSemaphorePtr,
                                    0,          // priority increment
                                    1,          // semaphore counter increment
                                    FALSE );    // immediate execution of next waiting thread

                DOUT (DBG_PRINT, ("NP88 POWER DOWN board=%d", m_BoardIndex));
			}
            break;
		}
    }

    if(NewState.DeviceState != PowerDeviceD0)
    {
        m_PIO->PIODisableIrq();

        m_PIO->PIOEnableSendMessage(FALSE);
    }

    //
    // Save the new state.  This local value is used to determine when to
    // cache property accesses and when to permit the driver from accessing
    // the hardware.
    //
    m_PowerState = NewState.DeviceState;
    DOUT (DBG_POWER, ("COMMON Entering D%d", (ULONG)m_PowerState - (ULONG)PowerDeviceD0));
}


/*****************************************************************************
 * CAdapterCommon::QueryPowerChangeState
 *****************************************************************************
 * Query to see if the device can change to this power state
 */
STDMETHODIMP_(NTSTATUS) CAdapterCommon::QueryPowerChangeState
(
    IN  POWER_STATE NewState
)
{
    PAGED_CODE ();

	DOUT (DBG_PRINT, ("CAdapterCommon::QueryPowerChangeState NewState = PowerDeviceD%d", (ULONG)NewState.DeviceState - (ULONG)PowerDeviceD0 ));

    // check the new state being requested
    switch (NewState.DeviceState)
    {
        case PowerDeviceD0:
			return STATUS_SUCCESS;
        // do not allow to pass to any Sleep state if there is a pipe defined !
        // DDK says : only communicate with hardware at PowerDeviceD0
        case PowerDeviceD1:
	    case PowerDeviceD2:
	    case PowerDeviceD3:
	        //are we powering down ?
            if ((m_PowerState == PowerDeviceD0) || (m_PowerState == PowerDeviceD1))
			{
				//yes : is there an input pipe defined ?
#if(DBG == 0)   //debug : allow sleep mode
				if (IsApiNpPipeDefined() || (m_pDSP && !m_pDSP->IDiag_IsDspRunning())) // does ANY board own a pipe
				{
					//yes : do not allow powering down
					DOUT (DBG_POWER, ("[CAdapterCommon::QueryPowerChangeState] REFUSED"));
					return( STATUS_UNSUCCESSFUL);
				}
#endif
			}

			//in any other case board can be safely be powered down
			DOUT (DBG_POWER, ("[CAdapterCommon::QueryPowerChangeState] GRANTED"));
			return STATUS_SUCCESS;

        default:
            DOUT (DBG_ERROR, ("Unknown device state: D%d.",
                 (ULONG)NewState.DeviceState - (ULONG)PowerDeviceD0));
            return STATUS_NOT_IMPLEMENTED;
    }
}


/*****************************************************************************
 * CAdapterCommon::QueryDeviceCapabilities
 *****************************************************************************
 * Called at startup to get the caps for the device.  This structure provides
 * the system with the mappings between system power state and device power
 * state.  This typically will not need modification by the driver.
 *
 */
STDMETHODIMP_(NTSTATUS) CAdapterCommon::QueryDeviceCapabilities
(
    IN  PDEVICE_CAPABILITIES PowerDeviceCaps
)
{
    PAGED_CODE ();

    DOUT (DBG_PRINT, ("[CAdapterCommon::QueryDeviceCapabilities]"));

/*
    for (ULONG i=ULONG(PowerSystemWorking); i<=ULONG(PowerSystemShutdown); i++)
    {
        DOUT (DBG_PRINT, ("PowerDeviceCaps->DeviceState[%d] = %d\n", i, PowerDeviceCaps->DeviceState[i]));
    }
    // PowerDeviceCaps->DeviceState[1] = PowerDeviceD0
    // PowerDeviceCaps->DeviceState[2] = PowerDeviceD1
    // PowerDeviceCaps->DeviceState[3] = PowerDeviceUnspecified
    // PowerDeviceCaps->DeviceState[4] = PowerDeviceD3
    // PowerDeviceCaps->DeviceState[5] = PowerDeviceD3
    // PowerDeviceCaps->DeviceState[6] = PowerDeviceD3
*/
    return STATUS_SUCCESS;
}

/*****************************************************************************
 * CAdapterCommon::GetBoardType
 *****************************************************************************
 * Get adapter type information
 */
STDMETHODIMP_(NTSTATUS) CAdapterCommon::GetBoardType
(
    OUT PWORD           PmPFamilyType,
    OUT PWORD           PmPTypeNotAliased,
    OUT PWORD           PmPOptions
)
{
    PAGED_CODE ();

    if (PmPFamilyType)      *PmPFamilyType      = m_PIO->PIOGetType() ;
    if (PmPTypeNotAliased)  *PmPTypeNotAliased  = m_PIO->PIOGetTypeNoAlias() ;
    if (PmPOptions)         *PmPOptions         = m_PIO->PIOGetOptions() ;

    return STATUS_SUCCESS ;
}

/*****************************************************************************
 * Non paged code begins here
 *****************************************************************************
 */

#pragma code_seg()


VOID CAdapterCommon::RegisterBoardAPINP()
{
    BOARD_LOCK();

	APHNotifyBoardPnpEvent(m_BoardIndex,BOARD_EVENT_NEW_BOARD);

    BOARD_UNLOCK();

    // vxcall - DspUse
    //
	PDSP_USE_REQ_INFO            LcRegistrationRequest = NULL;
	PANY_UNMAPPING_REPLY_INFO    LcRegistrationReply = NULL;

	LcRegistrationRequest = (PDSP_USE_REQ_INFO) DDK_ALLOC_WITH_TAG(
									NonPagedPool,
									sizeof(*LcRegistrationRequest),
									PCX_COMMON_TAG);
	LcRegistrationReply = (PANY_UNMAPPING_REPLY_INFO) DDK_ALLOC_WITH_TAG(
									NonPagedPool,
									sizeof(*LcRegistrationReply),
									PCX_COMMON_TAG);

	if( LcRegistrationRequest!=NULL && LcRegistrationReply != NULL )
	{
		RtlZeroMemory(LcRegistrationRequest,sizeof(*LcRegistrationRequest));
		RtlZeroMemory(LcRegistrationReply,sizeof(*LcRegistrationReply));

		LcRegistrationRequest->Header.hdFamily   = GENE_FAM;
		LcRegistrationRequest->Header.hdComNum   = DSP_MANAGE_CMD;
		LcRegistrationRequest->Header.hdBrSize   = sizeof(ANY_UNMAPPING_REPLY_INFO);
		LcRegistrationRequest->Header.hdHandle   = DS_FAKED_APPINDEX + 1;

		// Filling the other fields of the request block
		// ---------------------------------------------
		LcRegistrationRequest->duqBoardMask        = UTIMaskWord(m_BoardIndex);
		LcRegistrationRequest->duqDspMask          = 1;
		LcRegistrationRequest->duqNbBuff           = DSP_USE_CMD_CDE;
		LcRegistrationRequest->duqDspFeatures      = 0;
		LcRegistrationRequest->duqMustFakeFrequency= 0;
		LcRegistrationRequest->duqNumLog           = m_pDSP->IInit_GetDspSoftNum();

	    BOARD_LOCK_AGAIN();

		APHDispatch((LPDWORD)LcRegistrationRequest,
					(LPDWORD)LcRegistrationReply,
					NULL);

	    BOARD_UNLOCK();
	}
	if(LcRegistrationRequest) DDK_FREE_WITH_TAG(LcRegistrationRequest, PCX_COMMON_TAG) ;
	if(LcRegistrationReply) DDK_FREE_WITH_TAG(LcRegistrationReply, PCX_COMMON_TAG) ;

	return;
}


VOID CAdapterCommon::InitEtherConfig()
{
	CHAR LcSerialNum[SERIAL_NUMBER_LEN]= "LX6464ES SerNum";

	LcSerialNum[SERIAL_NUMBER_LEN-1] = 0;
	LOGFileTrace("-> Board Serial Number : %s\n", LcSerialNum );

    // get registry config per SerialNumber : ConfES !

    UNICODE_STRING  LcUnicodeSerial;
    WCHAR           LcUnicode[SERIAL_NUMBER_LEN];
    ANSI_STRING     LcAnsiString;
    DWORD           LcConfES;

    RtlInitAnsiString(&LcAnsiString,LcSerialNum);
    LcUnicodeSerial.MaximumLength   = sizeof(LcUnicode);
    LcUnicodeSerial.Length          = 0;
    LcUnicodeSerial.Buffer          = LcUnicode;
    RtlAnsiStringToUnicodeString(&LcUnicodeSerial,&LcAnsiString,FALSE);

    GetSerialnumConfigFromRegistry(
            &LcUnicodeSerial,
            &LcConfES);

	LOGFileTrace("-> ES (write) sample rate : %d\n", (LcConfES >> FREQ_RATIO_OFFSET ) & 0x0F );

	if (LcConfES & (1L<<MAC_PROG_SET_BIT)) LOGFileTrace("-> ES (write) : MAC PROG SET !!" );

    BOARD_LOCK();

    m_PIO->PIOUpdateESConfig( LcConfES );

    // get the new m_piStereoInOutNumber

    WORD LcStereoOut;
    WORD LcStereoIn;
    m_PIO->PIOGetNbIOs( &LcStereoOut, &LcStereoIn);

    BOARD_UNLOCK();

    LOGFileTrace("-> Config : %d IN  %d OUT\n", LcStereoIn*2, LcStereoOut*2);

    ASSERT(LcStereoOut <= m_ulWaveOutNumber);
    ASSERT(LcStereoIn <= m_ulWaveInNumber);

    m_ulWaveOutNumber = LcStereoOut;
    m_ulWaveInNumber = LcStereoIn;
    m_ulMaxWaveInOutNumber = MAX(m_ulWaveOutNumber, m_ulWaveInNumber);
}



// We must convert into a DSound value (1/65536 dB), the received 
// driver-coded value where 0x1b7 == 0dB (or 0x1ff = +18dB)
// This computation does not make any assumption about the underlying
// h/w
//
// The solution is :
//
//   dsound_level = ( driver_level - LcDriverCodedValueFor0dB ) * 16384 
//
STDMETHODIMP_(LONG) CAdapterCommon::ConvertDriver2DSound
(
    IN WORD PmDriverLevel,
    IN TopoNodes PmType
)
{
    // normalized levels (1 dB == 4 step)
    LONG LcDSoundLevel = (LONG)PmDriverLevel - (LONG)0x1b7;

    // DirectSound level (1 dB == 65536 step)
    LcDSoundLevel <<= 14; // (*=16384)

	DOUT( DBG_PROPERTY ,("Driver %x => DSound %d\n", PmDriverLevel, LcDSoundLevel>>16));

    return LcDSoundLevel;
}

// We must convert the value (1/65536 dB) we received into
// a driver coded value where 0x1b7 is full power for the board.
// We need a board-dependent constant to make the conversion.
// LcDriverCodecBoardValueFor0dB is the driver value that
// corresponds to a board setting of 0dBu.
// The solution is :
//
//     driver_level = ( dsound_level / 16384 ) + LcDriverCodecBoardValueFor0dB
//

STDMETHODIMP_(WORD) CAdapterCommon::ConvertDSound2Driver
(
    IN LONG PmDSound,
    IN TopoNodes PmType
)
{
    LONG LcDriverLevelMAX = 0x1b7;  // analog max level is driver coded 0dB
    LONG LcDriverLevel = 0;

    LcDriverLevel = PmDSound>>14; // (PmDSound/16384)
    LcDriverLevel += (LONG)0x1b7; // normalize to driver level 0 dB

    if( LcDriverLevel < 0 )
        LcDriverLevel = 0;

    if( LcDriverLevel > LcDriverLevelMAX )
        LcDriverLevel = LcDriverLevelMAX;

	DOUT( DBG_PROPERTY ,("DSound %d => Driver %x\n", PmDSound>>16, LcDriverLevel));

    return (WORD)LcDriverLevel;
}


/*****************************************************************************
 * CAdapterCommon::GetGeneralInfo
 *****************************************************************************
 * Get adapter type information
 */
VOID* CAdapterCommon::GetProtocolPtrAsVoid
(
    VOID
)
{
    return(m_pDSP) ;
}

VOID* CAdapterCommon::GetIOCommandsPtrAsVoid
(
    VOID
)
{
    return (m_PIO) ;
}

/*****************************************************************************
 * CAdapterCommon::AcquireBoardLock
 *****************************************************************************
 */
STDMETHODIMP_(VOID) CAdapterCommon::AcquireBoardLock(
    IN  PKIRQL  PmPOldIrql )
{
    //
    // Acquire the mapping spin lock.
    //
    KeAcquireSpinLock (&(m_pGeneralInfo->DpcIoctlSpinLock),PmPOldIrql);
}

/*****************************************************************************
 * CAdapterCommon::ReleaseBoardLock
 *****************************************************************************
 */
STDMETHODIMP_(VOID) CAdapterCommon::ReleaseBoardLock(
    IN  PKIRQL  PmPOldIrql )
{
    //
    // Release the mapping spin lock.
    //
    KeReleaseSpinLock (&(m_pGeneralInfo->DpcIoctlSpinLock),*PmPOldIrql);
}


/*****************************************************************************
 * CAdapterCommon::AcquireBoardLockAtDpcLevel
 *****************************************************************************
 */
STDMETHODIMP_(VOID) CAdapterCommon::AcquireBoardLockAtDpcLevel(
    VOID )
{
    //
    // Acquire the mapping spin lock.
    //
    KeAcquireSpinLockAtDpcLevel (&(m_pGeneralInfo->DpcIoctlSpinLock));
}

/*****************************************************************************
 * CAdapterCommon::ReleaseBoardLock
 *****************************************************************************
 */
STDMETHODIMP_(VOID) CAdapterCommon::ReleaseBoardLockAtDpcLevel(
    VOID )
{
    //
    // Release the mapping spin lock.
    //
    KeReleaseSpinLockFromDpcLevel(&(m_pGeneralInfo->DpcIoctlSpinLock));
}

/*****************************************************************************
 * CAdapterCommon::SetBoardClock
 *****************************************************************************
 * Called to set clock frequency and acquire clock control
 *
 */
STDMETHODIMP_(NTSTATUS) CAdapterCommon::SetBoardClock
(
    IN  DWORD   PmFrequency,
    IN  DWORDLONG PmInPipeMask,
    IN  DWORDLONG PmOutPipeMask
)
{
    WORD    LcRet = SUCCESS ;
    BYTE    LcClockType;
    BYTE    LcSyncPresent;
    DWORD   LcFrequency;
    WORD    LcSyncInNum;
    BOOL    LcFrequencyIsLocked;

#ifdef DBG_VIEWER
    static char* ClockType[] =
    {   "NONE",
	"INTERNAL",
	"SYNCHRO_CARD",
	"UER_SYNCHRO",
	"WORD_CLOCK",
	"PROGRAMMABLE",
	"VIDEO",
	"COBRANET",
	"AES3_ON_INTERNAL_CONNECTOR",
	"ETHERSOUND"};
	
#endif  // DBG_VIEWER

    // look if there is already a locked clock
    if( GetCurrentClock(PmInPipeMask, PmOutPipeMask, &LcFrequency, &LcFrequencyIsLocked, &LcClockType, &LcSyncInNum))
	{
		// does it match ?
		if( LcFrequencyIsLocked && (LcFrequency != PmFrequency) )
		{
			DOUT (DBG_ERROR, ("ERROR AC::SetBoardClock CurrentSampleRate = %d / new = %d\n",LcFrequency, PmFrequency));
			return STATUS_UNSUCCESSFUL;
		}

        DOUT (DBG_VSR, ("WARNING AC::SetBoardClock Locked Clock ! set to %d %s [sync %d]\n", LcFrequency, ClockType[LcClockType], LcSyncInNum));
	}
    else    // the clock is free
    {
        // re-use the current LcClockType and LcSyncInNum !
        //
        // Test if the external clock is available:
        if( LcClockType != CLOCK_TYPE_INTERNAL )
        {
            if( LcFrequency != PmFrequency )
            {
	            // if the card could not calc the frequency of the new external clock
	            // it returns frequency 0; continue in this case we have to set the new
	            // clock before knowing it's frequency !
	            if( LcFrequency != 0 )
	            {
		            DOUT (DBG_ERROR, ("AC::SetBoardClock(%d) NOK ==> wrong external frequency (=%d)\n", PmFrequency, LcFrequency));
		            return STATUS_UNSUCCESSFUL ;
	            }
            }
        }
        DOUT (DBG_VSR, ("AC::SetBoardClock TO %d %s\n", PmFrequency, ClockType[LcClockType]));
    }

    BOARD_LOCK();

    SAMPLE_CLK_REQ_INFO       LcRequest;

    RtlZeroMemory( &LcRequest, sizeof(SAMPLE_CLK_REQ_INFO) );

    LcRequest.scqInputMask64.QuadPart = PmInPipeMask;
    LcRequest.scqOutputMask64.QuadPart = PmOutPipeMask;
    LcRequest.scqSource			= LcClockType;
    LcRequest.scqFrequency		= PmFrequency;
    LcRequest.scqClockBoardNum	= m_BoardIndex ;

    if( LcClockType == CLOCK_TYPE_UER_SYNCHRO )
    {
        LcRequest.scqSyncInputNum   = LcSyncInNum;
    }
    // WordClockNum and others : always 0 !

    LcRet = APHSampleClockSynchronizeOnlyOneBoard(  &LcRequest,
                                                    m_BoardIndex );
    BOARD_UNLOCK();

    if ( LcRet & ERROR_MASK ) {

		DOUT (DBG_ERROR, ("AC::SetBoardClock NOK ==> (to %ld %s) API error = %lx\n", PmFrequency, ClockType[LcClockType], LcRet));
        return STATUS_INVALID_PARAMETER ;
    }

    return  STATUS_SUCCESS ;
}

/*****************************************************************************
 * CAdapterCommon::GetCurrentClock
 *****************************************************************************
 * Get current clock settings
 * Return FALSE if released (hence unknown)
 * Return TRUE if locked by some pipe(s)
 */
STDMETHODIMP_(BOOL) CAdapterCommon::GetCurrentClock
(
	DWORDLONG PmMaskPipeIN,
	DWORDLONG PmMaskPipeOUT,
    LPDWORD PmPFrequency,
    LPBOOL  PmPFrequencyLocked,
    LPBYTE  PmPClockType,
    LPWORD  PmPSyncInNum
)
{
    WORD    LcRet;
    WORD    LcSyncInNum;
    BYTE    LcClockType;
    BYTE    LcSyncPresent;
    DWORD   LcFrequency;
	BOOL    LcIsLocked;
    BOOLEAN LcFrequencyLocked;
	
	// retrieve current board clock configuration 
	//
    BOARD_LOCK();

	LcRet = m_PIO->PIOReadActualSyncForBoard( &LcClockType, &LcSyncPresent, &LcFrequency );

    LcSyncInNum = (WORD)m_PIO->PIOGetActualClkInputNum();

	// are there Pipes allocated by DSound
	//
	if( (APH_Board_Info_Array[m_BoardIndex].biPipeClockMask[NO_MASK_PIPE_OUT] & ~PmMaskPipeOUT) | 
		(APH_Board_Info_Array[m_BoardIndex].biPipeClockMask[NO_MASK_PIPE_IN] & ~PmMaskPipeIN ) )
    {
        LcIsLocked = TRUE;
        LcFrequencyLocked = TRUE;
    }
	else
    {
        LcIsLocked = APHIsClockDHSLocked(m_BoardIndex, &LcFrequencyLocked);
    }

    BOARD_UNLOCK();

    if(PmPFrequency)
        *PmPFrequency = LcFrequency;
    if(PmPClockType)
        *PmPClockType = LcClockType;
    if(PmPSyncInNum)
        *PmPSyncInNum = LcSyncInNum;
    if(PmPFrequencyLocked)
        *PmPFrequencyLocked = (BOOL)LcFrequencyLocked;

	return LcIsLocked;
}

/*****************************************************************************
 * CAdapterCommon::ReleaseClockSettings
 *****************************************************************************
 * Called to release clock control
 *
 */
STDMETHODIMP_(NTSTATUS) CAdapterCommon::ReleaseClockSettings
(
    IN  DWORDLONG PmInPipeMask,
    IN  DWORDLONG PmOutPipeMask
)
{
    NTSTATUS    ntStatus = STATUS_SUCCESS;
    WORD        LcRet;

    BOARD_LOCK();

	DOUT (DBG_PRINT, ("AC::ReleaseClockSettings in:%x  out:%x\n", PmInPipeMask, PmOutPipeMask ) );
	DOUT (DBG_PRINT, (" -> APH reserved are : owners/in:%x  owners/out:%x\n",
						APH_Board_Info_Array[m_BoardIndex].biPipeClockMask[NO_MASK_PIPE_IN],
						APH_Board_Info_Array[m_BoardIndex].biPipeClockMask[NO_MASK_PIPE_OUT] ) );
	
	LcRet = APHSampleClockSetNoClock(	DS_FAKED_APPINDEX,
										PmInPipeMask,
										PmOutPipeMask );
    BOARD_UNLOCK();

    if (LcRet)
	{
		ntStatus = STATUS_UNSUCCESSFUL;
        DOUT (DBG_ERROR, ("AC::ReleaseClockSettings => APHSampleClockSetNoClock failed with %lx", LcRet));
    }
    
    return ntStatus ;
}
