// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

// *************************************************************************
//
// STATIC WORD SampleClockCheckPipes(...)
//
// Input Parameters:
// *****************
//
//  LPSAMPLE_CLK_REQ_INFO PmLPtrReq     : the request block
//
// Output Parameters:
// *******************
//
//  PBOOLEAN PmPMoreThanOneBoard  : TRUE if the pipes belong to more than one board.
//  PBOOLEAN PmPPmPStateChange    : TRUE if this request changes something on the
//                                  pipes state.
//  PBOOLEAN PmPFrequencyChange   : TRUE if the requested frequency is different
//                                  from the previous one.
//  PWORD    PmPABoardNum         : one of the boards on which the pipes are.
//
// *************************************************************************
//
//  This is a SampleClockPipeFct subfunction.
//  It checks whether the request is correct from the pipes point of view.
//
// *************************************************************************
STATIC WORD SampleClockCheckPipes(
    IN  LPSAMPLE_CLK_REQ_INFO PmLPtrReq,
    OUT PBOOLEAN PmPMoreThanOneBoard,
    OUT PBOOLEAN PmPStateChange,
    OUT PBOOLEAN PmPFrequencyChange,
    OUT PWORD    PmPABoardNum)
{
    WORD                LcRet          = SUCCESS;
    SHORT               LcNumPrecBoard = -1;
    SHORT               LcPipeIndex;
    DWORDLONG           LcPipeMask;
    DWORDLONG           LcPipeMaskIn;
    DWORDLONG           LcPipeMaskOut;
    PPIPE_INFO          LcPtPipe;
    WORD                LcNumBoard;
    WORD                i;
    LPBC_HEADER_INFO    LcLPtrReqHeader;
    BYTE                LcAppIndex;

    // Retrieve application index
    // **************************
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmLPtrReq;
    LcAppIndex      = DecodeHandle( LcLPtrReqHeader->hdHandle );

    *PmPMoreThanOneBoard   = FALSE;
    *PmPStateChange        = TRUE;
    *PmPFrequencyChange    = TRUE;
    *PmPABoardNum          = 0;

    // Retrieve pipe mask
    // ******************
    LcPipeMaskOut = PmLPtrReq->scqOutputMask64.QuadPart;
    LcPipeMaskIn  = PmLPtrReq->scqInputMask64.QuadPart;

    // Check all the pipe are actually owned by the application
    // ********************************************************
    LcRet = ChkPmManyInOrOutPipeMask( LcAppIndex,
                                      LcPipeMaskOut,
                                      LcPipeMaskIn );
    if ( LcRet != SUCCESS )
    {
        return( LcRet );
    }

    // Verify the mask of pipes is correct
    // ***********************************
    for ( i = 0 ; i < 2 ; i++ )
    {
        LcPipeIndex = -1;

        switch( i )
        {
        case 0 :
            LcPipeMask  = LcPipeMaskOut;
            LcPtPipe    = &TbPipeOutInfo[0];
            break;
        case 1 :
            LcPipeMask  = LcPipeMaskIn;
            LcPtPipe    = &TbPipeInInfo[0];
            break;
        }

        while ( LcPipeMask != 0 )
        {
            BOOLEAN PmFreqDhsFixed;
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );
            LcNumBoard = LcPtPipe[LcPipeIndex].piNumBoard;

            // If the pipe is already clocked with other pipes
            // -----------------------------------------------
            if (    ( LcPtPipe[LcPipeIndex].piClockSource != CLOCK_TYPE_NONE )
                 && (    ( LcPtPipe[LcPipeIndex].piPipeInClockMask  != LcPipeMaskIn )
                      || ( LcPtPipe[LcPipeIndex].piPipeOutClockMask != LcPipeMaskOut ) ) )
            {
                EXIT_PCX_ERROR( ED_SET_CLOCK_THESEPIPES_REFUSED );
            }

            // If the board is different of the preceding one
            // ----------------------------------------------
            if (    ( LcNumPrecBoard != -1 )
                 && ( LcNumPrecBoard != LcNumBoard ) )
            {
                *PmPMoreThanOneBoard = TRUE;
            }

            // Has DHS taken control over clocks ?
            //
            if( APHIsClockDHSLocked( LcNumBoard, &PmFreqDhsFixed ) )
            {
                BOOL LcError = FALSE;

                while( PmLPtrReq->scqSource != CLOCK_TYPE_NONE )
                {
                    // if compatible clock, do as if NOT FrequencyFixed, to avoid error messages
                    //
                    if( APH_Commands_Array[LcNumBoard]->PIOGetActualClkSource() != PmLPtrReq->scqSource )
                    {
                        // accept changing between internal and programmable clock
                        switch (APH_Commands_Array[LcNumBoard]->PIOGetActualClkSource() )
                        {
                        case CLOCK_TYPE_INTERNAL :
                            if (PmLPtrReq->scqSource != CLOCK_TYPE_PROGRAMMABLE_CLOCK)
                                LcError = TRUE;
                            break;
                        case CLOCK_TYPE_PROGRAMMABLE_CLOCK :
                            if (PmLPtrReq->scqSource != CLOCK_TYPE_INTERNAL)
                                LcError = TRUE;
                            break;
                        default :
                            LcError = TRUE;
                            break;
                        }
                    }
                    if( (APH_Commands_Array[LcNumBoard]->PIOGetActualClkType() != INDEPENDANT_CLOCK)                ||
	                    (APH_Commands_Array[LcNumBoard]->PIOGetActualClkInputNum() != PmLPtrReq->scqSyncInputNum)   )
                    {
                        LcError = TRUE;
                        break;
                    }
                    if( (PmLPtrReq->scqSource == CLOCK_TYPE_INTERNAL)   ||
                        (PmLPtrReq->scqSource == CLOCK_TYPE_PROGRAMMABLE_CLOCK) )
                    {
                        if( (PmFreqDhsFixed != FALSE) &&
                            (APH_Commands_Array[LcNumBoard]->PIOGetActualClockFrequency() != PmLPtrReq->scqFrequency) )
                        {
                            LcError = TRUE; // test frequency only for internal clocks !
                            break;
                        }
                    }
                    break;  // LcError = FALSE; while only once !
                }

                if( LcError )
                {
                    DWORDLONG LcAudioMask = LcPtPipe[LcPipeIndex].piPhysAudioMask;
                    WORD k=0;
                    while(LcAudioMask)
                    {
                        if(LcAudioMask & 1)
                        {
                            if( APHChkIsAudioDHSControlled(LcNumBoard, k, i!=0 ) )
                            {
                                EXIT_PCX_ERROR( ED_CLOCK_MANAGED_BY_DHS_PANEL );
                            }
                        }
                        LcAudioMask >>= 1;
                        k++;
                    }

                    EXIT_PCX_ERROR( ED_SET_CLOCK_REFUSED );
                }
            }

            LcNumPrecBoard = LcNumBoard;
        } // for each pipe

        // if there was at least one pipe of this type
        // Remark : Perhaps this tests will be done twice (for in and out pipe).
        // That does not matter because it will give the same result
        // -------------------------------------------
        if ( LcPipeIndex != -1 )
        {
            if (    ( LcPtPipe[LcPipeIndex].piClockSource   == PmLPtrReq->scqSource )
                 && ( LcPtPipe[LcPipeIndex].piClockBoardNum == PmLPtrReq->scqClockBoardNum )
                 && ( LcPtPipe[LcPipeIndex].piSyncInputNum  == PmLPtrReq->scqSyncInputNum ))
            {
                *PmPStateChange = FALSE;
            }

            if ( LcPtPipe[LcPipeIndex].piFrequency == PmLPtrReq->scqFrequency )
            {
                *PmPFrequencyChange = FALSE;
            }
        }
    }
    // End verify this mask of pipes is correct
    // ****************************************

    *PmPABoardNum = LcNumPrecBoard;

    return( LcRet );
}


// *************************************************************************
//
// WORD APHSampleClockSetNoClock(...)
//
// Input Parameters:
// *****************
//
//  BYTE     PmAppIndex    : index of the application
//  DWORD    PmPipeMaskIn  : input pipe mask
//  DWORD    PmPipeMaskOut : output pipe mask
//
// Output Parameters:
// *******************
//
//  None
//
// *************************************************************************
//
//  This is a SampleClockPipeFct subfunction.
//  It set CLOCK_TYPE_NONE for all the pipes passed in parameters, and eventually
//  set CLOCK_TYPE_NONE for all the concerned boards.
//
// *************************************************************************
WORD APHSampleClockSetNoClock(
    IN  BYTE     PmAppIndex,
    IN  DWORDLONG PmPipeMaskIn,
    IN  DWORDLONG PmPipeMaskOut )
{
    WORD        LcRet     = SUCCESS;
    SHORT       LcPipeIndex;
    DWORDLONG   LcPipeMask;
    PPIPE_INFO  LcPtPipe;
    PVOIE_INFO  LcPtVoie;
    WORD        LcNumBoard;
    WORD        i;

    // For all the pipes concerned by the command
    // ------------------------------------------
    for ( i = 0 ; i < 2 ; i++ )
    {
        if( i == 0 )
        {
            LcPipeMask = PmPipeMaskOut;
            LcPtPipe   = &TbPipeOutInfo[0];
        } else {
            LcPipeMask = PmPipeMaskIn;
            LcPtPipe   = &TbPipeInInfo[0];
        }

        while ( LcPipeMask != 0 )
        {
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            LcNumBoard = LcPtPipe[LcPipeIndex].piNumBoard;
            ASSERT( LcNumBoard < MAX_BOARD );

            BYTE LcNumAudio = LcPtPipe[LcPipeIndex].piNumPipeForDsp;

            if( i == 0 ) {
                ASSERT( LcNumAudio < MAX_BOARD_OUTPUTS );
                LcPtVoie   = &TbVoieOutInfo[LcNumBoard][LcNumAudio];
            } else {
                ASSERT( LcNumAudio < MAX_BOARD_INPUTS );
                LcPtVoie   = &TbVoieInInfo[LcNumBoard][LcNumAudio];
            }

			if(LcPtVoie->IndexAppli != (PmAppIndex + 1))
			{
				// protection for DSound who calls this function directly
				EXIT_PCX_ERROR( ED_INVALID_PIPE );
			}
			
            // The pipe LcPipeIndex is in the requested mask
            // Set CLOCK_TYPE_NONE for this pipe
            LcPtPipe[LcPipeIndex].piClockSource   = CLOCK_TYPE_NONE;
            LcPtPipe[LcPipeIndex].piClockBoardNum = 0;
            LcPtPipe[LcPipeIndex].piFrequency     = 0;
            LcPtPipe[LcPipeIndex].piSyncInputNum  = 0;

            // Update the APH_Board_Info_Array record related to the board on which
            // the current pipe belong to.

            APH_Board_Info_Array[LcNumBoard].biPipeClockMask[NO_MASK_PIPE_OUT] &= ~PmPipeMaskOut ;
            APH_Board_Info_Array[LcNumBoard].biPipeClockMask[NO_MASK_PIPE_IN]  &= ~PmPipeMaskIn ;
        }
        // for each pipe
    }
    // End for all the pipes
    // ---------------------

    // End for all the boards
    // ----------------------

    return( LcRet );
}


// *************************************************************************
//
// STATIC VOID UpdateBoardAndPipeStructure(...)
//
// Input Parameters:
// *****************
//
//  LPSAMPLE_CLK_REQ_INFO   PmLPtrReq      : the request block
//  WORD                    PmBoardNum     : the board number on which the
//                                           pipes to be synchronized
//                                           belong to.
//
// Output Parameters:
// ******************
//
//  None
//
// *************************************************************************
//  This is a SampleClockSynchronizeOnlyOneBoard subfunction.
//  It updates the board ans pipes structures.
// *************************************************************************
STATIC VOID UpdateBoardAndPipeStructure(
    IN  LPSAMPLE_CLK_REQ_INFO PmLPtrReq,
    IN  WORD                  PmBoardNum )
{
    PPIPE_INFO          LcPtPipe;
    DWORDLONG           LcPipeMaskIn;
    DWORDLONG           LcPipeMaskOut;
    DWORDLONG           LcPipeMask;
    DWORD               i;
    SHORT               LcPipeIndex;

    // Retrieve pipe mask
    //
    LcPipeMaskOut = PmLPtrReq->scqOutputMask64.QuadPart ;
    LcPipeMaskIn  = PmLPtrReq->scqInputMask64.QuadPart ;

    // Update the board and pipe structures
    //
    APH_Board_Info_Array[PmBoardNum].biPipeClockMask[NO_MASK_PIPE_OUT] |= LcPipeMaskOut;
    APH_Board_Info_Array[PmBoardNum].biPipeClockMask[NO_MASK_PIPE_IN]  |= LcPipeMaskIn;

    // For all the pipes implied in the command
    //
    for ( i = 0 ; i < 2 ; i++ )
    {
        switch( i )
        {
        case 0 :
            LcPipeMask = LcPipeMaskOut;
            LcPtPipe   = &TbPipeOutInfo[0];
            break;
        case 1 :
            LcPipeMask = LcPipeMaskIn;
            LcPtPipe   = &TbPipeInInfo[0];
            break;
        }

        while ( LcPipeMask != 0 )
        {
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            ASSERT( PmBoardNum == LcPtPipe[LcPipeIndex].piNumBoard );

            LcPtPipe[LcPipeIndex].piPipeInClockMask  = LcPipeMaskIn;
            LcPtPipe[LcPipeIndex].piPipeOutClockMask = LcPipeMaskOut;
            LcPtPipe[LcPipeIndex].piClockSource      = PmLPtrReq->scqSource;
            LcPtPipe[LcPipeIndex].piClockBoardNum    = PmLPtrReq->scqClockBoardNum;
            LcPtPipe[LcPipeIndex].piFrequency        = PmLPtrReq->scqFrequency;
            LcPtPipe[LcPipeIndex].piSyncInputNum     = PmLPtrReq->scqSyncInputNum;
        } // for each pipe
    }
}


// *************************************************************************
//
// STATIC WORD SampleClockSyncOneBoardSubFct(...)
//
// Input Parameters:
// *****************
//
//  LPTASK_INFO             PmTaskInfoPtr  : the information about the task
//
// Output Parameters:
// ******************
//
//  None
//
// *************************************************************************
//  This is the SampleClockSynchronizeOnlyOneBoard subfunction.
// *************************************************************************
STATIC WORD SampleClockSyncOneBoardSubFct(  LPSAMPLE_CLK_REQ_INFO LcLPtrReq,
                                            WORD  LcBoardNum)
{
    WORD        LcRet           ;
    WORD        LcCurrentBoard  ;
    BOOLEAN     LcComputeFreq   ;
	WORD        LcPipeIndex     ;
	DWORDLONG   LcAllInPipeAudioMask;
	DWORDLONG   LcPipeMaskSRC   ;
	BYTE        LcUER_SRC       ;

	DWORDLONG	LcPipeMaskOut = LcLPtrReq->scqOutputMask64.QuadPart;
	DWORDLONG	LcPipeMaskIn  = LcLPtrReq->scqInputMask64.QuadPart;
	BYTE		LcAppIndex    = DecodeHandle( LcLPtrReq->Header.hdHandle );

	if( LcLPtrReq->scqValidationMask & (UER_SYNC_DATA | UER_ASYNC_DATA) )
	{
		if(LcLPtrReq->scqValidationMask == (UER_SYNC_DATA | UER_ASYNC_DATA))
		{
			EXIT_PCX_ERROR( ED_BAD_VALIDATION_MASK );
		}
		LcUER_SRC = (LcLPtrReq->scqValidationMask == UER_SYNC_DATA) ? IN_PROPERTIES_UER_SYNC : IN_PROPERTIES_UER_ASYNC;
	}
	else	LcUER_SRC = IN_NO_PROPERTIES;

	LcPipeIndex = 0;
	LcAllInPipeAudioMask = 0;
	LcPipeMaskSRC = LcPipeMaskIn;
	while( LcPipeMaskSRC )
	{
		if( LcPipeMaskSRC & 1 )
		{
			LcAllInPipeAudioMask |= APH_Get_Pipe_Audio_Mask(LcPipeIndex);
		}
		LcPipeMaskSRC >>= 1;
		LcPipeIndex++;
	}

    LcCurrentBoard = LcBoardNum;

	LcRet = CUR_PROTOCOL_PTR->IResource_Source_Check_Audios_New_Clock(	LcLPtrReq->scqSource,
																		LcLPtrReq->scqSyncInputNum,
																		LcLPtrReq->scqFrequency,
																		LcAllInPipeAudioMask,
																		LcUER_SRC );	// Check the audio sources with the new clock
	if( LcRet & ERROR_MASK ) EXIT_PCX_ERROR( LcRet );

    // Set the clock for the board
    // ---------------------------

    LcRet = CUR_COMMANDS_PTR->PIODefClock(
        INDEPENDANT_CLOCK,
        LcLPtrReq->scqSource,
        LcLPtrReq->scqFrequency,
        LcLPtrReq->scqSyncInputNum,
		LcLPtrReq->scqClockFormat);

    // Keeps caller imformed
    //
    if (LcRet != SUCCESS) {
        DOUT ( DBG_ERROR,("SynchronizeOnlyOneBoard : PIODefClock FAILED = %lx", LcRet));
        return LcRet;
    }

    // LXES
    if( LcLPtrReq->scqSource == CLOCK_TYPE_ETHERSOUND )
    {
        DWORD LcFrequency;
        BOOLEAN LcDummy;
        LcRet = CUR_PROTOCOL_PTR->IClock_BoardGetClockFrequency( &LcFrequency, &LcDummy);
        if(LcRet == SUCCESS)    CUR_COMMANDS_PTR->PIOSetActualClockFrequency(LcFrequency);
    }

    // Set the frequency to 0 because the frequency is unknown.
    // This correct a bug that prevents making a play while at the same
    // time, a record (with a clock equal to CLOCK_TYPE_UER_SYNCHRO) is done.
    //
    if (    ( LcLPtrReq->scqSource == CLOCK_TYPE_UER_SYNCHRO )
         || ( LcLPtrReq->scqSource == CLOCK_TYPE_WORD_CLOCK ) )
    {
        LcComputeFreq = TRUE;
    }
    else
    {
        LcComputeFreq = FALSE;
    }

    // MBR (14/08/2003) This call was in PIODefClock before,
    // but as the PIOCommands does not know the NPProtocol,
    // call it here :
    //
    CUR_PROTOCOL_PTR->IClock_BoardModifyClock(	TRUE,               // Synchronize the FIFO
												LcComputeFreq );    // Compute the frequency

    UpdateBoardAndPipeStructure(LcLPtrReq,LcBoardNum) ;

	return LcRet;
}


// *************************************************************************
//
// STATIC VOID SampleClockSynchronizeOnlyOneBoard(...)
//
// Input Parameters:
// *****************
//
//  LPSAMPLE_CLK_REQ_INFO   PmLPtrReq      : the request block
//  IN  LPRESP_HEADER_INFO  PmLPtrRepHeader: the reply block
//  WORD                    PmBoardNum     : the board number on which the
//                                           pipes to be synchronized
//                                           belong to.
//
// Output Parameters:
// ******************
//
//  None
//
// *************************************************************************
//  This is a SampleClockPipeFct subfunction.
//  It synchronizes the pipes in the case where there is only one board.
// *************************************************************************
STATIC VOID SampleClockSynchronizeOnlyOneBoard(
    IN  LPSAMPLE_CLK_REQ_INFO PmLPtrReq,
    IN  LPRESP_HEADER_INFO    PmLPtrRepHeader,
    IN  WORD                  PmBoardNum)
{
    DWORDLONG   LcPipeMaskIn;
    DWORDLONG   LcPipeMaskOut;

	// compatible clock
	//
	if (   ( APH_Commands_Array[PmBoardNum]->PIOGetActualClkType() == INDEPENDANT_CLOCK )
		&& ( APH_Commands_Array[PmBoardNum]->PIOGetActualClkSource() == PmLPtrReq->scqSource )
		&& ( APH_Commands_Array[PmBoardNum]->PIOGetActualClkInputNum() == PmLPtrReq->scqSyncInputNum ) )
	{
      if(  ( APH_Commands_Array[PmBoardNum]->PIOGetActualClockFrequency() == PmLPtrReq->scqFrequency) ||
           ( PmLPtrReq->scqSource == CLOCK_TYPE_UER_SYNCHRO ) ||
           ( PmLPtrReq->scqSource == CLOCK_TYPE_ETHERSOUND ) ||
           ( PmLPtrReq->scqSource == CLOCK_TYPE_WORD_CLOCK ) )
      {
		UpdateBoardAndPipeStructure(PmLPtrReq,PmBoardNum) ;

		PmLPtrRepHeader->rhCptr = SUCCESS;
		return;
      }
	}

    // Retrieve pipe mask
    // ******************
    LcPipeMaskOut = PmLPtrReq->scqOutputMask64.QuadPart ;
    LcPipeMaskIn  = PmLPtrReq->scqInputMask64.QuadPart ;

    // If the board has already a clock assigned to some other pipes
    // or the board is reserved
    // -----------------------------------------------------
    if (    (     (    ( APH_Board_Info_Array[PmBoardNum].biPipeClockMask[NO_MASK_PIPE_OUT] != 0 )
                    || ( APH_Board_Info_Array[PmBoardNum].biPipeClockMask[NO_MASK_PIPE_IN]  != 0 )
                  )
               && (    ( APH_Board_Info_Array[PmBoardNum].biPipeClockMask[NO_MASK_PIPE_OUT]
                              != LcPipeMaskOut)
                    || ( APH_Board_Info_Array[PmBoardNum].biPipeClockMask[NO_MASK_PIPE_IN]
                              != LcPipeMaskIn )
                  )
            )
       )
    {
        LOAD_PCX_ERROR( PmLPtrRepHeader->rhCptr, ED_SET_CLOCK_IMPOSSIBLE );
        return ;
    }

    // Else (the board has no clock or all boards pipes change the clock
    // --------------------------------------------------

    PmLPtrRepHeader->rhCptr = SampleClockSyncOneBoardSubFct( PmLPtrReq, PmBoardNum );
	return;
}


// *************************************************************************
//
// STATIC VOID SampleClockPipeFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the sample clock pipe command.
//  Update the sample clock pipe command response block.
//
// *************************************************************************
// This function supposes :
//   - There is only one ressource clock for the entire board
//   - A pipe is on only one board
//
// *************************************************************************
STATIC VOID SampleClockPipeFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    WORD                   LcRet = SUCCESS;
    LPSAMPLE_CLK_REQ_INFO  LcLPtrReq;
    LPBC_HEADER_INFO       LcLPtrReqHeader;
    LPRESP_HEADER_INFO     LcLPtrRepHeader;
    BYTE                   LcAppIndex;
    BOOLEAN                LcMoreThanOneBoard;
    BOOLEAN                LcStateChange;
    BOOLEAN                LcFrequencyChange;
    WORD                   LcABoardNum; // One of the boards on which the pipes are.

    // Pointers initialisation
    // ***********************
    LcLPtrReq       = (LPSAMPLE_CLK_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application index
    // **************************
    LcAppIndex  = DecodeHandle( LcLPtrReqHeader->hdHandle );

    if( LcAppIndex == DHSAppHandle )
    {
        LcLPtrRepHeader->rhCptr = DHSSamplingClock(  LcLPtrReq );
        return;
    }

    // Verify the mask of pipes is correct
    // ***********************************
    LcRet = SampleClockCheckPipes(
        LcLPtrReq,
        &LcMoreThanOneBoard,
        &LcStateChange,
        &LcFrequencyChange,
        &LcABoardNum );

    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // If there is nothing to do
    // *************************
    if ( ( LcStateChange == FALSE ) && ( LcFrequencyChange == FALSE ) )
    {
        return;
    }

    // If the requested clock is CLOCK_TYPE_NONE
    // **********************************
    if ( LcLPtrReq->scqSource == CLOCK_TYPE_NONE )
    {
        DWORDLONG   LcPipeMaskIn = LcLPtrReq->scqInputMask64.QuadPart;
        DWORDLONG   LcPipeMaskOut = LcLPtrReq->scqOutputMask64.QuadPart;

        LcRet = APHSampleClockSetNoClock( LcAppIndex,LcPipeMaskIn,LcPipeMaskOut );

        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // ## FS (12/09/97) -- FA #55: Let's control the board number
    //  it should be between 0 and MAX_BOARD - 1 and the board must
    // be known to the driver.
    // ************************************************************
    if (    ( LcLPtrReq->scqClockBoardNum >= MAX_BOARD )
         || ( APH_Board_Info_Array[LcLPtrReq->scqClockBoardNum].biCarac != CARAC_USED ) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_SET_CLOCK_REFUSED );
        return;
    }

    if ( LcMoreThanOneBoard )
    {
        // The pipes belong to more than one board
        // ***************************************
            LcLPtrRepHeader->rhCptr = ED_SET_CLOCK_REFUSED;
            return;
    }
    else
    {
        // The pipes belong to one board
        // *****************************
        SampleClockSynchronizeOnlyOneBoard(
            LcLPtrReq,
            LcLPtrRepHeader,
            LcABoardNum);
    }
}


// *************************************************************************
//
// WORD APHSampleClockSynchronizeOnlyOneBoard(...)
//
// Input Parameters:
// *****************
//
//  LPSAMPLE_CLK_REQ_INFO   PmLPtrReq      : the request block
//  WORD                    PmBoardNum     : the board number on which the
//                                           pipes to be synchronized
//                                           belong to.
//
// Output Parameters:
// ******************
//
//  None
//
// *************************************************************************
//  This is a SampleClockPipeFct subfunction.
//  It synchronizes the pipes in the case where there is only one board.
// *************************************************************************
WORD APHSampleClockSynchronizeOnlyOneBoard(
    IN  LPSAMPLE_CLK_REQ_INFO PmLPtrReq,
    IN  WORD                  PmBoardNum
)
{
    LPRESP_HEADER_INFO PmLPtrRepHeader = (LPRESP_HEADER_INFO) PmLPtrReq;

    SampleClockSynchronizeOnlyOneBoard( PmLPtrReq, PmLPtrRepHeader, PmBoardNum );

    return( PmLPtrRepHeader->rhCptr );
}

