// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************
// **************************************************************************
//
//  DigigramHardwareSettings implementation
//
// **************************************************************************

#if( MAX_DRV_INPIPE > MAX_DRV_OUTPIPE )
#define MAX_DHS_PIPE    MAX_DRV_INPIPE
#else
#define MAX_DHS_PIPE    MAX_DRV_OUTPIPE
#endif

#define LEVEL_DRV_VALUE_0_DB    0x1b7

typedef struct _DHS_PIPE_INFO {
    BOOLEAN Allocated;      // this audio is allocated for DHS
    BYTE    BoardNum;
    WORD    AudioIdx;       // 0-based DSP audio index [0 to 7 on 882]
    BYTE    PipeIdx;        // 0-based index of the DHS Pipe Info Structure
    BYTE    AppAudioNum;    // 0-based Application Audio Index (0 (mono or left), 1 (stereo right), ...)
    BYTE    unused[2];
    DWORDLONG   AudioMask;
} DHS_PIPE_INFO, *PDHS_PIPE_INFO;


typedef struct _DHS_AUDIO_INFO {
    BOOLEAN IsControlled;   // audio parameters controlled by DHS
    BYTE    PipeInfoIdx;    // index in DHS_PIPE_INFO array
    BYTE    unused[2];

    union   _INOUT_LEVEL_VALUES_UNION {
        struct _OUT_LEVEL_VALUES_INFO {
            WORD        olviDigitalLevel;
            WORD        olviMonitorLevel;
            WORD        olviMuteLevel;
            WORD        olviM1Level;
            WORD        olviM2Level;
            WORD        olviMonitorSource;
        } sOutLevelValues;
        #define solviDigitalLevel u.sOutLevelValues.olviDigitalLevel
        #define solviMonitorLevel u.sOutLevelValues.olviMonitorLevel
        #define solviMuteLevel u.sOutLevelValues.olviMuteLevel
        #define solviM1Level u.sOutLevelValues.olviM1Level
        #define solviM2Level u.sOutLevelValues.olviM2Level
        #define solviMonitorSource u.sOutLevelValues.olviMonitorSource

        struct _IN_LEVEL_VALUES_INFO {
            WORD        ilviDigitalLevel;
        } sInLevelValues;
        #define silviDigitalLevel u.sInLevelValues.ilviDigitalLevel
    } u;
} DHS_AUDIO_INFO, *PDHS_AUDIO_INFO;

typedef struct _DHS_BOARD_CLOCK {
    BOOLEAN Locked;     // this board's clock is controlled by DHS
    BOOLEAN FrequFixed; // this board's clock is controlled by DHS and the internal Frequency is fixed
    BYTE    unused[2];
} DHS_BOARD_CLOCK, *PDHS_BOARD_CLOCK;

STATIC  DHS_PIPE_INFO   TbDhsInPipeInfo[MAX_DHS_PIPE];
STATIC  DHS_PIPE_INFO   TbDhsOutPipeInfo[MAX_DHS_PIPE];

STATIC  DHS_AUDIO_INFO  TbDhsInAudioInfo[MAX_BOARD][MAX_BOARD_AUDIO];
STATIC  DHS_AUDIO_INFO  TbDhsOutAudioInfo[MAX_BOARD][MAX_BOARD_AUDIO];

STATIC  DHS_BOARD_CLOCK TbDhsBoardClock[MAX_BOARD];

STATIC  APP_HANDLE      DHSAppHandle;


STATIC VOID DHSInit(BOOLEAN PmUnregister)
{
    DHSAppHandle = MAX_PCX_HANDLE;

    if(PmUnregister)
    {
        DOUT(DBG_PRINT, ("DHS app unregister TODO - apply default handling !\n"));
    }

    BZERO2( TbDhsInPipeInfo,    TbDhsInPipeInfo,    1 );
    BZERO2( TbDhsOutPipeInfo,   TbDhsOutPipeInfo,   1 );
    BZERO2( TbDhsInAudioInfo,   TbDhsInAudioInfo,   1 );
    BZERO2( TbDhsOutAudioInfo,  TbDhsOutAudioInfo,  1 );
    BZERO2( TbDhsBoardClock,    TbDhsBoardClock,    1 );
}

// *************************************************************************
//
// *************************************************************************
STATIC WORD ChkPmManyDHSPipeMask(
    IN  DWORDLONG   PmPipeInMask,
    IN  DWORDLONG   PmPipeOutMask  )
{
    WORD        LcPipeIndex ;
    DWORDLONG   LcPipeMask ;

    // verify this mask specify at least one pipe ...
    // **********************************************
    if ( (PmPipeInMask == 0) && (PmPipeOutMask == 0) )
    {
        EXIT_PCX_ERROR( ED_INVALID_PIPE );
    }

    // Check all pipes in the mask, one by one
    // ***************************************
    LcPipeMask = PmPipeInMask ;
    while ( LcPipeMask != 0 )
    {
        LcPipeIndex = UTI64Mask2Word( LcPipeMask ) ;

        // Remove this pipe from the mask
        // ******************************
        LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

        // does the pipe actually belong to the application ?
        // **************************************************
        if (   ( LcPipeIndex >= MAX_DHS_PIPE )
            || ( TbDhsInPipeInfo[LcPipeIndex].Allocated == FALSE )
            || ( TbDhsInPipeInfo[LcPipeIndex].PipeIdx != LcPipeIndex ) )
        {
            EXIT_PCX_ERROR( ED_INVALID_PIPE );
        }
    }
    LcPipeMask = PmPipeOutMask ;
    while ( LcPipeMask != 0 )
    {
        LcPipeIndex = UTI64Mask2Word( LcPipeMask ) ;

        // Remove this pipe from the mask
        // ******************************
        LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

        // does the pipe actually belongs to the application ?
        // ***************************************************
        if (   ( LcPipeIndex >= MAX_DHS_PIPE )
            || ( TbDhsOutPipeInfo[LcPipeIndex].Allocated == FALSE )
            || ( TbDhsOutPipeInfo[LcPipeIndex].PipeIdx != LcPipeIndex ) )
        {
            EXIT_PCX_ERROR( ED_INVALID_PIPE );
        }
    }

    // Parameter is correct
    // ********************
    return SUCCESS ;
}

// *************************************************************************
//
// *************************************************************************
STATIC VOID DHSPipeRemove( LPPIPE_REM_REQ_INFO PmPtrReq, LPRESP_HEADER_INFO PmPtrRepHeader)
{
    BYTE    LcAppIndex = PmPtrReq->Header.hdHandle -1;
    DWORDLONG LcPipeMask;
    DWORDLONG LcPipeInMask = PmPtrReq->prqInMask64.QuadPart ;
    DWORDLONG LcPipeOutMask = PmPtrReq->prqOutMask64.QuadPart ;
    WORD    LcRet;
    int     i, j, LcPipeIndex;

    LcRet = ChkPmManyDHSPipeMask( LcPipeInMask, LcPipeOutMask );
    if(LcRet != SUCCESS)
    {
        PmPtrRepHeader->rhCptr = LcRet;
        return;
    }

    LcPipeMask = LcPipeOutMask;
    while ( LcPipeMask )
    {
        LcPipeIndex = UTI64Mask2Word( LcPipeMask );
        LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

        // all the audios belonging to the pipe
        for (j=0; j<MAX_DHS_PIPE; j++)
        {
            if ( TbDhsOutPipeInfo[j].Allocated &&
                (TbDhsOutPipeInfo[j].PipeIdx == LcPipeIndex) )
            {
                PDHS_AUDIO_INFO LcDhsOutAudioInfo;

                DOUT(DBG_PRINT, ("DHSPipeRemove OUT Pipe(%d) Audio(%d) PipeInfoIdx(%d)\n", LcPipeIndex, TbDhsOutPipeInfo[j].AudioIdx, j));

                LcDhsOutAudioInfo = &TbDhsOutAudioInfo[TbDhsOutPipeInfo[j].BoardNum][TbDhsOutPipeInfo[j].AudioIdx];

                // free the DHS pipe entry
                BZERO2( &TbDhsOutPipeInfo[j], DHS_PIPE_INFO, 1 );

                ASSERT(LcDhsOutAudioInfo->PipeInfoIdx == j);

                if(LcDhsOutAudioInfo->IsControlled)
                {
                    ;
                    // TODO
                    //PipeResetLevelsSubFct(  LcPtPipe[LcPipeIndex].piResetAnaLevelsAction,
                    //                        LcPipeIndex,
                    //                        LcIsCapture );
                }

                // free the DHS audio entry
                BZERO2( LcDhsOutAudioInfo, *LcDhsOutAudioInfo, 1 );
            }
        }
    } // for all out masks

    LcPipeMask = LcPipeInMask;
    while ( LcPipeMask )
    {
        LcPipeIndex = UTI64Mask2Word( LcPipeMask );
        LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

        // all the audios belonging to the pipe
        for (j=0; j<MAX_DHS_PIPE; j++)
        {
            if ( TbDhsInPipeInfo[j].Allocated &&
                (TbDhsInPipeInfo[j].PipeIdx == LcPipeIndex) )
            {
                WORD LcCurrentAudio = TbDhsInPipeInfo[j].AudioIdx;
                BYTE LcCurrentBoard = TbDhsInPipeInfo[j].BoardNum;

                PDHS_AUDIO_INFO LcDhsInAudioInfo;

                DOUT(DBG_PRINT, ("DHSPipeRemove IN Pipe(%d) Audio(%d) PipeInfoIdx(%d)\n", LcPipeIndex, LcCurrentAudio, j));

                LcDhsInAudioInfo = &TbDhsInAudioInfo[LcCurrentBoard][LcCurrentAudio];

                // free the DHS pipe entry
                BZERO2( &TbDhsInPipeInfo[j], DHS_PIPE_INFO, 1 );

                ASSERT(LcDhsInAudioInfo->PipeInfoIdx == j);

                if(LcDhsInAudioInfo->IsControlled)
                {
                    PVOIE_INFO  LcPtVoie;

                    // If there is no real pipe using this audio (AUDIO_VIRTUAL is not handled)
                    // release the audios
                    //
                    LcRet = LookupBoardAudio(LcCurrentBoard, 0, LcCurrentAudio, AUDIO_PHYSICAL | AUDIO_IN, &LcPtVoie);
                    if( (LcRet == SUCCESS) && (LcPtVoie == NULL) )
                    {
                        DOUT(DBG_PRINT, ("DHSPipeRemove IN Pipe(%d) Release Audio(%d)\n", LcPipeIndex, LcCurrentAudio));

                        CUR_PROTOCOL_PTR->IResource_Source_Release_Audios( UTIMask64bit( LcCurrentAudio ) );
                    }

                    // TODO: necessary ?
                    //PipeResetLevelsSubFct(  LcPtPipe[LcPipeIndex].piResetAnaLevelsAction,
                    //                        LcPipeIndex,
                    //                        LcIsCapture );
                }

                // free the DHS audio entry
                BZERO2( LcDhsInAudioInfo, *LcDhsInAudioInfo, 1 );
            }
        }
    } // for all in masks
}


// **************************************************************************
// 
// **************************************************************************
STATIC VOID DHSPipeDef(LPPIPE_DEF_REQ_INFO PmRequest, LPPIPE_DEF_RESP_INFO PmResp)
{
    LPPIPE_STREAM_INFO  LcPtrStream;
    LPPIPE_AUDIO_INFO   LcPtrAudio;
    PDHS_PIPE_INFO      LcPtrDhsPipeInfo;
    PDSP_INFO           LcPtrDsp;
    WORD                LcOper;
    WORD                LcAppIndex = PmRequest->Header.hdHandle - 1;
    BYTE                LcCurrentBoard, i, j;
    BYTE                LcFirstAudio = 0;
    BOOLEAN             LcFirstDone;
    DWORDLONG           LcAudioMask;

    LcOper = PmRequest->pdqAttributes & PIPE_PLAYREC_MASK;

    // no buffers, no streams
    if( (PmRequest->pdqBuffersNum != 0) || (PmRequest->pdqMaxStream != 0) )
    {
        LOAD_PCX_ERROR( PmResp->RespHeader.rhCptr, ED_INCOMPATIBLE_PIPE_AUDIO_ATTRIB );
        return;
    }
    LcPtrStream = (LPPIPE_STREAM_INFO)(PmRequest + 1);
    LcPtrAudio = (LPPIPE_AUDIO_INFO)(LcPtrStream);  // there is always a unused structure in PipeDefFct !

    LcAudioMask = 0;

    // For each audio requested do some tests
    // --------------------------------------
    for( i = 0 ; i < PmRequest->pdqAudioNum ; i++ )
    {
        DWORD     LcDspManagedAudios;

        LcCurrentBoard = LcPtrAudio[i].paBoard ;

        if( (LcCurrentBoard >= MAX_BOARD) || (APH_Board_Info_Array[LcCurrentBoard].biCarac != CARAC_USED) )
        {
            LOAD_PCX_ERROR( PmResp->RespHeader.rhCptr, ED_INVALID_BOARD );
            return ;
        }
        if( (LcPtrAudio[i].paAudioAttributes & AUDIO_PHYSICAL) == 0 )
        {
            LOAD_PCX_ERROR( PmResp->RespHeader.rhCptr, ED_INCOMPATIBLE_PIPE_AUDIO_ATTRIB );
            return ;
        }

        LcPtrDsp = &(APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0]);

        // dsp OK ? (DHS is not registered at DSP)
        if( (LcPtrDsp->dsEtat < DSP_FIRST_SOFT) ||
            (LcPtrDsp->dsEtat > DSP_LAST_SOFT))
        {
            LOAD_PCX_ERROR( (PmResp->RespHeader).rhCptr, ED_INVALID_DSP );
            return;
        }

        if ( LcOper == OPER_PLAY )
        {
            LcDspManagedAudios = LcPtrDsp->dsManagedPhysicalOutputNumber;
            LcPtrDhsPipeInfo = TbDhsOutPipeInfo;
        }
        else
        {
            LcDspManagedAudios = LcPtrDsp->dsManagedPhysicalInputNumber;
            LcPtrDhsPipeInfo = TbDhsInPipeInfo;
        }

        // audio exists ?
        if( LcPtrAudio[i].paAudioNum >= LcDspManagedAudios )
        {
            LOAD_PCX_ERROR( (PmResp->RespHeader).rhCptr, ED_INVALID_PIPE_AUDIO );
            return;
        }

        for (j=0; j<MAX_DHS_PIPE; j++)
        {
            if ( LcPtrDhsPipeInfo[j].Allocated &&
                (LcPtrDhsPipeInfo[j].BoardNum == LcCurrentBoard) &&
                (LcPtrDhsPipeInfo[j].AudioIdx == LcPtrAudio[i].paAudioNum) )
            {
                // the audio is already allocated by DHS
                LOAD_PCX_ERROR( (PmResp->RespHeader).rhCptr, ED_INVALID_PIPE_AUDIO );
                return;
            }
        }

        // the boards physical audio mask
        LcAudioMask |= UTIMask64bit( LcPtrAudio[i].paAudioNum );
    }
    // tests OK, now reserve it
    LcFirstDone = FALSE;
    for( i = 0 ; i < PmRequest->pdqAudioNum ; i++ )
    {
        LcCurrentBoard = LcPtrAudio[i].paBoard ;
        if ( LcOper == OPER_PLAY )
        {
            LcPtrDhsPipeInfo = TbDhsOutPipeInfo;
        }
        else
        {
            LcPtrDhsPipeInfo = TbDhsInPipeInfo;
        }

        for (j=0; j<MAX_DHS_PIPE; j++)
        {
            BYTE k = j; // MAX_DHS_PIPE-j-1; // use other index than normal pipes (for debug issues)

            if(LcPtrDhsPipeInfo[k].Allocated == FALSE)
            {
                BOOLEAN LcControlAudioDHS = ((LcPtrAudio[i].paAudioAttributes & AUDIO_DHS_CONTROL_MASK) != 0);
                // If 1st audio allocated, let's remember it and assign the future pipe number
                if ( !LcFirstDone )
                {
                    LcFirstDone = TRUE;
                    LcFirstAudio = k;
                }
                LcPtrDhsPipeInfo[k].Allocated = TRUE;
                LcPtrDhsPipeInfo[k].BoardNum = LcCurrentBoard;
                LcPtrDhsPipeInfo[k].AudioIdx = LcPtrAudio[i].paAudioNum;
                LcPtrDhsPipeInfo[k].PipeIdx = LcFirstAudio;
                LcPtrDhsPipeInfo[k].AppAudioNum = i;
                LcPtrDhsPipeInfo[k].AudioMask = LcAudioMask;

                DOUT(DBG_PRINT, ("DHSPipeDef Pipe(%d) Audio(%d) AudioInfoIdx(%d) handled(%d)\n", LcFirstAudio, LcPtrAudio[i].paAudioNum, k, LcControlAudioDHS));

                if( LcOper == OPER_PLAY )
                {
                    PDHS_AUDIO_INFO LcDhsOutAudioInfo = &TbDhsOutAudioInfo[LcCurrentBoard][LcPtrAudio[i].paAudioNum];
                    LcDhsOutAudioInfo->IsControlled = LcControlAudioDHS;
                    LcDhsOutAudioInfo->PipeInfoIdx = k;
                    // init the default volumes like default pipe settings
                    LcDhsOutAudioInfo->solviDigitalLevel = LEVEL_DRV_VALUE_0_DB;
                    LcDhsOutAudioInfo->solviMonitorLevel = LEVEL_DRV_VALUE_0_DB;
                    LcDhsOutAudioInfo->solviM1Level = MUTE_ON;
                    // monitoring default : use input audio face to the output
                    LcDhsOutAudioInfo->solviMonitorSource = LcPtrAudio[i].paAudioNum;
                }
                else
                {
                    PDHS_AUDIO_INFO LcDhsInAudioInfo = &TbDhsInAudioInfo[LcCurrentBoard][LcPtrAudio[i].paAudioNum];
                    LcDhsInAudioInfo->IsControlled = LcControlAudioDHS;
                    LcDhsInAudioInfo->PipeInfoIdx = k;
                    LcDhsInAudioInfo->silviDigitalLevel = LEVEL_DRV_VALUE_0_DB;
                }
                break; // j
            }
        }
    }

    // Fill Response bloc: Pipe Mask
    // *****************************
    PmResp->pdpPipeMask64.QuadPart = UTIMask64bit( LcFirstAudio );
}


// **************************************************************************
// 
// **************************************************************************
STATIC WORD DHSSetOutAudioLevels(LPOUT_AUDIO_SET_LEVELS_REQ_INFO PmPtrReq)
{
    WORD        LcPipeIndex;
    DWORDLONG   LcPipeOutMask = PmPtrReq->oslqPipeMask64.QuadPart;
    WORD    LcRet = SUCCESS;
    int     i;

    // do some tests : no differed command and no zero audio mask
    if(PmPtrReq->oslqDiffered  != 0)    EXIT_PCX_ERROR(ED_DIFFERED_CMD_REFUSED);
    if(PmPtrReq->oslqAudioMask == 0)    EXIT_PCX_ERROR(ED_INVALID_BOARD_AUDIO);
    // test mask of pipes
    LcRet = ChkPmManyDHSPipeMask( 0, LcPipeOutMask );
    if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

	// For all the pipes
    // -----------------
    while( LcPipeOutMask )
    {
        LcPipeIndex = UTI64Mask2Word( LcPipeOutMask );
        LcPipeOutMask &= ~UTIMask64bit( LcPipeIndex );

        for ( i=0; i<MAX_DHS_PIPE; i++ )
        {
            if( (TbDhsOutPipeInfo[i].Allocated != FALSE) &&
                (TbDhsOutPipeInfo[i].PipeIdx == LcPipeIndex) &&
                (UTIMaskWord( TbDhsOutPipeInfo[i].AppAudioNum ) & PmPtrReq->oslqAudioMask) )
            {
                BYTE LcCurrentBoard = TbDhsOutPipeInfo[i].BoardNum;
                WORD LcCurrentAudio = TbDhsOutPipeInfo[i].AudioIdx;

                if( TbDhsOutAudioInfo[LcCurrentBoard][LcCurrentAudio].IsControlled )
                {
                    POUT_AUDIO_SET_LEVEL_REQ_INFO LcLevelInfo = &PmPtrReq->oslqLevelInfo;

                    // if analog level
                    if( LcLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_ANALOG_MASK )
                    {
                        LcRet = CUR_COMMANDS_PTR->PIOSetAudioAnalogLevel(   LcCurrentAudio,
                                                                            OPER_PLAY,
                                                                            LcLevelInfo->oasliAnalogLevel );

                        DOUT(DBG_PRINT, ("DHSSetOutAudioLevels Pipe(%d) Audio(%d) analog Level(%x)\n", LcPipeIndex, LcCurrentAudio, LcLevelInfo->oasliAnalogLevel));

                        if( LcRet != SUCCESS ) EXIT_PCX_ERROR(LcRet);
                    }
                    if( LcLevelInfo->oasliPar1 & OUT_AUDIO_SET_NUM_LEVEL )
                    {
                        LEVEL_AUDIO_INFO    LcLevelAudioInf;
                        PVOIE_INFO          LcPtVoie = NULL;
                        PDHS_AUDIO_INFO     LcDhsOutAudioInfo = &TbDhsOutAudioInfo[LcCurrentBoard][LcCurrentAudio];

                        if ( LcLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_DIGITAL_MASK )
                        {
                            LcLevelAudioInf.gaiHasDigitalLevel = TRUE;
                            LcLevelAudioInf.gaiDigitalLevel = LcLevelInfo->oasliDigitalLevel;
                            LcDhsOutAudioInfo->solviDigitalLevel = LcLevelInfo->oasliDigitalLevel;
                        }
                        else LcLevelAudioInf.gaiHasDigitalLevel = FALSE;

                        if ( LcLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_MONITOR_MASK )
                        {
                            LcLevelAudioInf.gaiHasMonitorLevel = TRUE;
                            LcLevelAudioInf.gaiMonitorLevel = LcLevelInfo->oasliMonitorLevel;
                            LcDhsOutAudioInfo->solviMonitorLevel = LcLevelInfo->oasliMonitorLevel;
                        }
                        else LcLevelAudioInf.gaiHasMonitorLevel = FALSE;

                        if ( LcLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_MUTE_MASK )
                        {
                            LcLevelAudioInf.gaiHasMuteLevel = TRUE;
                            LcLevelAudioInf.gaiMuteLevel = LcLevelInfo->oasliMute;
                            LcDhsOutAudioInfo->solviMuteLevel = LcLevelInfo->oasliMute;
                        }
                        else LcLevelAudioInf.gaiHasMuteLevel = FALSE;

                        if ( LcLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_MUTE_M1_MASK )
                        {
                            LcLevelAudioInf.gaiHasM1Level = TRUE;
                            LcLevelAudioInf.gaiM1Level = LcLevelInfo->oasliMonitorMute1;
                            LcDhsOutAudioInfo->solviM1Level = LcLevelInfo->oasliMonitorMute1;
                        }
                        else LcLevelAudioInf.gaiHasM1Level = FALSE;

                        if ( LcLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_MUTE_M2_MASK )
                        {
                            LcLevelAudioInf.gaiHasM2Level = TRUE;
                            LcLevelAudioInf.gaiM2Level = LcLevelInfo->oasliMonitorMute2;
                            LcDhsOutAudioInfo->solviM2Level = LcLevelInfo->oasliMonitorMute2;
                        }
                        else LcLevelAudioInf.gaiHasM2Level = FALSE;

                        // Is there a real pipe ? (AUDIO_VIRTUAL is not handled)
                        //
                        LcRet = LookupBoardAudio(LcCurrentBoard, 0, LcCurrentAudio, AUDIO_PHYSICAL | AUDIO_OUT, &LcPtVoie);
                        if( (LcRet == SUCCESS) && (LcPtVoie != NULL) )
                        {
                            TARGET_INFO    LcTarget;
                            BZERO2( &LcTarget, LcTarget, 1 );
                            LcTarget.tgCaracPipeVoie = OPER_PLAY;

                            // adjust the audio digital volumes (TODO : here only mono, could be more channels if same real pipe)
                            //
                            LcRet = CUR_PROTOCOL_PTR->ILevel_AudioSetDigitalLevel(	&LcTarget,
																	                UTIMask64bit( LcCurrentAudio ),
																	                &LcLevelAudioInf );

                            DOUT(DBG_PRINT, ("DHSSetOutAudioLevels Pipe(%d) Audio(%d) digital Level(%x)\n", LcPipeIndex, LcCurrentAudio, LcLevelInfo->oasliDigitalLevel));

                            if( LcRet != SUCCESS ) EXIT_PCX_ERROR(LcRet);
                        }
                    }
                }
                else
                {
                    // is not DHS controlled !
                    EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );
                }
            }
        }
    }
    return LcRet;
}


// **************************************************************************
// 
// **************************************************************************
STATIC WORD DHSGetOutAudioLevels(
    IN  DWORDLONG                           PmPipeMask,
    IN  WORD                                PmAudioMask,
    OUT LPOUT_AUDIO_GET_LEVELS_RESP_INFO    PmLPtrAudio,
    OUT PWORD                               PmNbAudioPtr )
{
    WORD    LcOutPipeIndex ;
    WORD    LcRet;
    int     i;

    *PmNbAudioPtr = 0;

    // verify this mask specify one pipe ...
    // **********************************************
    LcRet = ChkPmManyDHSPipeMask( 0, PmPipeMask );
    if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

    LcOutPipeIndex = UTI64Mask2Word( PmPipeMask ) ;

    // ... and only one pipe
    // **********************************************
    if ( PmPipeMask & ~UTIMask64bit( LcOutPipeIndex ) ) EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );

    for ( i=0; i<MAX_DHS_PIPE; i++ )
    {
        if(PmAudioMask == 0)    break;  // for

        if( (TbDhsOutPipeInfo[i].Allocated) &&
            (TbDhsOutPipeInfo[i].PipeIdx == LcOutPipeIndex) )
        {
            BYTE LcAppAudioNum = TbDhsOutPipeInfo[i].AppAudioNum;
            WORD LcOneAudioMask = UTIMaskWord( LcAppAudioNum );

            if( LcOneAudioMask & PmAudioMask )
            {
                WORD LcLevel;
                BYTE LcCurrentBoard = TbDhsOutPipeInfo[i].BoardNum;
                WORD LcAudioIdx = TbDhsOutPipeInfo[i].AudioIdx;
                PVOIE_INFO LcPtVoie = NULL;

                PDHS_AUDIO_INFO LcDhsOutAudioInfo = &TbDhsOutAudioInfo[LcCurrentBoard][LcAudioIdx];
                ASSERT(LcDhsOutAudioInfo->PipeInfoIdx == i); // check coherence

                PmAudioMask &= ~LcOneAudioMask;

                // Get analog level
                LcRet = CUR_COMMANDS_PTR->PIOGetAudioAnalogLevel(   LcAudioIdx,
                                                                    OPER_PLAY,
                                                                    &LcLevel );
                if(LcRet & ERROR_MASK)  EXIT_PCX_ERROR(LcRet);

                DOUT(DBG_PRINT, ("DHSGetOutAudioLevels Pipe(%d) Audio(%d) Level(%x)\n", LcOutPipeIndex, LcAudioIdx, LcLevel));

                if(LcRet == SUCCESS)    PmLPtrAudio->olpAnalogLevel = LcLevel;
                else                    PmLPtrAudio->olpAnalogLevel = LEVEL_DRV_VALUE_0_DB;

                // Digital Levels : Is there a real pipe ? AUDIO_VIRTUAL is not handled
                // TODO : is this really necessary ???
                LcRet = LookupBoardAudio(LcCurrentBoard, 0, LcAudioIdx, AUDIO_PHYSICAL | AUDIO_OUT, &LcPtVoie);
                if( (LcRet == SUCCESS) && (LcPtVoie != NULL) )
                {
                    LEVEL_AUDIO_INFO LcLevelAudioInf;

                    // TODO : get info for only one audio. could be more if same pipe
                    //
                    LcRet = CUR_PROTOCOL_PTR->ILevel_AudioGetLevels(	FALSE,
														                UTIMask64bit( LcAudioIdx ),
														                1,
														                &LcLevelAudioInf );
                    if(LcRet & ERROR_MASK)  EXIT_PCX_ERROR(LcRet);

                    PmLPtrAudio->olpDigitalLevel      = LcLevelAudioInf.gaiDigitalLevel;
                    PmLPtrAudio->olpMonitoringLevel   = LcLevelAudioInf.gaiMonitorLevel;
                    PmLPtrAudio->olpMute              = LcLevelAudioInf.gaiMuteLevel;
                    PmLPtrAudio->olpMonitoringMute1   = LcLevelAudioInf.gaiM1Level;
                    PmLPtrAudio->olpMonitoringMute2   = LcLevelAudioInf.gaiM2Level;
                }
                else
                {
                    if( LcDhsOutAudioInfo->IsControlled )
                    {
                        // if this audio is controlled by DHS, values are memorized
                        // Get digital level, monitoring level, mute and the monitoring mutes
                        PmLPtrAudio->olpDigitalLevel    = LcDhsOutAudioInfo->solviDigitalLevel;
                        PmLPtrAudio->olpMonitoringLevel = LcDhsOutAudioInfo->solviMonitorLevel;
                        PmLPtrAudio->olpMute            = LcDhsOutAudioInfo->solviMuteLevel;
                        PmLPtrAudio->olpMonitoringMute1 = LcDhsOutAudioInfo->solviM1Level;
                        PmLPtrAudio->olpMonitoringMute2 = LcDhsOutAudioInfo->solviM2Level;
                    }
                    else
                    {
                        PmLPtrAudio->olpDigitalLevel    = LEVEL_DRV_VALUE_0_DB;
                        PmLPtrAudio->olpMonitoringLevel = LEVEL_DRV_VALUE_0_DB;
                        PmLPtrAudio->olpMonitoringMute1 = MUTE_ON;
                    }
                }
                PmLPtrAudio++;
                (*PmNbAudioPtr)++;
            }
        }
    }

    return SUCCESS;
}


// **************************************************************************
// 
// **************************************************************************
STATIC WORD DHSSetInAudioLevels(LPIN_AUDIO_SET_LEVELS_REQ_INFO PmPtrReq)
{
    WORD        LcPipeIndex;
    DWORDLONG   LcPipeInMask = PmPtrReq->islqPipeMask64.QuadPart;
    WORD    LcRet = SUCCESS;
    int     i;

    // do some tests : no differed command and no zero audio mask
    if(PmPtrReq->islqDiffered  != 0)    EXIT_PCX_ERROR(ED_DIFFERED_CMD_REFUSED);
    if(PmPtrReq->islqAudioMask == 0)    EXIT_PCX_ERROR(ED_INVALID_BOARD_AUDIO);
    // test mask of pipes
    LcRet = ChkPmManyDHSPipeMask( LcPipeInMask, 0 );
    if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

	// For all the pipes
    // -----------------
    while( LcPipeInMask )
    {
        LcPipeIndex = UTI64Mask2Word( LcPipeInMask );
        LcPipeInMask &= ~UTIMask64bit( LcPipeIndex );

        for ( i=0; i<MAX_DHS_PIPE; i++ )
        {
            if( (TbDhsInPipeInfo[i].Allocated != FALSE) &&
                (TbDhsInPipeInfo[i].PipeIdx == LcPipeIndex) &&
                (UTIMaskWord( TbDhsInPipeInfo[i].AppAudioNum ) & PmPtrReq->islqAudioMask) )
            {
                BYTE LcCurrentBoard = TbDhsInPipeInfo[i].BoardNum;
                WORD LcCurrentAudio = TbDhsInPipeInfo[i].AudioIdx;

                if( TbDhsInAudioInfo[LcCurrentBoard][LcCurrentAudio].IsControlled )
                {
                    PIN_AUDIO_SET_LEVEL_REQ_INFO LcLevelInfo = &PmPtrReq->islqLevelInfo;

                    // if analog level
                    if( LcLevelInfo->iasliPar1 & ~IN_AUDIO_SET_LEVEL_DIGITAL_MASK )
                    {
                        LcRet = CUR_COMMANDS_PTR->PIOSetInAudioLevels(  LcCurrentAudio, LcLevelInfo );

                        DOUT(DBG_PRINT, ("DHSSetInAudioLevels Pipe(%d) Audio(%d) analog Level(%x)\n", LcPipeIndex, LcCurrentAudio, LcLevelInfo->iasliAnalogLevel));

                        if( LcRet != SUCCESS ) EXIT_PCX_ERROR(LcRet);
                    }
                    if( LcLevelInfo->iasliPar1 & IN_AUDIO_SET_LEVEL_DIGITAL_MASK )
                    {
                        PVOIE_INFO          LcPtVoie = NULL;
                        PDHS_AUDIO_INFO     LcDhsInAudioInfo = &TbDhsInAudioInfo[LcCurrentBoard][LcCurrentAudio];

                        LcDhsInAudioInfo->silviDigitalLevel = LcLevelInfo->iasliDigitalLevel;

                        // Is there a real pipe ? (AUDIO_VIRTUAL is not handled)
                        //
                        LcRet = LookupBoardAudio(LcCurrentBoard, 0, LcCurrentAudio, AUDIO_PHYSICAL | AUDIO_IN, &LcPtVoie);
                        if( (LcRet == SUCCESS) && (LcPtVoie != NULL) )
                        {
                            LEVEL_AUDIO_INFO    LcLevelAudioInf;
                            TARGET_INFO         LcTarget;

                            BZERO2( &LcTarget, LcTarget, 1 );
                            LcTarget.tgCaracPipeVoie = OPER_REC;

                            BZERO2(&LcLevelAudioInf, LcLevelAudioInf, 1);
                            LcLevelAudioInf.gaiHasDigitalLevel = TRUE;
                            LcLevelAudioInf.gaiDigitalLevel = LcLevelInfo->iasliDigitalLevel;

                            // adjust the audio digital volumes (TODO : here only mono, could be more channels if same real pipe)
                            //
                            LcRet = CUR_PROTOCOL_PTR->ILevel_AudioSetDigitalLevel(	&LcTarget,
																	                UTIMask64bit( LcCurrentAudio ),
																	                &LcLevelAudioInf );

                            DOUT(DBG_PRINT, ("DHSSetInAudioLevels Pipe(%d) Audio(%d) digital Level(%x)\n", LcPipeIndex, LcCurrentAudio, LcLevelInfo->iasliDigitalLevel));

                            if( LcRet != SUCCESS ) EXIT_PCX_ERROR(LcRet);
                        }
                    }
                }
                else
                {
                    // is not DHS controlled !
                    EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );
                }
            }
        }
    }
    return LcRet;
}


// **************************************************************************
// 
// **************************************************************************
STATIC WORD DHSGetInAudioLevels(
    IN  DWORDLONG                           PmPipeMask,
    IN  WORD                                PmAudioMask,
    OUT LPIN_AUDIO_GET_LEVELS_RESP_INFO     PmLPtrAudio,
    OUT PWORD                               PmNbAudioPtr )
{
    WORD    LcInPipeIndex ;
    WORD    LcRet;
    int     i;

    *PmNbAudioPtr = 0;

    // verify this mask specify one pipe ...
    // **********************************************
    LcRet = ChkPmManyDHSPipeMask( PmPipeMask, 0 );
    if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

    LcInPipeIndex = UTI64Mask2Word( PmPipeMask ) ;

    // ... and only one pipe
    // **********************************************
    if ( PmPipeMask & ~UTIMask64bit( LcInPipeIndex ) ) EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );


    for ( i=0; i<MAX_DHS_PIPE; i++ )
    {
        if(PmAudioMask == 0)    break;  // for

        if( (TbDhsInPipeInfo[i].Allocated) &&
            (TbDhsInPipeInfo[i].PipeIdx == LcInPipeIndex) )
        {
            BYTE LcAppAudioNum = TbDhsInPipeInfo[i].AppAudioNum;
            WORD LcOneAudioMask = UTIMaskWord( LcAppAudioNum );

            if( LcOneAudioMask & PmAudioMask )
            {
                BYTE LcCurrentBoard = TbDhsInPipeInfo[i].BoardNum;
                WORD LcAudioIdx = TbDhsInPipeInfo[i].AudioIdx;
                PVOIE_INFO LcPtVoie = NULL;

                PDHS_AUDIO_INFO LcDhsInAudioInfo = &TbDhsInAudioInfo[LcCurrentBoard][LcAudioIdx];
                ASSERT(LcDhsInAudioInfo->PipeInfoIdx == i); // check coherence

                PmAudioMask &= ~LcOneAudioMask;

                // Get analog level
                LcRet = CUR_COMMANDS_PTR->PIOGetInAudioLevels(  LcAudioIdx,
                                                                &PmLPtrAudio->ilpAnalogLevel,
                                                                &PmLPtrAudio->ilpAnalogMute,
                                                                &PmLPtrAudio->ilpMicroLevel,
                                                                &PmLPtrAudio->ilpMicroMute );
                if(LcRet & ERROR_MASK)  EXIT_PCX_ERROR(LcRet);

                DOUT(DBG_PRINT, ("DHSGetInAudioLevels Pipe(%d) Audio(%d) analog Level(%x)\n", LcInPipeIndex, LcAudioIdx, PmLPtrAudio->ilpAnalogLevel));

                // Digital Levels : Is there a real pipe ?
                // TODO : is this really necessary ???
                LcRet = LookupBoardAudio(LcCurrentBoard, 0, LcAudioIdx, AUDIO_PHYSICAL | AUDIO_IN, &LcPtVoie);
                if( (LcRet == SUCCESS) && (LcPtVoie != NULL) )
                {
                    LEVEL_AUDIO_INFO LcLevelAudioInf;

                    // TODO : get info for only one audio. could be more if same pipe
                    //
                    LcRet = CUR_PROTOCOL_PTR->ILevel_AudioGetLevels(	TRUE,
														                UTIMask64bit( LcAudioIdx ),
														                1,
														                &LcLevelAudioInf );
                    if(LcRet & ERROR_MASK)  EXIT_PCX_ERROR(LcRet);

                    PmLPtrAudio->ilpDigitalLevel = LcLevelAudioInf.gaiDigitalLevel;
                }
                else
                {
                    // if this audio is controlled by DHS, values are memorized
                    if(LcDhsInAudioInfo->IsControlled)
                        PmLPtrAudio->ilpDigitalLevel = LcDhsInAudioInfo->silviDigitalLevel;
                    else
                        PmLPtrAudio->ilpDigitalLevel = LEVEL_DRV_VALUE_0_DB;
                }

                PmLPtrAudio++;
                (*PmNbAudioPtr)++;
            }
        }
    }

    return SUCCESS;
}


// *************************************************************************
//
// *************************************************************************
STATIC WORD DHSGetAudioVuMeter( IN  DWORDLONG               PmOutPipeMask,
                                IN  DWORDLONG               PmInPipeMask,
                                IN  WORD                    PmAudioMask,
                                OUT LPVU_METER_RESP_INFO    PmLPtrRepVuMeter,
                                OUT PWORD                   PmNbAudioPtr )
{
    WORD    LcRet;
    WORD    LcNbAudio;
    WORD    LcCurrentBoard;
    BOOLEAN LcIsCapture;
    PDHS_PIPE_INFO LcDhsPipeInfo;
    int     i;
    STATIC  VU_PIC_METER_INFO    LcVuPicMeterInf[MAX_BOARD_AUDIO];
    DWORDLONG   LcAudioMask;
    DWORDLONG   LcPipeMask;

    *PmNbAudioPtr = 0;

    // in or out pipe, not both
    if( (PmInPipeMask != 0) && (PmOutPipeMask != 0) )   EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );

    // verify this mask specify valid DHS pipes
    LcRet = ChkPmManyDHSPipeMask( PmInPipeMask, PmOutPipeMask );
    if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

    if(PmInPipeMask)
    {
        LcIsCapture = TRUE;
        LcPipeMask = PmInPipeMask;
        LcDhsPipeInfo = TbDhsInPipeInfo;
    }
    else
    {
        LcIsCapture = FALSE;
        LcPipeMask = PmOutPipeMask;
        LcDhsPipeInfo = TbDhsOutPipeInfo;
    }

    LcAudioMask = 0;
    LcNbAudio = 0;
    // create the mask of all audios (play or rec) on one board
    for ( i=0; i<MAX_DHS_PIPE; i++ )
    {
        if( (LcDhsPipeInfo[i].Allocated) &&
            (UTIMask64bit( LcDhsPipeInfo[i].PipeIdx ) & LcPipeMask ) &&
            (UTIMaskWord( LcDhsPipeInfo[i].AppAudioNum ) & PmAudioMask ) )
        {
            if(LcNbAudio == 0)  LcCurrentBoard = LcDhsPipeInfo[i].BoardNum;
            else
            {
                // all pipes have to be on the same board !
                if( LcCurrentBoard != LcDhsPipeInfo[i].BoardNum )
                    EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );
            }
            LcAudioMask |= UTIMask64bit( LcDhsPipeInfo[i].AudioIdx );
            LcNbAudio++;
        }
    }
    DOUT(DBG_PRINT, ("DHSGetAudioVuMeter AudioMask(%x) NbAudio(%d)\n", LcAudioMask, LcNbAudio));

    if((LcNbAudio == 0) || (LcNbAudio > MAX_BOARD_AUDIO))   EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );

    // TODO test if real pipe exist and build a new audio mask !!!

    LcRet = CUR_PROTOCOL_PTR->ILevel_AudioGetVuPicMeter(	LcIsCapture,
															LcAudioMask,
															LcNbAudio,
															LcVuPicMeterInf );
    // No audio meters returned in error case
    if ( LcRet != SUCCESS ) LcNbAudio = 0;

    // Fill output parameters
    // -----------------------
    *PmNbAudioPtr = LcNbAudio ;

    for ( i=0; i<LcNbAudio; i++, PmLPtrRepVuMeter++ )
    {
        PmLPtrRepVuMeter->vmSaturation = LcVuPicMeterInf[i].vpmiSaturation;
        PmLPtrRepVuMeter->vmVuMeter    = LcVuPicMeterInf[i].vpmiVuMeter;
        PmLPtrRepVuMeter->vmPeakMeter  = LcVuPicMeterInf[i].vpmiPicMeter;
    }

    return LcRet;
}


// *************************************************************************
//
// *************************************************************************
STATIC WORD DHSSetOutPipeSettings(LPOUT_PIPE_SET_REQ_INFO PmPtrReq)
{
    WORD        LcRet;
    DWORDLONG   LcPipeMaskOut = PmPtrReq->opqOutPipeMask64.QuadPart;
    BOOLEAN LcParaUerOut = FALSE;
    BOOLEAN LcParaUerExtra = FALSE;

    // verify this mask specify valid DHS pipes
    LcRet = ChkPmManyDHSPipeMask( 0, LcPipeMaskOut );
    if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

    // only UER_OUT and UER_EXTRA params
    if ( PmPtrReq->opqPar1 & ~(OUT_PIPE_SET_PARAM_UER_MASK | OUT_PIPE_SET_PARAM_UER_EXTRA_MASK) )
        EXIT_PCX_ERROR( ED_BAD_UER_MODE );

    if ( PmPtrReq->opqPar1 & OUT_PIPE_SET_PARAM_UER_MASK )
    {
        // Verify UER parameters values
        // ----------------------------
        if( ( PmPtrReq->opqUer != UER_CONSUMER     ) &&
            ( PmPtrReq->opqUer != UER_PROFESSIONAL ) )
        {
            EXIT_PCX_ERROR( ED_BAD_UER_MODE );
        }
        LcParaUerOut = TRUE;
    }
    if ( PmPtrReq->opqPar1 & OUT_PIPE_SET_PARAM_UER_EXTRA_MASK )
    {
		if( PmPtrReq->opqUerExtra & ~(UER_AUDIO_DATA_MASK | UER_COPYRIGHT_MASK | UER_CHANNEL_MODE_MASK) )
		{
            EXIT_PCX_ERROR( ED_BAD_UER_MODE );
		}
        LcParaUerExtra = TRUE;
    }

    // for all pipes specified
    // ***********************
    while( LcPipeMaskOut )
    {
        DWORD   LcFrequency;
        WORD    LcPipeIndex, LcCurrentBoard, LcAudioNum;
        BYTE    LcUerParam;

        LcPipeIndex = UTI64Mask2Word( LcPipeMaskOut );
        LcPipeMaskOut &= ~UTIMask64bit( LcPipeIndex );

        LcCurrentBoard = TbDhsOutPipeInfo[LcPipeIndex].BoardNum;
        LcAudioNum = TbDhsOutPipeInfo[LcPipeIndex].AudioIdx;

        // if this audio is not controlled
        if( TbDhsOutAudioInfo[LcCurrentBoard][LcAudioNum].IsControlled == FALSE )
            EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );

        // combine information opqUer and opqUerExtra !!
        if( LcParaUerOut )      LcUerParam = PmPtrReq->opqUer;
        else                    LcUerParam = CUR_COMMANDS_PTR->PIOGetAudioUerOutMode(LcAudioNum) & UER_MODE_MASK;

        if( LcParaUerExtra )    LcUerParam |= PmPtrReq->opqUerExtra;
        else                    LcUerParam |= CUR_COMMANDS_PTR->PIOGetAudioUerOutMode(LcAudioNum) & ~UER_MODE_MASK;

        // get the current frequency
        LcFrequency = CUR_COMMANDS_PTR->PIOGetActualClockFrequency();

        if( LcFrequency == 0 ) LcFrequency = PmPtrReq->opqFrequency;

        DOUT(DBG_PRINT, ("DHSSetOutPipeSettings LcAudioNum(%d) LcFrequency(%d) LcUerParam(0x%x)\n", LcAudioNum, LcFrequency, LcUerParam));

        // It's OK, let's process the command
        // ----------------------------------
        LcRet = CUR_COMMANDS_PTR->PIOSetParamOutPipe(   LcAudioNum,
                                                        LcFrequency,
                                                        LcUerParam );
        if ( LcRet != SUCCESS )
        {
            EXIT_PCX_ERROR( LcRet );
        }
    } // End for each pipe

    return SUCCESS;
}


// *************************************************************************
//
// *************************************************************************
STATIC WORD DHSGetAudioUer( IN  DWORDLONG       PmInPipeMask,
                            IN  WORD            PmAudioMask,
                            OUT LPUER_RESP_INFO PmUerDescPtr,
                            OUT PWORD           PmNbAudioPtr )
{
    WORD    LcInPipeIndex ;
    WORD    LcRet;
    int     i;

    *PmNbAudioPtr = 0;

    // verify this mask specify one pipe ...
    // **********************************************
    LcRet = ChkPmManyDHSPipeMask( PmInPipeMask, 0 );
    if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

    LcInPipeIndex = UTI64Mask2Word( PmInPipeMask ) ;

    // ... and only one pipe
    // **********************************************
    if ( PmInPipeMask & ~UTIMask64bit( LcInPipeIndex ) ) EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );

    for ( i=0; i<MAX_DHS_PIPE; i++ )
    {
        if(PmAudioMask == 0)    break;  // for

        if( (TbDhsInPipeInfo[i].Allocated) &&
            (TbDhsInPipeInfo[i].PipeIdx == LcInPipeIndex) )
        {
            BYTE LcAppAudioNum = TbDhsInPipeInfo[i].AppAudioNum;
            WORD LcOneAudioMask = UTIMaskWord( LcAppAudioNum );

            if( LcOneAudioMask & PmAudioMask )
            {
                BYTE LcCurrentBoard = TbDhsInPipeInfo[i].BoardNum;
                WORD LcAudioIdx = TbDhsInPipeInfo[i].AudioIdx;

                PDHS_AUDIO_INFO LcDhsInAudioInfo = &TbDhsInAudioInfo[LcCurrentBoard][LcAudioIdx];
                ASSERT(LcDhsInAudioInfo->PipeInfoIdx == i); // check coherence

                // Retrieve the UER information
                // ****************************
                LcRet = CUR_COMMANDS_PTR->PIOGetAudioUer(   LcAudioIdx,
                                                            &PmUerDescPtr->urpData,
                                                            &PmUerDescPtr->urpMode,
                                                            &PmUerDescPtr->urpFrequency,
                                                            &PmUerDescPtr->urpExtra );
                if(LcRet != SUCCESS) break;
                DOUT(DBG_PRINT, ("DHSGetAudioUer Audio(%d) Mode(%d) freq(%d)\n", LcAudioIdx, PmUerDescPtr->urpMode, PmUerDescPtr->urpFrequency));

                PmAudioMask &= ~LcOneAudioMask;
                PmUerDescPtr++;
                (*PmNbAudioPtr)++;
            }
        }
    }

    return LcRet;
}


// **************************************************************************
// 
// **************************************************************************
STATIC WORD DHSSetInPipeSettings(   IN DWORDLONG PmPipeMaskIn,
                                    IN DWORD PmMaskPara1,
                                    IN BYTE  PmSource,
                                    IN BYTE  PmUER_SRC )
{
    WORD    LcPipeIndex;
    WORD    LcRet = SUCCESS;
    WORD    LcError;
    WORD    LcCurrentBoard;
    DWORDLONG   LcCurrentAudioMask;
    int     i;

    // test mask of pipes
    LcRet = ChkPmManyDHSPipeMask( PmPipeMaskIn, 0 );
    if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

	// For all the pipes
    // -----------------
    while( PmPipeMaskIn )
    {
        LcPipeIndex = UTI64Mask2Word( PmPipeMaskIn );
        PmPipeMaskIn &= ~UTIMask64bit( LcPipeIndex );

        LcCurrentBoard = TbDhsInPipeInfo[LcPipeIndex].BoardNum;
        LcCurrentAudioMask = TbDhsInPipeInfo[LcPipeIndex].AudioMask;

        // if this audio is not controlled
        if( TbDhsInAudioInfo[LcCurrentBoard][TbDhsInPipeInfo[LcPipeIndex].AudioIdx].IsControlled == FALSE )
            EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );

        if ( PmMaskPara1 & IN_PIPE_SET_PARAM_SOURCE_MASK )
        {
            if ( PmSource == DATA_FROM_NO_INPUT )
            {
                CUR_PROTOCOL_PTR->IResource_Source_Release_Audios(LcCurrentAudioMask);
                continue;
            }
        }
        else    // only SRC to change
        {
            if( PmUER_SRC == 0 )    EXIT_PCX_ERROR( ED_BAD_VALIDATION_MASK );

            PmSource = CUR_COMMANDS_PTR->PIOGetOneAudioSource(TbDhsInPipeInfo[LcPipeIndex].AudioIdx);

            if( (PmSource != DATA_FROM_DIGITAL_SYNCHRO) &&
                (PmSource != DATA_FROM_DIGITAL_DATA) &&
                (PmSource != DATA_FROM_SYNC_CONNECTOR) )
            {
                EXIT_PCX_ERROR( ED_BAD_VALIDATION_MASK );
            }
        }

        LcError = CUR_PROTOCOL_PTR->IResource_Source_Change_Audios( LcCurrentAudioMask, PmSource, PmUER_SRC );
        if (LcError != SUCCESS)
        {
	        LcRet = LcError;	// warning possible -> continue !
	        if(LcRet & ERROR_MASK) break;
        }
    }
    return LcRet;
}


// *************************************************************************
//
// *************************************************************************
STATIC WORD DHSSamplingClock( LPSAMPLE_CLK_REQ_INFO PmPtrReq )
{
    BOOLEAN LcLockClock = TRUE;
    BOOLEAN LcFixFreq = TRUE;
    WORD    LcPipeIndex, LcRet;
    DWORDLONG   LcInPipeMask, LcOutPipeMask;
    WORD    LcCurrentBoard;

    if( PmPtrReq->scqSource == CLOCK_TYPE_NONE )
    {
        LcLockClock = FALSE;
        LcFixFreq = FALSE;
    }
    else
    if( (PmPtrReq->scqSource == CLOCK_TYPE_INTERNAL) ||
        (PmPtrReq->scqSource == CLOCK_TYPE_PROGRAMMABLE_CLOCK) )
    {
        if(PmPtrReq->scqValidationMask & 0x10)
        {
            DOUT(DBG_PRINT, ("DHSSamplingClock : frequency not locked !\n"));
            LcFixFreq  = FALSE;
        }
    }

    LcInPipeMask = PmPtrReq->scqInputMask64.QuadPart;
    LcOutPipeMask = PmPtrReq->scqOutputMask64.QuadPart;

    // test mask of pipes
    // only one pipe allowed
    if( (LcInPipeMask != 0) && (LcOutPipeMask != 0))    EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );

    LcRet = ChkPmManyDHSPipeMask( LcInPipeMask, LcOutPipeMask );
    if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

    if(LcInPipeMask)
    {
        LcPipeIndex = UTI64Mask2Word( LcInPipeMask );
        // only one pipe
        if ( LcInPipeMask & ~UTIMask64bit( LcPipeIndex ) )  EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );

        LcCurrentBoard = TbDhsInPipeInfo[LcPipeIndex].BoardNum;

        // if this audio is not controlled
        if( TbDhsInAudioInfo[LcCurrentBoard][TbDhsInPipeInfo[LcPipeIndex].AudioIdx].IsControlled == FALSE )
            EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );
    }
    else
    {
        LcPipeIndex = UTI64Mask2Word( LcOutPipeMask );
        // only one pipe
        if ( LcOutPipeMask & ~UTIMask64bit( LcPipeIndex ) ) EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );

        LcCurrentBoard = TbDhsOutPipeInfo[LcPipeIndex].BoardNum;

        // if this audio is not controlled
        if( TbDhsOutAudioInfo[LcCurrentBoard][TbDhsOutPipeInfo[LcPipeIndex].AudioIdx].IsControlled == FALSE )
            EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );
    }

    if( PmPtrReq->scqSource != CLOCK_TYPE_NONE )
    {
        // set the clock of the concerned board!
        //
        LcRet = CUR_COMMANDS_PTR->PIODefClock(
            INDEPENDANT_CLOCK,
            PmPtrReq->scqSource,
            PmPtrReq->scqFrequency,
            PmPtrReq->scqSyncInputNum,
		    PmPtrReq->scqClockFormat);

        if(LcRet & ERROR_MASK)  EXIT_PCX_ERROR(LcRet);

        // LXES
        if( PmPtrReq->scqSource == CLOCK_TYPE_ETHERSOUND )
        {
            DWORD LcFrequency;
            BOOLEAN LcDummy;
            LcRet = CUR_PROTOCOL_PTR->IClock_BoardGetClockFrequency( &LcFrequency, &LcDummy);
            if(LcRet == SUCCESS)    CUR_COMMANDS_PTR->PIOSetActualClockFrequency(LcFrequency);
        }
    }

    TbDhsBoardClock[LcCurrentBoard].Locked = LcLockClock;
    TbDhsBoardClock[LcCurrentBoard].FrequFixed = LcFixFreq;

    return SUCCESS;
}


// *************************************************************************
//
// *************************************************************************
STATIC WORD DHSGetBoardAudioFromOnePipe(DWORDLONG PmOutPipeMask, DWORDLONG PmInPipeMask, PWORD PmPBoard, PWORD PmPAudio)
{
    PDHS_PIPE_INFO  LcDhsPipeInfo;
    DWORDLONG       LcPipeMask;

    // verify this mask specify one pipe ...
    WORD    LcRet = ChkPmManyDHSPipeMask( PmInPipeMask, PmOutPipeMask );
    if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

    if(PmInPipeMask)
    {
        LcPipeMask = PmInPipeMask;
        LcDhsPipeInfo = TbDhsInPipeInfo;
    }
    else
    {
        LcPipeMask = PmOutPipeMask;
        LcDhsPipeInfo = TbDhsOutPipeInfo;
    }
    WORD LcPipeIndex = UTI64Mask2Word( LcPipeMask );

    // ... and only one pipe
    if( (LcPipeMask & ~UTIMask64bit( LcPipeIndex )) ||
        ((PmInPipeMask != 0) && (PmOutPipeMask != 0)) )
    {
        EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );
    }

    *PmPBoard = LcDhsPipeInfo[LcPipeIndex].BoardNum;
    *PmPAudio = LcDhsPipeInfo[LcPipeIndex].AudioIdx;

    return SUCCESS;
}


// *************************************************************************
//
// *************************************************************************
STATIC WORD DHSChangeMonitoringSrc(LPCHANGE_DESC_INFO PmChangeDescPtr, int PmNbChanges)
{
    WORD LcRet;
    int  i, k;
    LPCHANGE_DESC_INFO LcChangeDescPtr = PmChangeDescPtr;

    // process each structure
    // **********************
    for ( i=0; i<PmNbChanges; i++, LcChangeDescPtr++ )
    {
        WORD LcAppAudioNum;
        WORD LcPipeIndex;
        PDHS_PIPE_INFO LcPPipeInfo;

        DWORDLONG LcOutPipeMask = LcChangeDescPtr->cdOutPipeMask64.QuadPart;
        DWORDLONG LcInPipeMask = LcChangeDescPtr->cdInPipeMask64.QuadPart;
        DWORD     LcOutAudioMask;
        DWORD     LcInAudioMask;

        // verify this mask specify valid DHS in and out pipes
        LcRet = ChkPmManyDHSPipeMask( LcInPipeMask, LcOutPipeMask );
        if(LcRet != SUCCESS)    EXIT_PCX_ERROR( LcRet );

        if(LcOutPipeMask == 0)      EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );

        LcPipeIndex = UTI64Mask2Word( LcOutPipeMask );
        // and only one pipe
        if( LcOutPipeMask & ~UTIMask64bit( LcPipeIndex ) )  EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );

        // check AppAudio
        LcOutAudioMask = LcChangeDescPtr->cdOutMask;
        if(LcOutAudioMask == 0)  EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );

        LcAppAudioNum = UTIDWMask2Word( LcOutAudioMask );
        // and each mask only one audio
        if( LcOutAudioMask & ~UTIMaskDWord( LcAppAudioNum ) )    EXIT_PCX_ERROR( ED_MANY_AUDIOS_REFUSED );

        LcPPipeInfo = TbDhsOutPipeInfo;
        for( k=0; k<MAX_DHS_PIPE; k++, LcPPipeInfo++)
        {
            if( (LcPPipeInfo->Allocated) &&
                (LcPPipeInfo->PipeIdx == LcPipeIndex) &&
                (LcPPipeInfo->AppAudioNum == LcAppAudioNum))
            {
                break;  // for k_out
            }
        }
        // AppAudio not found
        if( k >= MAX_DHS_PIPE )  EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );

        // the DHS out audio should be a managed pipe (in pipe can be unmanaged)
        if( ! TbDhsOutAudioInfo[LcPPipeInfo->BoardNum][LcPPipeInfo->AudioIdx].IsControlled )
        {
            EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );
        }

        // params ok, store the resulting params in each structure
        LcChangeDescPtr->cdPrivateBoard     = LcPPipeInfo->BoardNum;
        LcChangeDescPtr->cdPrivateAudioOut  = LcPPipeInfo->AudioIdx;

        // if no in pipe, reset out pipe monitoring source (in audio face to out audio)
        if( LcInPipeMask == 0 )
        {
            LcChangeDescPtr->cdPrivateAudioIn = LcPPipeInfo->AudioIdx;
        }
        else
        {
            LcPipeIndex = UTI64Mask2Word( LcInPipeMask );
            // and each mask only one pipe
            if( LcInPipeMask & ~UTIMask64bit( LcPipeIndex ) )   EXIT_PCX_ERROR( ED_MANY_PIPES_REFUSED );

            // check AppAudio
            LcInAudioMask = LcChangeDescPtr->cdInMask;
            if(LcInAudioMask == 0)  EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );

            LcAppAudioNum = UTIDWMask2Word( LcInAudioMask );
            // and each mask only one audio
            if( LcInAudioMask & ~UTIMaskDWord( LcAppAudioNum ) ) EXIT_PCX_ERROR( ED_MANY_AUDIOS_REFUSED );

            // seek for pipes AppAudio
            LcPPipeInfo = TbDhsInPipeInfo;
            for( k=0; k<MAX_DHS_PIPE; k++, LcPPipeInfo++)
            {
                if( (LcPPipeInfo->Allocated) &&
                    (LcPPipeInfo->PipeIdx == LcPipeIndex) &&
                    (LcPPipeInfo->AppAudioNum == LcAppAudioNum))
                {
                    break;  // for k_in
                }
            }
            // AppAudio not found
            if( k >= MAX_DHS_PIPE )  EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );

            // in and out pipe should be on the same board !
            if( LcChangeDescPtr->cdPrivateBoard != LcPPipeInfo->BoardNum )
            {
                EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );
            }
            // params ok, store the resulting param in each structure
            LcChangeDescPtr->cdPrivateAudioIn = LcPPipeInfo->AudioIdx;
        }
    }

    // process each structure
    // **********************

    LcChangeDescPtr = PmChangeDescPtr;

    for ( i=0; i<PmNbChanges; i++, LcChangeDescPtr++ )
    {
        BYTE LcCurrentBoard = LcChangeDescPtr->cdPrivateBoard;
        PVOIE_INFO LcPtVoie = NULL;

        // Save the routing
        TbDhsOutAudioInfo[LcCurrentBoard][LcChangeDescPtr->cdPrivateAudioOut].solviMonitorSource = LcChangeDescPtr->cdPrivateAudioIn;

        // Send the request if a real pipe out exists
        // ------------------------------------------
        LcRet = LookupBoardAudio(LcCurrentBoard, 0, LcChangeDescPtr->cdPrivateAudioOut, AUDIO_PHYSICAL | AUDIO_OUT, &LcPtVoie);
        if( (LcRet == SUCCESS) && (LcPtVoie != NULL) )
        {
            LcRet = CUR_PROTOCOL_PTR->IFlow_ChangeMonitoringSource( LcChangeDescPtr->cdPrivateAudioOut,
                                                                    LcChangeDescPtr->cdPrivateAudioIn );

            DOUT(DBG_PRINT, ("IFlow_ChangeMonitoringSource Board(%d) Audio %d->%d\n", LcCurrentBoard, LcChangeDescPtr->cdPrivateAudioIn, LcChangeDescPtr->cdPrivateAudioOut));

            // Give up if an error occurs
            // --------------------------
            if ( LcRet != SUCCESS )
            {
                EXIT_PCX_ERROR( LcRet );
            }
        }
    }
    return SUCCESS;
}


// *************************************************************************
//
// *************************************************************************
BOOLEAN APHChkIsAudioDHSControlled(
    IN  WORD    PmBoard,
    IN  WORD    PmAudioIdx,
    IN  BYTE    PmIsCapture )
{
    ASSERT(PmBoard < MAX_BOARD);
    ASSERT(PmAudioIdx < MAX_BOARD_AUDIO);

    // verify this audio is controlled by DHS
    // **************************************
    if( PmIsCapture )
    {
        if(TbDhsInAudioInfo[PmBoard][PmAudioIdx].IsControlled)
            return TRUE;
    }
    else
    {
        if(TbDhsOutAudioInfo[PmBoard][PmAudioIdx].IsControlled)
            return TRUE;
    }
    return FALSE;   // this audio is not controlled
}


// *************************************************************************
//
// *************************************************************************
WORD APHNewPipeSetDHSLevels( IN WORD PmBoard, IN WORD PmAudioIdx, IN BYTE PmIsCapture )
{
    LEVEL_AUDIO_INFO    LcLevelAudioInf;
    TARGET_INFO         LcTarget;
    WORD                LcRet;
    WORD                LcCurrentBoard = PmBoard;

    ASSERT(PmBoard < MAX_BOARD);
    ASSERT(PmAudioIdx < MAX_BOARD_AUDIO);

    BZERO2( &LcTarget, LcTarget, 1 );

    // verify this audio is controlled by DHS
    // **************************************
    if( PmIsCapture )
    {
        PDHS_AUDIO_INFO LcDhsInAudioInfo = &TbDhsInAudioInfo[PmBoard][PmAudioIdx];

        if(LcDhsInAudioInfo->IsControlled == FALSE)
            EXIT_PCX_ERROR(ED_INVALID_PIPE_AUDIO);

        LcTarget.tgCaracPipeVoie = OPER_REC;

        BZERO2( &LcLevelAudioInf, LcLevelAudioInf, 1 );
        LcLevelAudioInf.gaiHasDigitalLevel  = TRUE;
        LcLevelAudioInf.gaiDigitalLevel = LcDhsInAudioInfo->silviDigitalLevel;
    }
    else
    {
        PDHS_AUDIO_INFO LcDhsOutAudioInfo = &TbDhsOutAudioInfo[PmBoard][PmAudioIdx];

        if(LcDhsOutAudioInfo->IsControlled == FALSE)
            EXIT_PCX_ERROR(ED_INVALID_PIPE_AUDIO);

        LcTarget.tgCaracPipeVoie = OPER_PLAY;

        LcLevelAudioInf.gaiHasDigitalLevel  = TRUE;
        LcLevelAudioInf.gaiHasMonitorLevel  = TRUE;
        LcLevelAudioInf.gaiHasMuteLevel     = TRUE;
        LcLevelAudioInf.gaiHasM1Level       = TRUE;
        LcLevelAudioInf.gaiHasM2Level       = TRUE;
        LcLevelAudioInf.gaiDigitalLevel = LcDhsOutAudioInfo->solviDigitalLevel;
        LcLevelAudioInf.gaiMonitorLevel = LcDhsOutAudioInfo->solviMonitorLevel;
        LcLevelAudioInf.gaiMuteLevel    = LcDhsOutAudioInfo->solviMuteLevel;
        LcLevelAudioInf.gaiM1Level      = LcDhsOutAudioInfo->solviM1Level;
        LcLevelAudioInf.gaiM2Level      = LcDhsOutAudioInfo->solviM2Level;

        // change monitoring source if not default:
        if( PmAudioIdx != LcDhsOutAudioInfo->solviMonitorSource )
        {
            LcRet = CUR_PROTOCOL_PTR->IFlow_ChangeMonitoringSource( PmAudioIdx, LcDhsOutAudioInfo->solviMonitorSource );

            DOUT(DBG_PRINT, ("IFlow_ChangeMonitoringSource Board(%d) Audio %d->%d ret(%x)\n", LcCurrentBoard, LcDhsOutAudioInfo->solviMonitorSource, PmAudioIdx, LcRet));
        }
    }

    // adjust the audio digital volumes (TODO : here only mono, could be more channels if same real pipe)
    //
    LcRet = CUR_PROTOCOL_PTR->ILevel_AudioSetDigitalLevel(	&LcTarget,
															UTIMask64bit( PmAudioIdx ),
															&LcLevelAudioInf );

    DOUT(DBG_PRINT, ("APHNewPipeSetDHSLevels %s Audio(%d) ret(%x)\n", PmIsCapture ? "REC" : "PLAY", PmAudioIdx, LcRet));

    return LcRet;
}


// *************************************************************************
//
// *************************************************************************
BOOLEAN APHIsClockDHSLocked( IN WORD PmBoard, OUT PBOOLEAN PmFrequencyFixed )
{
    ASSERT(PmBoard < MAX_BOARD);

    if(TbDhsBoardClock[PmBoard].Locked)
    {
        if(PmFrequencyFixed)    *PmFrequencyFixed = TbDhsBoardClock[PmBoard].FrequFixed;
        return TRUE;
    }
    if(PmFrequencyFixed)    *PmFrequencyFixed = FALSE;
    return FALSE;
}
