// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

#if !defined(_PIOCOMMANDS_H__INCLUDED_)
#define _PIOCOMMANDS_H__INCLUDED_

#include "Platform.h"

#include "drvdef.h"	// Added by ClassView

class CProtocol;

#ifndef BOOL
#define BOOL int
#endif

// TAG_DEBUG_DRIVER : DEB

/**
*   LEVEL_INFO
*/
typedef struct LEVEL_INFO
{
    LONG liMinReal;                   /**< min real value */
    LONG liMinCoded;                  /**< min coded value */
    LONG liMaxCoded;                  /**< max coded value */
} LEVEL_INFO, *PLEVEL_INFO;

/**
*   BOARD_LEVEL_INFO
*/
typedef struct BOARD_LEVEL_INFO
{
    LEVEL_INFO bliInputLevel;         /**< Input  levels of the board */
    LEVEL_INFO bliOutputLevel;        /**< Output levels of the board */
} BOARD_LEVEL_INFO, *PBOARD_LEVEL_INFO;

// TAG_DEBUG_DRIVER : FIN


/* ------------ HOST 8 BITS 56303, 56002 ------------------------*/

// ICR bits
#define HI08_ICR_RREQ       0x01        // Receive Request Enable
#define HI08_ICR_TREQ       0x02        // Transmit Request Enable
#define HI08_ICR_HDRQ       0x04        // Double Host Request. (reserved on 56002)
#define HI08_ICR_HF0        0x08
#define HI08_ICR_HF1        0x10
#define HI08_ICR_HLEND      0x20        // 2 bits : Host Little Endian
#define HI08_ICR_INIT       0x80

// ISR bits
#define HI08_ISR_RXDF       0x01        // Receive Data Register Full
#define HI08_ISR_TXDE       0x02        // Transmit Data Register Empty
#define HI08_ISR_TRDY       0x04        // Transmitter Ready
#define HI08_ISR_HF2        0x08        // rmh ERR
#define HI08_ISR_HF3        0x10        // emh CHK
#define HI08_ISR_HREQ       0x80        // Host Request

// protocol bits
#define HI08_ISR_ERR        HI08_ISR_HF2
#define HI08_ISR_CHK        HI08_ISR_HF3

// CVR bits
#define HI08_CVR_HC         0x80

// Timeout de 500 ms ( 9 * 55 ms )
//
#define timeoutDSP          9

// Tell that no irq was issued by the tested card
//
#define PCX_IRQ_NONE 0  


// constants ( PIOSendMessage)
//*****************

#define MASK_MORE_THAN_1_WORD_COMMAND   (1<<BIT_MSG_BIT_M)
#define MASK_1_WORD_COMMAND             ~MASK_MORE_THAN_1_WORD_COMMAND


// Constants used to access C Bits
// **********************************
#define UER_MODE_BIT_OFFSET         0
#define UER_DATA_MODE_OFFSET        1

// Pro mode
#define UER_PRO_FREQ_BIT_OFFSET     6  // and 7
#define UER_PRO_STEREO_BIT_OFFSET   9
#define UER_PRO_AUX1_BIT_OFFSET     18
#define UER_PRO_SWL3_BIT_OFFSET     19
#define UER_PRO_SWL5_BIT_OFFSET     21

// Con mode
#define UER_CON_COPYRIGHT_OFFSET    2
#define UER_CON_CATEGORY_OFFSET     9
#define UER_CON_FREQ_BIT_OFFSET     24 // and 25

// Used fields values
// ------------------

// Pro mode
#define UER_PRO_48K_CBITS_VAL       1
#define UER_PRO_44K_CBITS_VAL       2
#define UER_PRO_32K_CBITS_VAL       3
#define UER_PRO_XXK_CBITS_VAL       0

// Con mode
#define UER_CON_48K_CBITS_VAL       1
#define UER_CON_44K_CBITS_VAL       0
#define UER_CON_32K_CBITS_VAL       3

// sampling frequency Byte 4
// -------------------------

#define UER_EXT_FREQ_BIT_OFFSET     35      // until 38
#define UER_EXT_FREQ_BYTE4_MASK     0x1E    // 00011110

#define UER_EXT_FREQ__24000_VAL     0x10    // 00010000
#define UER_EXT_FREQ__96000_VAL     0x08    // 00001000
#define UER_EXT_FREQ_192000_VAL     0x18    // 00011000
#define UER_EXT_FREQ__22050_VAL     0x12    // 00010010
#define UER_EXT_FREQ__88200_VAL     0x0A    // 00001010
#define UER_EXT_FREQ_176400_VAL     0x1A    // 00011010

typedef enum _SRC_CONFIG
{
    SRC_off	 = 0,	// SRC is switched off
    SRC_in	 = 1,	// SRC is enabled with the input stereo pair
    SRC_out	 = 2	// SRC is enabled with the output stereo pair
} SRC_CONFIG;



typedef struct IOPCIDevice
{
    PBYTE AddressPLX;
    PBYTE AddressDSP;
    ULONG SizePLX;
    ULONG SizeDSP;
    DWORD dwBoardVersion;
    BYTE  ucBoardFlavor;
} IOPCIDevice;


typedef enum
{
    // PLX
    ePLX_PCICR,

    ePLX_MBOX0,
    ePLX_MBOX1,
    ePLX_MBOX2,
    ePLX_MBOX3,
    ePLX_MBOX4,
    ePLX_MBOX5,
    ePLX_MBOX6,
    ePLX_MBOX7,

    ePLX_L2PCIDB,
    ePLX_IRQCS,
    ePLX_CHIPSC,

    eMaxPort

} PCXPORT;


/**
* CPIOCommands class is used as a common interface for the derived classes
* CP92IOCommands class (VX222, PCX924)
*/
class CPIOCommands
{
public:
    /**
    *   Constructor. Instance will never be created directly. Create objects of it's derived classes
    */
	CPIOCommands(IOPCIDevice* pIOPCIDevice);
    /**
    *   Destructor.
    */
	virtual ~CPIOCommands();

	virtual PKSPIN_LOCK GetMsgSpinLock() = 0;

    static  VOID  WaitForN( ULONG PmDuration );

/**
 * \defgroup PIO PIOCommands exported Interface
 */
/*@{*/

            /**
            * This function is needed to keep locally the path to the driver files. This
            * path is used in various routines such as PIOTestConfig, PIOResetBoard, etc.
            *
            */ 
            VOID    PIOSetBoardIndex( IN WORD PmBoardIndex) {m_BoardIndex = PmBoardIndex;};
            WORD    PIOGetBoardIndex( ) { return m_BoardIndex;};


            /**
            *   Get data from a PLX mailbox port.
            *   @return a DWORD read, or 0xdeadbeef if not implemented.
            */
    virtual DWORD PIOReadPLXMailBox( BYTE PmMailBoxNumber) = 0;

            /**
            *   Put data to a PLX mailbox port.
            *   @return a false if not implemented or not permitted
            */
    virtual BOOL   PIOWritePLXMailBox( BYTE PmMailBoxNumber, DWORD PmData) = 0;
    
            /**
            *   
            */
    virtual BOOL PIOSendItRBufferUpdate() { return FALSE; };
    

            /**
            *   Set the audio analog level
            *   @param PmBoardAudioNum  IN audio number
            *   @param PmCaracPipeVoie  IN (OPER_PLAY or OPER_REC)
            *   @param PmDriverCodedLevel  IN driver coded level
            *   @see PIOGetAudioAnalogLevel()
            *   @return  SUCCESS or error code.
            */
            WORD    PIOSetAudioAnalogLevel(WORD PmBoardAudioNum, BYTE PmCaracPipeVoie, WORD PmDriverCodedLevel );
            /**
            *   Set the audio analog in levels
            *   @param PmBoardAudioNum  IN audio number
            *   @param PmPLevels        IN pointer to structure with analog line and micro levels/mutes
            *   @see PIOGetInAudioLevels()
            *   @return  SUCCESS or error code.
            */
            WORD    PIOSetInAudioLevels(WORD PmBoardAudioNum, LPIN_AUDIO_SET_LEVEL_REQ_INFO PmPLevels );
            /**
            *   Set the in audio effects
            *   @param PmBoardAudioNum      IN audio number
            *   @param PmPInAudioEffects    IN pointer to structure with in audio effects
            *   @see PIOGetInAudioLevels()
            *   @return  SUCCESS or error code.
            */
            WORD    PIOSetInAudioEffects(WORD PmBoardAudioNum, LPIN_AUDIO_EFFECT_REQ_INFO PmPInAudioEffects );

            /**
            *   Send message to Dsp.
            *   @param PmPtrCmdRmh   INOUT pointer to RMH structure (PCMD_RMH_INFO)
            *   @return  SUCCESS or error code.
            */
    virtual WORD    PIOSendMessage( INOUT PCMD_RMH_INFO PmPtrCmdRmh) = 0;

            VOID    PIOEnableSendMessage(BOOL bEnable){ m_EnableSendMessage = bEnable;}

            /**
            *   Devalidate Interrupts on the card side.
            *   @see PIODisableIrq()
            */
    virtual VOID    PIODisableIrq( ) = 0;
            /**
            *   Validate Interrupts on the card side.
            *   @see PIOEnableIrq()
            */
    virtual VOID    PIOEnableIrq( ) = 0;
            /**
            *   Set Default Clock and CBit config.
            *   on vx222/PCX924, it also resets the codec.
            *
            */
            VOID   PIODefaultConfig( );
            /**
            *   Test if the card fired an interrupt.
            *   @return InterruptMask or PCX_IRQ_NONE (there was / was not an Interrupt).
            */
    virtual DWORD   PIOTestAndAckIt( ) = 0;
            /**
            *   Get the card type.
            *   @return PCX924_BOARD, PCX924_MIC_BOARD, VX222_BOARD, ...
            */
            BYTE    PIOGetTypeNoAlias( ){ return( m_bliBoardTypeNoAlias );}
            /**
            *   Get the card type.
            *   @return PCXVX_BOARD or PCXVX_2_BOARD
            */
            BYTE    PIOGetType( ){ return( m_piType );}
            BYTE    PIOGetFlavor( ){return m_ucBoardFlavor;}
   
    virtual DWORD   PIOGetFrequencyRatio(){return 1;}

            /**
            *   Get the bus type.
            */
            BYTE    PIOGetBusType( ){ return( m_bliBusType );}

            /*
            *   Get number of audios that provide a sync input
            */
            BYTE    PIOGetUERSyncNumber() { return m_piSn; };

            /*
            *   Get number of audios that provide a sync input
            */
            BYTE    PIOGetWCKSyncNumber() { return m_piNbWcl; };
 
            /*
            *   Get number of audios that provide a video sync input
            */
            BYTE    PIOGetVideoSyncNumber() { return m_piNbVideoSync; }

            /*
            *   Get number of audios that provide a Ethersound sync input
            */
            BYTE    PIOGetEthersoundSyncNumber() { return m_piEtherSoundSync; }

            /**
            *   Get the audio analog level
            *   @param PmBoardAudioNum  IN audio number
            *   @param PmCaracPipeVoie  IN (OPER_PLAY or OPER_REC)
            *   @param PmPDriverCodedLevel  OUT driver coded level
            *   @see PIOSetAudioAnalogLevel()
            *   @return SUCCESS or error code.
            */
            WORD    PIOGetAudioAnalogLevel(WORD PmBoardAudioNum, BYTE PmCaracPipeVoie, OUT PWORD PmPDriverCodedLevel );
            /**
            *   Get the audio analog and micro input levels and their mute state
            *   @param PmBoardAudioNum  IN audio number
            *   @param PmPDriverCodedAnalogLevel  OUT driver coded analog level
            *   @param PmPAnalogMute  OUT true if the analog gain is muted
            *   @param PmPDriverCodedMicroLevel  OUT driver coded micro level
            *   @param PmPMicroMute  OUT true if the microphone gain is muted
            *   @see PIOGetAudioAnalogLevel()
            *   @return SUCCESS or error code.
            */
            WORD    PIOGetInAudioLevels(WORD PmBoardAudioNum, OUT PWORD PmPDriverCodedAnalogLevel, OUT PBYTE PmPAnalogMute, OUT PWORD PmPDriverCodedMicroLevel, OUT PBYTE PmPMicroMute);
            /**
            *   Enable Phantom Power Supply
            *   @param PmBoardAudioNum  IN audio number
            *   @param PmPermanent IN phantom power supply remains on even if PmEnable == FALSE
            *   @param PmEnable IN true if the phantom power supply has to be activated
            *   @return SUCCESS or error code.
            */
    virtual WORD    PIOEnablePhantomPowerSupply(WORD PmBoardAudioNum, BOOL PmPermanent, BOOL PmEnable) { return WD_NO_HARDWARE_SUPPORT;};
           /**
            *   Get the audio source
            *   @return source type (DATA_FROM_ANALOG, DATA_FROM_MICRO, DATA_FROM_DIGITAL_SYNCHRO).
            */
            BYTE    PIOGetOneAudioSource( IN WORD PmAudioNum) { return m_piTypeSrc[PmAudioNum]; }
            /**
            *   Set the audio source
            *   @param PmAudioMask  IN audio mask
            *   @param PmSource  IN source type (DATA_FROM_ANALOG, DATA_FROM_MICRO, DATA_FROM_DIGITAL_SYNCHRO)
            *   @return SUCCESS or error code.
            */
            WORD    PIOSetOneAudioSource( IN WORD PmAudioNum, BYTE PmSource );

            /**
            *   Get audio dependency for source settings 
            *   @param PmAudioNum  IN audio number 
            *   @return DWORD Mask of dependent audios or 0 if PmAudioNum is invalid
            */
            DWORD   PIOGetSourceAudioDependency( IN WORD PmAudioNum );

            /**
            *   Get UER input information (C bits).
            *   @param PmBoardAudio IN  audio number
            *   @param PmDataPtr    OUT pointer to data presence ( UER_DATA_UNKNOWN, UER_DATA_PRESENT )
            *   @param PmModePtr    OUT pointer to AES/EBU mode  ( UER_MODE_UNKNOWN, UER_PROFESSIONAL, UER_CONSUMER )
            *   @param PmCBitFrequ  OUT pointer to AES/EBU frequency (32000, 44100, 48000, ...)
            *   @see PIOReadUerSync()
            *   @return SUCCESS or error code.
            */
            WORD    PIOGetAudioUer( WORD PmBoardAudio, OUT PBYTE PmDataPtr, OUT PBYTE PmModePtr, OUT PDWORD PmCBitFrequ, OUT PBYTE PmUerExtra );

            WORD    PIOGetAudioCByte( WORD PmBoardAudio, BYTE PmByteID, OUT PBYTE PmValidity, OUT PBYTE PmCByte );
            /**
            *   Set UER output information (C bits).
            *   @param PmBoardAudio IN audio number
            *   @param PmFrequency  IN frequency    ( 32000, 44100, 48000 )
            *   @param PmTypeUer    IN AES/EBU mode ( UER_PROFESSIONAL, UER_CONSUMER )
            *   @return SUCCESS or error code.
            */
            WORD    PIOSetParamOutPipe( WORD PmBoardAudio, DWORD PmFrequency, BYTE PmTypeUer = 0xff);

            /**
            *   Get current UER output mode settings.
            *   Does NOT check wheter the audio actually supports UER (default is pro).
            *   @param PmBoardAudio IN audio number
            *   @return UER_PROFESSIONAL, UER_CONSUMER.
            */
            BYTE    PIOGetAudioUerOutMode(WORD PmBoardAudio) { return m_ucUerByte[PmBoardAudio/2];};

            /**
            *   Get UER input information.
            *   @param PmSynchroNum IN  synchro input number
            *   @param PmSync       OUT pointer to synchro presence ( DRV_SYNC_UNKNOWN, DRV_NO_SYNC, DRV_SYNC_PRESENT )
            *   @param PmFrequency  OUT pointer to frequency ( 0, 32000, 44100, 48000 )
            *   @see PIOGetAudioUer()
            *   @return SUCCESS or error code.
            */
            WORD    PIOReadUerSync( WORD PmSynchroNum, OUT PBYTE PmSync, OUT PDWORD PmFrequency, IN BOOL PmWaitLong = FALSE );

    virtual WORD    PIOReadWordClockSync( WORD PmWordClockNum, OUT PBYTE PmSync, OUT PDWORD PmFrequency ) 
                                            { 
                                                if (PmSync) *PmSync = DRV_SYNC_UNKNOWN;
                                                return ED_UNAVAILABLE_FEATURE;
                                            };

    virtual WORD    PIOReadVideoSync( WORD PmVideoSyncNum, OUT PBYTE PmSync, OUT PDWORD PmFrequency ) 
                                            { 
                                                if (PmSync) *PmSync = DRV_SYNC_UNKNOWN;
                                                return ED_UNAVAILABLE_FEATURE;
                                            };
            /**
            *   Get actual clock frequency.
            *   @return actual clock frequency ( 0 ... 48000 )
            */
            DWORD   PIOGetActualClockFrequency() {return m_piActualClkFrequency; }
            /**
            *   Get the actual clock source
            *   @return clock source ( NO_CLOCK, CLOCK_TYPE_INTERNAL, UER_SYNC or PROGRAMMABLE_CLOCK ) 
            */
            WORD    PIOGetActualClkSource(){return m_piActualClkSource;}

            /**
            *   Get the actual clock input num
            *   @return clock input num ( 0: no or Sync, 1: In1, 2, In2, .. ) 
            */
            DWORD   PIOGetActualClkInputNum(){return m_piActualClkInputNum;}

            WORD    PIOGetActualClkType(){return m_piActualClkType;}
            WORD    PIOSetClockDependency( WORD PmClockType );

            WORD    PIOGetActualClkMasterNum(){return m_piActualClkMasterNum;}
            VOID    PIOSetActualClkMasterNum(WORD LcClkMasterNum){m_piActualClkMasterNum = LcClkMasterNum;}
            DWORD   PIOGetActualClkSlaveMask(){return m_piActualClkSlaveMask;}
            VOID    PIOSetActualClkSlaveMask(DWORD LcClkSlaveMask){m_piActualClkSlaveMask = LcClkSlaveMask;}
            VOID    PIOSetSlaveClkSource(WORD LcClkSource){if( m_piActualClkType == SLAVE_CLOCK ) m_piActualClkSource = LcClkSource;}

            /**
            *   Set actual clock frequency to match the one computed by the DSP (for UER/Wordclock)
            */
            VOID    PIOSetActualClockFrequency(DWORD PmComputedFreq) { m_piActualClkFrequency = PmComputedFreq;};
            
            /**
            *   Get synchro input information of the card.
            *   @param PmSourcePtr    OUT clock source ( CLOCK_TYPE_INTERNAL, PROGRAMMABLE_CLOCK, UER_SYNC )
            *   @param PmSyncPtr      OUT synchro      ( DRV_SYNC_UNKNOWN, DRV_NO_SYNC, DRV_SYNC_PRESENT )
            *   @param PmFrequencyPtr OUT frequency    ( 0 ... 48000 )
            *   @see PIOGetAudioUer()
            *   @see PIOReadUerSync()
            *   @return SUCCESS or error code.
            */
            WORD    PIOReadActualSyncForBoard( OUT PBYTE  PmSourcePtr, OUT PBYTE  PmSyncPtr, OUT PDWORD PmFrequencyPtr, IN BOOL PmWaitLong = FALSE );
 
            /**
            *   Set clock of card.
            *   @param PmFreq               IN  Frequency
            *   @param PmClockType          IN  clock type ( CLOCK_TYPE_INTERNAL, PROGRAMMABLE_CLOCK, UER_SYNC )
            *   @see PIODefClock()
            *   @return SUCCESS.
            */
    virtual WORD    PIOSetClock( DWORD PmFreq, WORD PmClockType, DWORD PmSyncInputNum, BYTE PmClockFormat ) = 0;

            /**
            *   For boards with HW options.
            *   @return SUCCESS
            */
    virtual BOOL    PIOTestConfig( INOUT PWORD PmDspType ){ (void)PmDspType; return TRUE;}
            /**
            *   Enable / Disable Timecode Managment in DSP.
            *   @param PmManage     IN  Enable / Disable
            *   @return SUCCESS or error code
            */
    virtual WORD    PIOManageTimeCode( BOOL PmManage ) { return ED_UNAVAILABLE_FEATURE; };
            /**
            *   Get the pointer to this cards specific BOARD_LEVEL_INFO structure.
            *   @param PmPPBoardLevelInfo   OUT  set pointer to BOARD_LEVEL_INFO structure
            *   @param PmPSize              OUT  *PmPSize = sizeof( BOARD_LEVEL_INFO )
            *   @see BOARD_LEVEL_INFO
            *   @return SUCCESS
            */
            WORD    PIOGetBoardLevelInfo( OUT PBYTE *PmPPBoardLevelInfo, OUT PWORD PmPSize );
            /**
            *   Write to the GPIO interface.
            *   @param PmPIOWriteMask   IN mask for the bits to write
            *   @param PmPIODataToWrite IN new bit values
            *   @see PIOInitGPIO()
            *   @see PIOReadGPIO()
            *   @return SUCCESS or error code
            */
            WORD    PIOWriteGPIO( DWORD PmPIOWriteMask, DWORD PmPIODataToWrite );
            /**
            *   Read the GPIO interface.
            *   @param PmPIOReadMask    IN  mask for the bits to read
            *   @param PmPIOData        OUT bit values
            *   @see PIOInitGPIO()
            *   @see PIOWriteGPIO()
            *   @return SUCCESS or error code
            */
            WORD    PIOReadGPIO( DWORD PmPIOReadMask, OUT PDWORD PmPIOData );
            /**
            *   Init the GPIO interface.
            *   @param PmConfigMaskGPIO IN  GPIO bits to open
            *   @param PmConfigMaskDIR  IN  direction (IN/OUT) of the bits
            *   @see PIOGetGPIOFeatures()
            *   @see PIOReadGPIO()
            *   @see PIOWriteGPIO()
            *   @return SUCCESS or error code
            */
            WORD    PIOInitGPIO( DWORD PmConfigMaskGPIO, DWORD PmConfigMaskDIR );
            /**
            *   Get features of the GPIO interface.
            *   @param PmReadAccess     OUT returns mask of the bits that can be read
            *   @param PmWriteAccess    OUT returns mask of the bits that can be written
            *   @param PmProgrammable   OUT returns mask of the bits that can be programmed
            *   @see PIOInitGPIO()
            *   @return SUCCESS or error code
            */
            WORD    PIOGetGPIOFeatures( OUT PDWORD PmReadAccess, OUT PDWORD PmWriteAccess, OUT PDWORD PmProgrammable);
            /**
            *   Define the boards clock (and input source).
            *   @param PmClockType              IN  INDEPENDANT_CLOCK/SLAVE_CLOCK/MASTER_CLOCK
            *   @param PmSource                 IN  CLOCK_TYPE_INTERNAL, UER_SYNC or PROGRAMMABLE_CLOCK
            *   @param PmFrequency              IN  Frequency ( 0 ... 48000 )
            *   @param PmSyncInputNum           IN  number of synchro input (if UER_SYNC used)
            *   @param PmClockFormat            IN  format of clock signal : e.g. VideoFormat PAL/NTSC
            *   @see PIOSetClock()
            *   @warning if the AutomatonState returned not 0, resync is eventually needed ( send the RMH( CMD_RESYNC_AUDIO_INPUTS ) after this method )
            * and the DSP needs to be informed about the new clock ( send the RMH( CMD_MODIFY_CLOCK ) after this method )
            *   @return SUCCESS or error code
            */
            WORD    PIODefClock( IN WORD PmClockType, WORD PmSource, DWORD PmFrequency, DWORD PmSyncInputNum, BYTE PmClockFormat );

            /**
            *   Load the boot and DSP file to the card (the Xilinx must be loaded before!).
            *   @param PmBootLength    IN  size of boot file in 24 bit words (size in bytes / 3)
            *   @param PmBootFile      IN  pointer to boot file
            *   @param PmNbDSPBuff     IN  number of entries of the following array
            *   @param PmDSPBuff       IN  array of struct FIRMWARE_BUFFER that helds the DSP file in several parts
            *   @see PIOXilinxLoad()
            *   @return TRUE or FALSE (error while download)
            */
            // overload requested !
    virtual BOOL    HALDspLoad( DWORD PmNbDSPBuff,IN LPFIRMWARE_BUFFER PmDSPBuff ) = 0;

            /**
            *   Load the bootstrap on NP boards.
            */
    virtual WORD    PIODownloadBootStrap( IN PBYTE PmEepromFile, DWORD PmEepromLength, IN PBYTE PmBootFile, DWORD PmBootLength ) = 0;
    virtual void*   PIOGetBootStrapPtrAsVoid() { return NULL; };


            /**
            *   Method used to transform dB values to driver coded values and the inverse operation.
            *   @param PmX1     IN  parameter X1 (first range)
            *   @param PmY1     IN  parameter Y1
            *   @param PmX2     IN  parameter X2 (second range)
            *   @param PmY2     IN  parameter Y2
            *   @param PmX      IN  parameter X  (value to transform)
            *   @return Transformed value
            */
            LONG    PIOConvert( LONG PmX1, LONG PmY1, LONG PmX2, LONG PmY2, LONG PmX );
            /**
            *   Get the features of each stereo audio output
            *   @return Combination of flags AUDIO_ANALOG_MSK, AUDIO_LEVEL_MSK, AUDIO_DIGITAL1_MSK 
            */
            WORD    PIOGetDescAudioOut(DWORD i){if(i<(MAX_BOARD_OUTPUTS/2)) return m_piDescAudioOut[i]; else return 0;}

            BYTE    PIOGetPhysOutFeature2 (DWORD i){if(i<(MAX_BOARD_OUTPUTS/2)) return m_bPhysOutFeature2[i]; else return 0;}
            /**
            *   Get the features of each stereo audio input
            *   @return Combination of flags AUDIO_ANALOG_MSK, AUDIO_MICRO_MSK, AUDIO_MICRO_IN_LEVEL_MSK, AUDIO_DIGITAL1_MSK
            */
            WORD    PIOGetDescAudioIn (DWORD i){if(i<(MAX_BOARD_INPUTS/2)) return m_piDescAudioIn[i]; else return 0;}

            BYTE    PIOGetPhysInFeature2 (DWORD i){if(i<(MAX_BOARD_INPUTS/2)) return m_bPhysInFeature2[i]; else return 0;}
            /**
            *   Get the miscancellous features of the card
            *   @return Combination of flags INTERSYNC_PRESENT_MASK, SMPTE_PRESENT_MASK, PROGRAMMABLE_CLOCK_PRESENT_MASK
            */
            BYTE    PIOGetMiscFeatures(){ return m_piMiscFeatures;}

            BYTE    PIOGetMiscFeatures2(){ return ETHERSOUND_CLOCK_PRESENT_MASK;}
            /**
            *   Get the options of the card
            *   @return Combination of flags VX_PRO_MODE_OPTION, HAS_PLX_OPTION, PCI_MASTER_OPTION, ASYNC_IO_OPTION
            */
            WORD    PIOGetOptions(){return m_piOptions;}

            /**
            *   Get the number of stereo inputs and outputs
            */
            VOID    PIOGetNbIOs( WORD *PmOuts, WORD *PmIns)
            { 
                if (PmOuts) *PmOuts = m_piStereoOutNumber;
                if (PmIns)  *PmIns  = m_piStereoInNumber;            
            }

            /**
            *   Update the number of stereo inputs and outputs and the frequency ratio
            *   and configure the card with the two parameters
            */
    virtual WORD    PIOUpdateESConfig( DWORD PmConfES ) = 0;

            /**
            *   Get the MACAddress of the card
            */
	virtual WORD    PIOGetMACAddress( PBYTE PmMACAddressPtr /* array of 6 bytes */) = 0;

    virtual VOID    PIOGetESConfig( PDWORD PmConfES ) = 0;

            VOID    PIOOptionsRemoveMPEG123(){m_piOptions &= ~OPT_MPEG123_FORMATS;}
            /**
            *   Load the boot and Xilinx file to the card.
            *   @param PmXilinx1File    IN  pointer to first Xilinx file
            *   @param PmXilinx1Length  IN  size of first Xilinx file in bytes
            *   @param PmXilinx2File    IN  pointer to second Xilinx file
            *   @param PmXilinx2Length  IN  size of second Xilinx file in bytes
            *   @see PIODspLoad()
            *   @return TRUE or FALSE (error while download)
            */
    virtual BOOL    PIOXilinxLoad( IN PBYTE PmXilinx1File, DWORD PmXilinx1Length, IN PBYTE PmXilinx2File, DWORD PmXilinx2Length );

            /**
            *   Xilinx file(s) needs to be loaded before BootDSP file(s).
            *   @return TRUE/FALSE
            */
    virtual BOOL    PIOLoadXilinxFirst() = 0;

    virtual BOOL    PIOIsFrequencyAutoComputed(){return TRUE;} /* false for np boards */

            BOOL    PIOIsFrequencyFirstChange(){return m_piIsFrequencyFirstChange;}

            VOID    PIOSetFrequencyFirstChange(BOOL PmIsFirst){m_piIsFrequencyFirstChange = PmIsFirst;}

            /**
            *   Wait until CHK = 1 (np boards).
            */
    virtual DWORD PIOWaitForCHK( ) { return ED_INVALID_BOARD;};

    virtual PVOID   PIOGetCSUFAddr() = 0;
    virtual DWORD   PIOReadMemBar(int offset) = 0;

    virtual BOOL    PIOCheckESPipeline(){ return FALSE;}
    virtual BOOL    PIOWriteCRES(DWORD CresMsb, DWORD CresLsb, BOOL bBroadcast){ return FALSE;}
    virtual BOOL    PIOReadCRES(PDWORD addrCresMsb, DWORD nbDword ){ return FALSE;}

   virtual  BOOL    DefineClockRegisterValue(IN DWORD PmFreq,OUT PDWORD PmRealFreqPtr);


            /**
            *   Things to do on pwrdown.
            */
   virtual  VOID    PIOPowerDown() { return ;};

	virtual WORD    PIOSetConfigSRC( WORD PmAudio, IN  SRC_CONFIG PmSrcConfig ){return ED_UNAVAILABLE_FEATURE;}
	virtual WORD    PIOGetConfigSRC( WORD PmAudio, OUT SRC_CONFIG* PmSrcConfig, OUT BYTE* PmSrcRatio ){return ED_UNAVAILABLE_FEATURE;}

/*@}*/


protected:
    WORD    m_wOutAnalogLevels[MAX_BOARD_OUTPUTS];
    WORD    m_wInAnalogLevels[MAX_BOARD_INPUTS];
    WORD    m_wInMicroLevels[MAX_BOARD_INPUTS];         // VX222 MIC
    BYTE    m_bInAnalogMute[MAX_BOARD_INPUTS];          // new
    BYTE    m_bInMicroMute[MAX_BOARD_INPUTS];           // VX222 MIC
    WORD    m_piDescAudioOut[MAX_BOARD_OUTPUTS / 2];    // piDescAudioOut describes stereo audio
    WORD    m_piDescAudioIn[MAX_BOARD_INPUTS / 2];      // piDescAudioIn  describes stereo audio
	BYTE    m_bPhysInFeature2[MAX_BOARD_INPUTS / 2];	// contains CARD_FEATURES_INFO.ciPhysicalInFeature2 parameters
	BYTE    m_bPhysOutFeature2[MAX_BOARD_OUTPUTS / 2];	// contains CARD_FEATURES_INFO.ciPhysicalOutFeature2 parameters

    WORD    m_piStereoOutNumber;
    WORD    m_piStereoInNumber;

    BYTE    m_piSn;                 //< number of UER sync sources.>
    BYTE    m_piNbWcl;              //< number of WordClock sync sources.>
    BYTE    m_piNbVideoSync;        //< number of Video sync sources.>
    BYTE    m_piEtherSoundSync;
    WORD    m_piOptions;            // VX_PRO_MODE_OPTION, PCXVX_V2_OPTION, ...
    BYTE    m_piMiscFeatures;       // Presence of SMPTE...
    BYTE    m_bliBusType;           // Bus type (PCI_BUS or PCCARD_BUS)
    BYTE    m_piType;               // board family PCXVX_BOARD, PCXVX_2_BOARD
    BYTE    m_bliBoardTypeNoAlias;  // Real board type (ex: VXPOCKET_BOARD)

    BOARD_LEVEL_INFO    m_LevelInfo;
    static  const LEVEL_INFO  m_driverLevelInfo;

    DWORD   m_DriverCodedAnalogBoardValueFor0dB;

    DWORD   m_piActualClkFrequency;   // unit is Hz
    WORD    m_piActualClkSource;      // CLOCK_TYPE_INTERNAL/UER_SYNC...
    DWORD   m_piActualClkInputNum;    // relevant only if ClkSource == UER_SYNC | WORD_CLOCK

    WORD    m_piActualClkType;        // INDEPENDANT_CLOCK/SLAVE_CLOCK/MASTER_CLOCK...
    WORD    m_piActualClkMasterNum;   // relevant only if ClockType == SLAVE
    DWORD   m_piActualClkSlaveMask;   // relevant only if ClockType == MASTER

	DWORD	m_piMACAddressMSB;		// 24 MSB bits of MAC Address 0x00MMMMMM.
	DWORD	m_piMACAddressLSB;		// 24 LSB bits of MAC Address 0x00mmmmmm.


public:

    LONG    m_SendMessageLocked;

protected:

    // IO accesses to the card
    inline  BYTE    UnivINP(PCXPORT a)          { return READ_PORT_UCHAR( (PUCHAR) m_aIOOffset[a] ); }
    inline  DWORD   UnivINPL(PCXPORT a)         { return READ_PORT_ULONG( (PULONG) m_aIOOffset[a] ); }
    inline  VOID    UnivOUTP(PCXPORT a,BYTE v)  {       WRITE_PORT_UCHAR( (PUCHAR) m_aIOOffset[a], (UCHAR) v ); }
    inline  VOID    UnivOUTPL(PCXPORT a,DWORD v){       WRITE_PORT_ULONG( (PULONG) m_aIOOffset[a], (ULONG) v ); }

    PBYTE	m_aIOOffset[eMaxPort];		// address in the corresponding IO Space
    DWORD   m_dwBoardVersion;
    BYTE    m_ucBoardFlavor;

    WORD    m_BoardIndex;
    BOOL    m_piXilinxTestDone;

    BOOL    m_EnableSendMessage;

            BOOL    SetAudioSource( WORD LcBoardAudio, BYTE PmSource );

	virtual BOOL	HALReadCBitsBasic( WORD PmSyncInputNum, OUT PBYTE PmMode, OUT PDWORD PmFreq, OUT PBYTE PmUerExtra) = 0;
	virtual BOOL	HALReadCByte( WORD PmSyncInputNum, IN DWORD PmByteID, OUT PBYTE PmCByte ) = 0;
	
    
    // overload requested !
    virtual BOOL    HALSendItDsp( DWORD PmNoIt ) = 0;
    virtual DWORD   HALResetCHK( ) = 0;
    virtual DWORD   HALReadRMHStatus( INOUT PCMD_RMH_INFO PmPtrCmdRmh ) = 0;
    virtual BOOL    HALWaitForBitValue_Timeout_ByteReg( IN PCXPORT           Pm8bitPort,
                                                        IN BYTE              PmBitToTest,
                                                        IN BOOLEAN           PmValue,
                                                        IN WORD              PmTimeout) = 0;       

    virtual BOOL    HALSetAudioAnalogLevel( WORD, BOOL ) = 0;
    virtual BOOL    HALHandleCodecSpeedModes( DWORD PmFrequency, BOOLEAN PmUnMuteWhenDone ) { return SUCCESS; };

private:


    LONG    PIOConvertLevelDriverCoded2BoardCoded( PLEVEL_INFO PmPBoardLevelInfo, LONG PmDriverCodedLevel );
    LONG    PIOConvertLevelBoardCoded2DriverCoded( PLEVEL_INFO PmPBoardLevelInfo, LONG PmBoardCodedLevel );

// Clock selection
    BOOL    CheckClockCompatibility( WORD PmClockType, WORD PmSource, DWORD PmFrequency, DWORD PmSyncInputNum );

// Analog levels Get/set
    BOOL    CheckAnalogLevelsSupport( IN WORD PmBoardAudioNum);

private:

    // overload requested !
    virtual BOOL    HALSetInAudioEffect( WORD PmAudio, BOOL PmActivate, LONG PmCompThreshold, LONG PmCompRatio, LONG PmNoiseThreshold, LONG PmOutputGain ) = 0;
    virtual BOOL    HALReadGPIO ( DWORD, OUT PDWORD ) { return WD_NO_HARDWARE_SUPPORT; };
    virtual BOOL    HALWriteGPIO( DWORD, DWORD ) { return WD_NO_HARDWARE_SUPPORT; };
    virtual BOOL    HALWriteOneCBit( WORD PmAudio, WORD PmBitIndex, BYTE PmBit )= 0;

    // this is NO MORE the main C-Bit read function, but mainly a status function.
    virtual WORD    HALGetAudioUer(  WORD PmSyncInputNum, OUT PBYTE PmSyncPtr, OUT PBYTE PmDataPtr,	OUT PDWORD  PmPhysFreq, IN  BOOL PmWaitLong = FALSE) = 0;

    virtual WORD    HALSetAudioSource( IN  WORD PmAudioNum, IN  BYTE  PmSource ) = 0;

    virtual BOOL    HALXilinxReset( BOOL PmFirstXilinx ) = 0;
    virtual VOID    HALXlxOutFile( IN PBYTE, DWORD ) = 0;
    virtual BOOL    HALTestXilinx( ) = 0;
    virtual DWORD   HALDefaultConfig( ) = 0;

    // clock commands issued to each board
    virtual WORD    HALSetClockDependency(  IN BYTE PmClockType ) = 0;

    BYTE    m_piTypeSrc[MAX_BOARD_INPUTS];
    BYTE    m_ucUerByte[MAX_BOARD_INPUTS / 2];
    DWORD   m_ucUerCBitFreq[MAX_BOARD_INPUTS / 2];
    BOOL    m_piIsFrequencyFirstChange;

	// protected HALHandleCodecSpeedModes called by :
	friend VOID DriverDebugFct( LPDWORD, LPDWORD );
};

// Wait time in micro-second.
//
#define XX_CODEC_RESET_WAIT_TIME       10       // 10 milliseconds
#define XX_DSP_RESET_WAIT_TIME          2       // 2 milliseconds

// Interrupt Control register bits
// -------------------------------
#define RREQ                0x01            // Interrupt mode + RREQ set on for transfer (->PC) request
#define TREQ                0x02            // Interrupt mode + HREQ set on for transfer (->DSP) request

// Constants used to access C Bits
#define UER_CBITS_OFFSET_MASK       0x1F

#endif // !defined(_PIOCOMMANDS_H__INCLUDED_)

