// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

// Include files
// *************
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <crtdbg.h> /*for ASSERT*/

#include "apidef.h"

#include "ifapidrv.h"   // private interface between driver and API
#include "pcxapi_e.h"   // prototypes of functions
#include "pcxerr_e.h"   // shared list of errror codes

#include "common.h"     // common objects shared between all API family
                        // functions

#include "genasync.h"
#include "util.h"


EXTERN DWORD Linear_DSP_Gain_LUT[];


// ****************************************************************************
// STATIC FLOAT Drv2ApiLevel()
// ***************************
//
// Input parameters :
// ****************
//
//  WORD        PmDrvLevelValue:    the driver code for the level
//
// Return value :
// **************
//
//  The corresponding value in dB ( floating value )
//
// ****************************************************************************
//
//  Converts driver absolute integer codes standing for 1/4 dB steps
//  into a floating point value in dB
//
// ****************************************************************************
STATIC FLOAT Drv2ApiLevel(
    IN  WORD    PmDrvLevelValue )
{
    FLOAT   LcLevel ;

    if ( PmDrvLevelValue == 0 )
    {
       return  NEGATIVE_INFINITY_LEVEL_VALUE;
    }

    LcLevel = ( MAX_LEVEL_VALUE
               - ( (FLOAT)( MAX_LEVEL_DRV_VALUE - PmDrvLevelValue )
                           / DRV_LEVEL_STEP ) ) ;
    return LcLevel ;
}

// ****************************************************************************
// STATIC FLOAT Drv2ApiMeter()
// ***************************
//
// Input parameters :
// ****************
//
//  DWORD       PmDrvValue:    the driver code for the view/peak meter
//
// Return value :
// **************
//
//  The corresponding value in dB ( floating value )
//
// ****************************************************************************
//
//  Converts driver absolute integer codes
//  into a floating point value in dB
//
// ****************************************************************************
STATIC FLOAT Drv2ApiMeter(
    IN  DWORD    PmDrvValue )
{
    FLOAT LcfValRet;

    if ( PmDrvValue == 0 )
       LcfValRet = NEGATIVE_INFINITY_LEVEL_VALUE;
    else
       LcfValRet = ( (FLOAT)( 20*log10( (double)PmDrvValue/(double)0x7FFFFF ) ) );

    if (LcfValRet < NEGATIVE_INFINITY_LEVEL_VALUE)
       LcfValRet = NEGATIVE_INFINITY_LEVEL_VALUE;

    return LcfValRet;
}



// Exported functions
// ******************

// ****************************************************************************
// WORD PCXGetPipeSampleNumber()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// Output parameters
// *****************
//
// PPCX_TIME PmSampleNumber: Number of sample since the last start on the
//   concerned pipe.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// The DSP is involved in this command. It is important for the
// application to watch its calls frequency.
//
// Only one pipe can be specified.
//
// ****************************************************************************

WORD _CDECL_ PCXGetPipeSampleNumber(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    OUT PPCX_TIME PmSampleNumber )
{
    LPPIPE_STATUS_INFO LcPReq = 0;           // will point to the beginning of
                                             // request block
    LPSAMPLE_COUNTING_RESP_INFO LcPResp = 0; // will point to the beginning of
                                             // response block
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_STATUS_FAM,
                    SAMPLE_COUNTING_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( PIPE_STATUS_INFO ),
                    sizeof( SAMPLE_COUNTING_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->scqOutputPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->scqInputPipeMask64.QuadPart = PmPipeInMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq       ,
        0                               ,
        (RESP_HEADER_HANDLE) &LcPResp   ,
        NULL                            );

    // reading the response
    // --------------------
    LcError = LcPResp->RespHeader.rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        *PmSampleNumber = LcPResp->scpSampleCount;
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;

}

// ****************************************************************************
// WORD PCXGetPipeState()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// Output parameters
// *****************
//
// LPBYTE PmState: 1 means that pipe has started.
//                 0 means that pipe is paused.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// The DSP is involved in this command. It is important for the
// application to watch its calls frequency.
//
// Only one pipe can be specified.
//
// ****************************************************************************

WORD _CDECL_ PCXGetPipeState(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    OUT LPBYTE PmState )
{
    LPPIPE_STATUS_INFO LcPReq = 0;      // will point to the beginning of the
                                        // request block
    LPPIPE_STATE_RESP_INFO LcPResp = 0; // will point to the beginning of the
                                        // response block
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_STATUS_FAM,
                    PIPE_STATE_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( PIPE_STATUS_INFO ),
                    sizeof( PIPE_STATE_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->scqOutputPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->scqInputPipeMask64.QuadPart = PmPipeInMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq       ,
        0                               ,
        (RESP_HEADER_HANDLE) &LcPResp   ,
        NULL                            );

    // reading the response
    // --------------------
    LcError = LcPResp->RespHeader.rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        *PmState = LcPResp->pspStart;
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetPipeClock()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// Output parameters
// *****************
//
// PCLOCK_INFO PmPipeClockInfo: the pipe clock description
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Only one pipe can be specified.
//
// ****************************************************************************

#define UER_FREQ_CALCULATION_LATENCY    300     // in ms

WORD _CDECL_ PCXGetPipeClock(
    IN  PCX_HANDLE PmMyProcess,
    IN  DWORD PmPipeOutMask,
    IN  DWORD PmPipeInMask,
    OUT PCLOCK_INFO PmPipeClockInfo)
{
    WORD                       LcError = 0;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

#pragma message ("TODO - force read de la clock pour l'instant, car le pipeclockmask et nul dans le driver") 

    LcError = GENGetPipeClock( TRUE /* force read !! */,
                               PmMyProcess,
                               PmPipeOutMask,
                               PmPipeInMask,
                               PmPipeClockInfo );

    // The signal is present but the frequency is null: this is normally due
    // to the fact that the application set the clock to UER or WORD_CLOCK
    // and is asking to the frequency to soon : the frequency has not yet been
    // calculated by the DSP.
    //
    if ( ( LcError == SUCCESS ) &&
         ( ( PmPipeClockInfo->ckType == CLOCK_TYPE_UER_SYNCHRO ) ||
           ( PmPipeClockInfo->ckType == CLOCK_TYPE_WORD_CLOCK ) ) &&
           ( PmPipeClockInfo->ckSync == UER_SYNC_PRESENT ) &&
           ( PmPipeClockInfo->ckFrequency == 0 ) )
    {
        // Give the time for the DSP to compute the frequency.
        // This value is deduced from the timer latency.
        // See coredsp/dspcfreq.asm:  2 * 31,77503030303 ms.
        // We set 300 instead of 63 because of the precision of the Sleep (
        // under NT) !!!
        //
#if defined(WIN32)
{
        Sleep( UER_FREQ_CALCULATION_LATENCY );
}
#else
{
        DWORD   LcBeginTime;
        DWORD   LcCurrentTime;

        LcBeginTime = GetTickCount();
        LcCurrentTime = LcBeginTime;

        while ( ( LcCurrentTime - LcBeginTime ) < UER_FREQ_CALCULATION_LATENCY )
        {
            LcCurrentTime = GetTickCount();
        }
}
#endif

        LcError = GENGetPipeClock( FALSE /* no force read */,
                                   PmMyProcess,
                                   PmPipeOutMask,
                                   PmPipeInMask,
                                   PmPipeClockInfo );
    }

    COMReleaseExmut();

    // returning the error code
    return( LcError );
}

// ****************************************************************************
// WORD PCXGetNotification()
// ********************
//
// Input parameters :
// ****************
//
// PCX_HANDLE   PmMyProcess:    Handle of the calling process.
// DWORD        PmInPipeMask:   Mask of the Input pipe concerned by the notify
//                      (if it is significant).
// DWORD        PmOutPipeMask:  Mask of the Input pipe concerned by the notify
//                      (if it is significant).
//
// Output parameters:
// ******************
//
// PNOTIFY_INFO PmNotification: Description of the notified command
//
// Return value :
// **************
//
// This function returns :
//
//           WD_NO_MORE_DATA: The pipe exists and there is a
//               PCXWaitForNotification running and no differed command
//               has been notified yet, or there is no PCXWaitForNotification
//               running and all notifications have been retrieved.
//               The NOTIFY_INFO structure is not updated.
//
//           SUCCESS : The couple (pmMyProcess, pmIn/OutPipeMask) does
//               correspond and at least one command has been notified.
//               The earliest unread notification since last pipe stop is
//               returned in the NOTIFY_INFO structure.
//
// ****************************************************************************
// Function to retrieve a notified differed command that
// PCXWaitForNotification has signalled for the specified pipe
//
// -------------------------------------------------------------------------
WORD _CDECL_ PCXGetNotification(
    IN  PCX_HANDLE  PmMyProcess,
    IN  DWORD       PmInPipeMask,
    IN  DWORD       PmOutPipeMask,
    OUT PNOTIFY_INFO PmNotification )
{
    LPPIPE_NOTIFICATION_REQ_INFO  LcPReq = 0;  // will point to the beginning
                                               // of the request block
    LPPIPE_NOTIFICATION_RESP_INFO LcPResp = 0; // will point to the beginning
                                               // of the response block
    WORD                          LcError = 0;

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // ## FS (16/10/1997) -- Do not try to read notifications
    // if there is a pending wait that may be finished soon  - FA #69
    // --------------------------------------------------------------
    if ( !GENCanGetNotification( PmMyProcess, PmInPipeMask, PmOutPipeMask) )
    {
        COMReleaseExmut();
        return WA_NO_MORE_DATA ;
    }


    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_STATUS_FAM,
                    PIPE_GET_NOTIFICATION_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( PIPE_NOTIFICATION_REQ_INFO ),
                    sizeof( PIPE_NOTIFICATION_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->pnqOutPipeMask64.QuadPart = PmOutPipeMask;
    LcPReq->pnqInPipeMask64.QuadPart  = PmInPipeMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq       ,
        0                               ,
        (RESP_HEADER_HANDLE) &LcPResp   ,
        NULL                            );

    LcError = LcPResp->RespHeader.rhCptr ;

    // Reset the reply structure
    // --------------------------
    MEMSET2( PmNotification, 0, NOTIFY_INFO, 1 );

    // reading the response
    // --------------------
    if ( LcError == SUCCESS )
    {
        // command notified
        PmNotification->niCommandCode = LcPResp->pnrCommandCode;

        if ( LcPResp->pnrCommandCode == 0 )
        {
            LcError = WA_NO_MORE_DATA ;
        }
        else
        {
            _ASSERT(LcPResp->pnrOutPipeMask64.HighPart == 0);    // not handled yet
            _ASSERT(LcPResp->pnrInPipeMask64.HighPart == 0);     // not handled yet

            PmNotification->niOutPipeMask = LcPResp->pnrOutPipeMask64.LowPart;
            PmNotification->niInPipeMask  = LcPResp->pnrInPipeMask64.LowPart;

            // differed param of the command
            (PmNotification->niExecTime).tiDelayed = LcPResp->pnrDiffered;

            // notification time
            (PmNotification->niExecTime).tiScheduler = LcPResp->pnrHour;

            // 1st param of the command
            PmNotification->niCommandParam1 = LcPResp->pnrParam1;

            // 2nd param of the command (buffer tag)
            PmNotification->niCommandParam2 = LcPResp->pnrParam2;
        }
    }

    COMReleaseExmut();

    // returning the error code
    return( LcError );
}


// ****************************************************************************
// WORD PCXGetAudioVuMeter()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// WORD PmPipeAudioInOutMask: Mask of physical audio input / output in relation
//                with the concerned pipe.
//
// Output parameters
// *****************
//
// PVU_METER_INFO PmPipeVuMeterInfo: Array of VU_METER_INFO
//                   structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// The DSP is involved in this command. It is important for the
// application to watch its calls frequency.
//
// Only one pipe can be specified.
// This command can concern several audio inputs/outputs of the
// pipe.
//
// ****************************************************************************

WORD _CDECL_ PCXGetAudioVuMeter(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN WORD PmPipeAudioInOutMask,
    OUT PVU_METER_INFO PmPipeVuMeterInfo )
{
    LPAUDIO_VUMETER_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                          // request block
    LPRESP_HEADER_INFO LcPResp = 0;       // will point to the beginning of the
                                          // response block
    LPVU_METER_RESP_INFO LcVuMeterInfo;
    DWORD LcPipeAudioMask;
    WORD  LcAudiosPerPipe = 0;
    WORD  LcInOutPipeAudios = 0;
    WORD  i, LcSize;
    WORD  LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    // counting the number of audios
    // ------------------------------
    LcPipeAudioMask = PmPipeAudioInOutMask;
    while (LcPipeAudioMask)
    {
        LcAudiosPerPipe += (WORD) (LcPipeAudioMask & 0x1);
        LcPipeAudioMask >>= 1 ;
    }
    LcPipeAudioMask = PmPipeOutMask;
    while (LcPipeAudioMask)
    {
        if(LcPipeAudioMask & 0x01)
        {
            LcInOutPipeAudios += LcAudiosPerPipe;
        }
        LcPipeAudioMask >>= 1 ;
    }
    LcPipeAudioMask = PmPipeInMask;
    while (LcPipeAudioMask)
    {
        if(LcPipeAudioMask & 0x01)
        {
            LcInOutPipeAudios += LcAudiosPerPipe;
        }
        LcPipeAudioMask >>= 1 ;
    }

    if( (LcInOutPipeAudios == 0) || (LcInOutPipeAudios > MAX_BOARD_AUDIO) )
        return EA_BAD_AUDIO_NUMBER;

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_PIPE_STATUS_FAM,
                    AUDIO_VU_PEAK_METERS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( AUDIO_VUMETER_REQ_INFO ),
                    (WORD)(  sizeof( RESP_HEADER_INFO )
                     + LcInOutPipeAudios * sizeof( VU_METER_RESP_INFO ) ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->avqOutputPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->avqInputPipeMask64.QuadPart = PmPipeInMask;
    LcPReq->avqAudioMask = PmPipeAudioInOutMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        &LcPResp                    ,
        NULL                        );

    // reading the response
    // --------------------
    LcError = LcPResp->rhCptr ;
    if ( !( LcError & ERROR_MASK) )
    {
        for ( i = 0, LcSize = 0,
              LcVuMeterInfo =  (LPVU_METER_RESP_INFO)(LcPResp + 1);
              (LcSize < LcPResp->rhSize) && (i < LcInOutPipeAudios);
              i++, LcVuMeterInfo++, LcSize += sizeof(VU_METER_RESP_INFO) )
        {
            PmPipeVuMeterInfo[i].vmVuMeter =    Drv2ApiMeter( LcVuMeterInfo->vmVuMeter );
            PmPipeVuMeterInfo[i].vmPeakMeter =  Drv2ApiMeter( LcVuMeterInfo->vmPeakMeter );
            PmPipeVuMeterInfo[i].vmSaturation = LcVuMeterInfo->vmSaturation;
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetAudioUer()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// WORD PmPipeAudioInOutMask: Mask of physical audio input / output in relation
//                  with the concerned pipe.
//
// Output parameters
// *****************
//
// PUER_INFO PmUerInfo: Array of UER_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Only one pipe can be specified.
// This command can concern several audio inputs/outputs of the
// pipe.
//
// ****************************************************************************

WORD _CDECL_ PCXGetAudioUer(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeInMask,
    IN WORD PmPipeAudioInOutMask,
    OUT PUER_INFO PmUerInfo )
{
    LPUER_REQ_INFO LcPReq = 0;         // will point to the beginning of the
                                       // request block
    LPRESP_HEADER_INFO LcPResp = 0;    // will point to the beginning of the
                                       // response block
    LPUER_RESP_INFO LcUerInfo;
    DWORD LcNbAudio = 0;
    DWORD LcAudioMask = PmPipeAudioInOutMask;
    WORD i, LcSize;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    // counting the number of audios.
    while ( LcAudioMask )
    {
        LcNbAudio += LcAudioMask & 0x1;
        LcAudioMask >>= 1;
    }

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_PIPE_STATUS_FAM,
                    UER_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( UER_REQ_INFO ),
                    MAX_UER_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->urqInputPipeMask64.QuadPart = PmPipeInMask;
    LcPReq->urqAudioMask = PmPipeAudioInOutMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        &LcPResp                    ,
        NULL                        );

    // reading the response
    // --------------------
    LcError = LcPResp->rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        for ( i = 0, LcSize = 0,
              LcUerInfo =  (LPUER_RESP_INFO)(LcPResp + 1);
              (LcSize < LcPResp->rhSize) && (i < LcNbAudio);
              i ++, LcUerInfo ++, LcSize += sizeof(UER_RESP_INFO) )
        {
            PmUerInfo[i].uiData      = LcUerInfo->urpData     ;
            PmUerInfo[i].uiMode      = LcUerInfo->urpMode     ;
            PmUerInfo[i].uiFrequency = LcUerInfo->urpFrequency;
            PmUerInfo[i].uiExtra     = LcUerInfo->urpExtra    ;
            PmUerInfo[i].uiExtraValid= 1                      ;
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetOutAudioLevel()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// WORD PmPipeAudioInOutMask: Mask of physical audio input / output in relation
//                  with the concerned pipe.
//
// Output parameters
// *****************
//
// POUT_AUDIO_GET_LEVEL_INFO PmGetOutLevelInfo: Array of
//                                   OUT_AUDIO_GET_LEVEL_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Only one pipe can be specified.
// This command can concern several audio inputs/outputs of the
// pipe.
//
// ****************************************************************************

WORD _CDECL_ PCXGetOutAudioLevel(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN WORD PmPipeAudioInOutMask,
    OUT POUT_AUDIO_GET_LEVEL_INFO PmGetOutLevelInfo )
{
    LPOUT_AUDIO_GET_LEVELS_REQ_INFO LcPReq = 0;
                                       // will point to the beginning of the
                                       // request block
    LPRESP_HEADER_INFO LcPResp = 0;    // will point to the beginning of the
                                       // response block
    LPOUT_AUDIO_GET_LEVELS_RESP_INFO LcAudioLevelsInfo;
    DWORD LcNbAudio = 0;
    DWORD LcAudioMask = PmPipeAudioInOutMask;
    WORD i, LcSize;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    // counting the number of streams
    // ------------------------------
    while (LcAudioMask)
    {
        LcNbAudio += LcAudioMask & 0x1;
        LcAudioMask >>= 1;
    }

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_PIPE_STATUS_FAM,
                    OUT_AUDIO_GET_LEVELS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( OUT_AUDIO_GET_LEVELS_REQ_INFO ),
                    MAX_OUT_LEVELS_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->olqOutputPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->olqAudioMask = PmPipeAudioInOutMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        &LcPResp                    ,
        NULL                        );

    // reading the response
    // --------------------
    LcError = LcPResp->rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        for ( i = 0, LcSize = 0,
                    LcAudioLevelsInfo =
                        (LPOUT_AUDIO_GET_LEVELS_RESP_INFO)(LcPResp + 1);
              (LcSize < LcPResp->rhSize) && (i < LcNbAudio);
              i ++, LcAudioLevelsInfo ++,
                    LcSize += sizeof(OUT_AUDIO_GET_LEVELS_RESP_INFO) )
        {
            PmGetOutLevelInfo[i].oaglAnalogLevel =
                        Drv2ApiLevel( LcAudioLevelsInfo->olpAnalogLevel );
            PmGetOutLevelInfo[i].oaglDigitalLevel =
                        Drv2ApiLevel( LcAudioLevelsInfo->olpDigitalLevel );
            PmGetOutLevelInfo[i].oaglMute =
                        LcAudioLevelsInfo->olpMute ;
            PmGetOutLevelInfo[i].oaglMonitoringLevel =
                        Drv2ApiLevel( LcAudioLevelsInfo->olpMonitoringLevel );
            PmGetOutLevelInfo[i].oaglMonitoringMute1 =
                        LcAudioLevelsInfo->olpMonitoringMute1;
            PmGetOutLevelInfo[i].oaglMonitoringMute2 =
                        LcAudioLevelsInfo->olpMonitoringMute2;
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetInAudioLevel()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// WORD PmPipeAudioInOutMask: Mask of physical audio input / output in relation
//                  with the concerned pipe.
//
// Output parameters
// *****************
//
// PGET_IN_LEVEL_INFO PmGetInLevelInfo: Array of GET_IN_LEVEL_INFO
//                              structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Only one pipe can be specified.
// This command can concern several audio inputs/outputs of the
// pipe.
//
// ****************************************************************************

WORD _CDECL_ PCXGetInAudioLevel(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeInMask,
    IN WORD PmPipeAudioInOutMask,
    OUT PIN_AUDIO_GET_LEVEL_INFO PmGetInLevelInfo )
{
    LPIN_AUDIO_GET_LEVELS_REQ_INFO LcPReq = 0;
                                        // will point to the beginning of the
                                        // request block
    LPRESP_HEADER_INFO LcPResp = 0;     // will point to the beginning of the
                                        // response block
    LPIN_AUDIO_GET_LEVELS_RESP_INFO LcAudioLevelsInfo;
    DWORD LcNbAudio = 0;
    DWORD LcAudioMask = PmPipeAudioInOutMask;
    WORD i, LcSize;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    // counting the number of streams
    // ------------------------------
    while ( LcAudioMask )
    {
        LcNbAudio += LcAudioMask & 0x1;
        LcAudioMask >>= 1;
    }

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_PIPE_STATUS_FAM,
                    IN_AUDIO_GET_LEVELS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( IN_AUDIO_GET_LEVELS_REQ_INFO ),
                    MAX_IN_LEVELS_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->ilqInPipeMask64.QuadPart = PmPipeInMask;
    LcPReq->ilqAudioMask = PmPipeAudioInOutMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        &LcPResp                    ,
        NULL                        );

    // reading the response
    // --------------------
    LcError = LcPResp->rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        for ( i = 0, LcSize = 0,
                LcAudioLevelsInfo =
                    (LPIN_AUDIO_GET_LEVELS_RESP_INFO)(LcPResp + 1);
              (LcSize < LcPResp->rhSize) && (i < LcNbAudio);
              i ++, LcAudioLevelsInfo ++,
                LcSize += sizeof(IN_AUDIO_GET_LEVELS_RESP_INFO) )
        {
            PmGetInLevelInfo[i].iaglAnalogLevel =
                        Drv2ApiLevel( LcAudioLevelsInfo->ilpAnalogLevel );
            PmGetInLevelInfo[i].iaglMicroLevel =
                        Drv2ApiLevel( LcAudioLevelsInfo->ilpMicroLevel );
            PmGetInLevelInfo[i].iaglDigitalLevel =
                        Drv2ApiLevel( LcAudioLevelsInfo->ilpDigitalLevel );

            PmGetInLevelInfo[i].iaglAnalogMute = LcAudioLevelsInfo->ilpAnalogMute;
            PmGetInLevelInfo[i].iaglMicroMute = LcAudioLevelsInfo->ilpMicroMute;
        }
    }

    COMReleaseExmut();

     // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetAudioAES3ChannelStatusByte()
// ******************************
//
// Input parameters :
// ****************
//
// PAUDIO_DECL_INFO PmAudio : board audio decsription
//
// BYTE             PmByteID: Byte ID according to AES3 (0..4)
//
// Output parameters
// *****************
//
// PBYTE            PmValidity  : UER_DATA_UNKNOWN, UER_NO_DATA, UER_DATA_PRESENT
//
// PBYTE            PmByteValue : Byte Value according to AES3
//
// Return value :
// **************
// SUCCESS or ED_INVALID_BOARD_AUDIO.
//
// ****************************************************************************
//
// Only one audio can be specified.
//
// ****************************************************************************

WORD _CDECL_ PCXGetAudioAES3ChannelStatusByte(
    IN  PAUDIO_DECL_INFO    PmAudio,
    IN  BYTE                PmByteID,
   OUT  PBYTE               PmValidity,
   OUT  PBYTE               PmByteValue )
{
    LPIN_AUDIO_GET_CBYTE_REQ_INFO LcPReq = 0;
    LPIN_AUDIO_GET_CBYTE_RESP_INFO LcPResp = 0;
    WORD LcError;

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_BOARD_STATUS_FAM,
                    IN_AUDIO_GET_CBYTE_CMD,
                    (APP_HANDLE)0xFF,	// NOT_USED
                    sizeof( *LcPReq ),
                    sizeof( *LcPResp ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Copy the audio description into the request block
    // -------------------------------------------------
    GENAudioDecl2PipeAudioInfo( PmAudio, &(LcPReq->iacqAudioInf) );

	LcPReq->iacqByteID = PmByteID;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        (RESP_HEADER_HANDLE)&LcPResp,
        NULL                        );

    // reading the response
    // --------------------
    LcError = (LcPResp->RespHeader).rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
		if(PmValidity)	*PmValidity = LcPResp->iacpValidity;
		if(PmByteValue)	*PmByteValue = LcPResp->iacpByteValue;
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetBoardAudioVuMeter()
// ******************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PAUDIO_DECL_INFO PmAudio : board audio decsription
//
// Output parameters
// *****************
//
// PVU_METER_INFO PmPipeVuMeterInfo: Array of VU_METER_INFO
//                   structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// The DSP is involved in this command. It is important for the
// application to watch its calls frequency.
//
// Only one audio can be specified.
//
// ****************************************************************************

WORD _CDECL_ PCXGetBoardAudioVuMeter(
    IN PCX_HANDLE       PmMyProcess,
    IN PAUDIO_DECL_INFO PmAudio,
    OUT PVU_METER_INFO  PmPipeVuMeterInfo )
{
    LPBOARD_AUDIO_VUMETER_REQ_INFO LcPReq = 0;
                                            // will point to the beginning of the
                                            // request block
    LPBOARD_AUDIO_VUMETER_RESP_INFO LcPResp = 0;
                                            // will point to the beginning of the
                                            // response block
    LPVU_METER_RESP_INFO LcVuMeterInfo;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_BOARD_STATUS_FAM,
                    AUDIO_VU_PEAK_METERS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( BOARD_AUDIO_VUMETER_REQ_INFO ),
                    sizeof( BOARD_AUDIO_VUMETER_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Copy the audio description into the request block
    // -------------------------------------------------
    GENAudioDecl2PipeAudioInfo( PmAudio, &(LcPReq->bavqAudioInf) );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        (RESP_HEADER_HANDLE)&LcPResp,
        NULL                        );

    // reading the response
    // --------------------
    LcError = (LcPResp->RespHeader).rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        LcVuMeterInfo = &(LcPResp->bavrVuMeter);

        // Converts driver meters into user values in dB
        // ---------------------------------------------
        PmPipeVuMeterInfo->vmVuMeter =
            Drv2ApiMeter( LcVuMeterInfo->vmVuMeter );

        PmPipeVuMeterInfo->vmPeakMeter =
            Drv2ApiMeter( LcVuMeterInfo->vmPeakMeter );

        PmPipeVuMeterInfo->vmSaturation = LcVuMeterInfo->vmSaturation;

    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetBoardClockFeatures()
// ******************************
//
WORD _CDECL_ PCXGetBoardClockFeatures(
    IN WORD                 PmBoardType,
    IN PCLOCK_FEATURES_INFO PmClockFeaturesInfo )
{
    WORD LcError = SUCCESS;

    // better not write kernel code just for this

    switch (PmBoardType) {
        case LXES:
             PmClockFeaturesInfo->cfiFixedClockMax = 192000;
            break;

        default:
            LcError = ED_INVALID_BOARD;
    }
        return LcError;
} // PCXGetBoardClockFeatures

// ****************************************************************************
// WORD PCXGetBoardAudioUer()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PAUDIO_DECL_INFO PmAudio : board audio decsription
//
// Output parameters
// *****************
//
// PUER_INFO PmUerInfo: Array of UER_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Only one audio can be specified.
//
// ****************************************************************************

WORD _CDECL_ PCXGetBoardAudioUer(
    IN PCX_HANDLE PmMyProcess,
    IN PAUDIO_DECL_INFO PmAudio,
    OUT PUER_INFO PmUerInfo )
{
    LPBOARD_UER_REQ_INFO LcPReq = 0;    // will point to the beginning of the
                                        // request block
    LPBOARD_UER_RESP_INFO LcPResp = 0;  // will point to the beginning of the
                                        // response block
    LPUER_RESP_INFO LcUerInfo;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_BOARD_STATUS_FAM,
                    UER_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( BOARD_UER_REQ_INFO ),
                    sizeof( BOARD_UER_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Copy the audio description into the request block
    // -------------------------------------------------
    GENAudioDecl2PipeAudioInfo( PmAudio, &(LcPReq->buqAudioInf) );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        (RESP_HEADER_HANDLE)&LcPResp,
        NULL                        );

    // reading the response
    // --------------------
    LcError = (LcPResp->RespHeader).rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        LcUerInfo =  &(LcPResp->burUer);

        PmUerInfo->uiData      = LcUerInfo->urpData     ;
        PmUerInfo->uiMode      = LcUerInfo->urpMode     ;
		PmUerInfo->uiFrequency = LcUerInfo->urpFrequency;
		PmUerInfo->uiExtra     = LcUerInfo->urpExtra    ;
		PmUerInfo->uiExtraValid= 1                      ;
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetOutBoardAudioLevel()
// *******************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PAUDIO_DECL_INFO PmAudio : board audio decsription
//
// Output parameters
// *****************
//
// POUT_AUDIO_GET_LEVEL_INFO PmGetOutLevelInfo: Array of
//                                   OUT_AUDIO_GET_LEVEL_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Only one audio can be specified.
//
// ****************************************************************************

WORD _CDECL_ PCXGetOutBoardAudioLevel(
    IN PCX_HANDLE PmMyProcess,
    IN PAUDIO_DECL_INFO PmAudio,
    OUT POUT_AUDIO_GET_LEVEL_INFO PmGetOutLevelInfo )
{
    LPOUT_BOARD_AUDIO_GET_LEVELS_REQ_INFO LcPReq = 0;
                                        // will point to the beginning of the
                                        // request block
    LPOUT_BOARD_AUDIO_GET_LEVELS_RESP_INFO LcPResp = 0;
                                        // will point to the beginning of the
                                        // response block
    LPOUT_AUDIO_GET_LEVELS_RESP_INFO LcAudioLevelsInfo;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_BOARD_STATUS_FAM,
                    OUT_AUDIO_GET_LEVELS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( OUT_BOARD_AUDIO_GET_LEVELS_REQ_INFO ),
                    sizeof( OUT_BOARD_AUDIO_GET_LEVELS_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Copy the audio description into the request block
    // -------------------------------------------------
    GENAudioDecl2PipeAudioInfo( PmAudio, &(LcPReq->oblqAudioInf) );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        (RESP_HEADER_HANDLE)&LcPResp,
        NULL                        );

    // reading the response
    // --------------------
    LcError = (LcPResp->RespHeader).rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        LcAudioLevelsInfo = &(LcPResp->oblrLevels);

        // Converts all driver level values into user values in dB
        // -------------------------------------------------------
        PmGetOutLevelInfo->oaglAnalogLevel =
            Drv2ApiLevel( LcAudioLevelsInfo->olpAnalogLevel );

        PmGetOutLevelInfo->oaglDigitalLevel =
            Drv2ApiLevel( LcAudioLevelsInfo->olpDigitalLevel );

        PmGetOutLevelInfo->oaglMute =   LcAudioLevelsInfo->olpMute ;

        PmGetOutLevelInfo->oaglMonitoringLevel =
            Drv2ApiLevel( LcAudioLevelsInfo->olpMonitoringLevel );

        PmGetOutLevelInfo->oaglMonitoringMute1 =
            LcAudioLevelsInfo->olpMonitoringMute1;

        PmGetOutLevelInfo->oaglMonitoringMute2 =
            LcAudioLevelsInfo->olpMonitoringMute2;

    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetInBoardAudioLevel()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PAUDIO_DECL_INFO PmAudio : board audio decsription
//
// Output parameters
// *****************
//
// PGET_IN_LEVEL_INFO PmGetInLevelInfo: Array of GET_IN_LEVEL_INFO
//                              structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Only one audio can be specified.
//
// ****************************************************************************

WORD _CDECL_ PCXGetInBoardAudioLevel(
    IN PCX_HANDLE PmMyProcess,
    IN PAUDIO_DECL_INFO PmAudio,
    OUT PIN_AUDIO_GET_LEVEL_INFO PmGetInLevelInfo )
{
    LPIN_BOARD_AUDIO_GET_LEVELS_REQ_INFO LcPReq = 0;
                                        // will point to the beginning of the
                                        // request block
    LPIN_BOARD_AUDIO_GET_LEVELS_RESP_INFO LcPResp = 0;
                                        // will point to the beginning of the
                                        // response block
    LPIN_AUDIO_GET_LEVELS_RESP_INFO LcAudioLevelsInfo;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_BOARD_STATUS_FAM,
                    IN_AUDIO_GET_LEVELS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( IN_BOARD_AUDIO_GET_LEVELS_REQ_INFO ),
                    sizeof( IN_BOARD_AUDIO_GET_LEVELS_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Copy the audio description into the request block
    // -------------------------------------------------
    GENAudioDecl2PipeAudioInfo( PmAudio, &(LcPReq->iblqAudioInf) );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        (RESP_HEADER_HANDLE)&LcPResp,
        NULL                        );

    // reading the response
    // --------------------
    LcError = (LcPResp->RespHeader).rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        LcAudioLevelsInfo = &(LcPResp->iblrLevels);

        // Converts all driver level values into user values in dB
        // -------------------------------------------------------
        PmGetInLevelInfo->iaglAnalogLevel =
            Drv2ApiLevel( LcAudioLevelsInfo->ilpAnalogLevel );
        PmGetInLevelInfo->iaglMicroLevel =
            Drv2ApiLevel( LcAudioLevelsInfo->ilpMicroLevel );
        PmGetInLevelInfo->iaglDigitalLevel =
            Drv2ApiLevel( LcAudioLevelsInfo->ilpDigitalLevel );

        PmGetInLevelInfo->iaglAnalogMute = LcAudioLevelsInfo->ilpAnalogMute;
        PmGetInLevelInfo->iaglMicroMute = LcAudioLevelsInfo->ilpMicroMute;
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetStreamAudioVuMeter()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmStreamMask: Stream mask.
//
// WORD PmPipeAudioOutMask: Mask of physical audio output in relation
//                with the concerned pipe.
//
// Output parameters
// *****************
//
// PVU_METER_INFO PmPipeVuMeterInfo: Array of VU_METER_INFO
//                   structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// The DSP is involved in this command. It is important for the
// application to watch its calls frequency.
//
// Only one pipe can be specified. Several streams are allowed.
// This command can concern several audio outputs of the
// pipe.
//
// ****************************************************************************

WORD _CDECL_ PCXGetStreamAudioVuMeter(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmStreamMask,
    IN WORD PmPipeAudioOutMask,
    OUT PVU_METER_INFO PmPipeVuMeterInfo )
{
    LPSTREAM_VU_METER_REQ_INFO LcPReq = 0; // will point to the beginning of the
                                           // request block
    LPRESP_HEADER_INFO LcPResp = 0;        // will point to the beginning of the
                                           // response block
    LPSTREAM_VU_METER_RESP_INFO LcVuMeterInfo;
    DWORD LcStreamMask = PmStreamMask;
    DWORD LcAudioMask = PmPipeAudioOutMask;
    WORD  LcNbAudio = 0;
    WORD  LcNbStream = 0;
    WORD  i;
    WORD  LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // counting the number of streams
    // ------------------------------
    while (LcStreamMask)
    {
        LcNbStream += (WORD) (LcStreamMask & 0x1);
        LcStreamMask >>= 1 ;
    }

    // counting the number of channel
    // ------------------------------
    while (LcAudioMask)
    {
        LcNbAudio += (WORD) (LcAudioMask & 0x1);
        LcAudioMask >>= 1 ;
    }

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_STATUS_FAM,
                    STREAM_VU_PEAK_METERS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( AUDIO_VUMETER_REQ_INFO ),
                    (WORD)(  sizeof( RESP_HEADER_INFO )
                     + LcNbStream * LcNbAudio * sizeof( VU_METER_RESP_INFO ) ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->svqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->svqStreamMask = PmStreamMask;
    LcPReq->svqAudioMask = PmPipeAudioOutMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        &LcPResp                    ,
        NULL                        );

    // reading the response
    // --------------------
    LcError = LcPResp->rhCptr ;
    if ( !( LcError & ERROR_MASK) )
    {
        LcVuMeterInfo = (LPSTREAM_VU_METER_RESP_INFO)(LcPResp + 1);

        for ( i = 0; i < LcNbStream * LcNbAudio; i++ )
        {
                PmPipeVuMeterInfo[i].vmVuMeter      = Drv2ApiMeter( LcVuMeterInfo->svpVuMeter );
                PmPipeVuMeterInfo[i].vmPeakMeter    = Drv2ApiMeter( LcVuMeterInfo->svpPeakMeter );
                PmPipeVuMeterInfo[i].vmSaturation =   (BOOLEAN)(PmPipeVuMeterInfo[i].vmPeakMeter == 0);
                LcVuMeterInfo++;
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetStreamState()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// DWORD PmStreamMask : Mask of stream.
//
// Output parameters
// *****************
//
// PSTREAM_STATE_INFO PmStreamStateInfo: : Array of PSTREAM_STATE_INFO
//                                    structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// The DSP is involved in this command. It is important for the
// application to watch its calls frequency.
//
// Only one pipe can be specified.
// This command can concern several streams of the pipe.
//
// ****************************************************************************

WORD _CDECL_ PCXGetStreamState(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN DWORD PmStreamMask,
    OUT PSTREAM_STATE_INFO PmStreamStateInfo )
{
    LPSTREAM_STATE_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                         // request block
    LPRESP_HEADER_INFO LcPResp = 0;      // will point to the beginning of the
                                         // response block
    LPSTREAM_STATE_RESP_INFO LcStateInfo;
    DWORD LcNbStream = 0;
    DWORD LcStreamMask = PmStreamMask;
    WORD i, LcSize;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // counting the number of streams.
    while ( LcStreamMask )
    {
        LcNbStream += LcStreamMask & 0x1;
        LcStreamMask >>= 1;
    }

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_STATUS_FAM,
                    STREAM_STATE_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof(STREAM_STATE_REQ_INFO),
                    MAX_STREAM_STATE_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->ssqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->ssqInPipeMask64.QuadPart = PmPipeInMask;
    LcPReq->ssqStreamMask = PmStreamMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        &LcPResp                    ,
        NULL                        );

    // reading the response
    // --------------------
    LcError = LcPResp->rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        for ( i = 0, LcSize = 0,
              LcStateInfo=  (LPSTREAM_STATE_RESP_INFO)(LcPResp + 1);
              (LcSize < LcPResp->rhSize) && (i < LcNbStream);
              i ++, LcStateInfo ++, LcSize += sizeof(STREAM_STATE_RESP_INFO) )
        {
            PmStreamStateInfo[i].ssStart = LcStateInfo->sspStart;
            PmStreamStateInfo[i].ssTime = LcStateInfo->sspHour;
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetStreamSampleNumber()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// DWORD PmStreamMask : Mask of stream.
//
// Output parameters
// *****************
//
// PPCX_TIME PmSampleNumber: Array of number of samples since the last
// stop on the concerned streams.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// The DSP is involved in this command. It is important for the
// application to watch its calls frequency.
//
// Only one pipe can be specified.
// This command can concern several streams of the pipe.
//
// ****************************************************************************
WORD _CDECL_ PCXGetStreamSampleNumber(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN DWORD PmStreamMask,
    OUT PPCX_TIME PmSampleNumber )
{
    LPSTREAM_SAMPLE_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                         // request block
    LPRESP_HEADER_INFO LcPResp = 0;      // will point to the beginning of the
                                         // response block
    LPSTREAM_SAMPLE_RESP_INFO LcSampleInfo;
    DWORD LcNbStream = 0;
    DWORD LcStreamMask = PmStreamMask;
    WORD i, LcSize;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // counting the number of streams.
    while ( LcStreamMask )
    {
        LcNbStream += LcStreamMask & 0x1;
        LcStreamMask >>= 1;
    }

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_STATUS_FAM,
                    STREAM_SAMPLE_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof(STREAM_SAMPLE_REQ_INFO),
                    MAX_STREAM_SAMPLE_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->ssqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->ssqInPipeMask64.QuadPart = PmPipeInMask;
    LcPReq->ssqStreamMask = PmStreamMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        &LcPResp                    ,
        NULL                        );

    // reading the response
    // --------------------
    LcError = LcPResp->rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        for ( i = 0, LcSize = 0,
              LcSampleInfo=  (LPSTREAM_SAMPLE_RESP_INFO)(LcPResp + 1);
              (LcSize < LcPResp->rhSize) && (i < LcNbStream);
              i ++, LcSampleInfo ++, LcSize += sizeof(STREAM_SAMPLE_RESP_INFO) )
        {
            PmSampleNumber[i] = LcSampleInfo->sspHour;
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
} // PCXGetStreamSampleNumber

// ****************************************************************************
// WORD PCXGetStreamSamplePosition()
// ****************************************************************************
//
WORD _CDECL_ PCXGetStreamSamplePosition(
    IN  PCX_HANDLE          PmMyProcess,
    IN  DWORD               PmPipeOutMask,
    IN  DWORD               PmPipeInMask,
    IN  DWORD               PmStreamMask,
   OUT  PPCX_TIME           PmSamplePosition )
{
    LPSTREAM_SAMPLE_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                         // request block
    LPRESP_HEADER_INFO LcPResp = 0;      // will point to the beginning of the
                                         // response block
    LPSTREAM_SAMPLE_RESP_INFO LcSampleInfo;
    DWORD LcNbStream = 0;
    DWORD LcStreamMask = PmStreamMask;
    WORD i, LcSize;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // counting the number of streams.
    while ( LcStreamMask )
    {
        LcNbStream += LcStreamMask & 0x1;
        LcStreamMask >>= 1;
    }

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_STATUS_FAM,
                     STREAM_POSITION_CMD,
                     (APP_HANDLE)PmMyProcess,
                     sizeof(STREAM_SAMPLE_REQ_INFO),
                     MAX_STREAM_SAMPLE_RESP_SIZE,
                     (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->ssqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->ssqInPipeMask64.QuadPart = PmPipeInMask;
    LcPReq->ssqStreamMask = PmStreamMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        &LcPResp                    ,
        NULL                        );

    // reading the response
    // --------------------
    LcError = LcPResp->rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        for ( i = 0, LcSize = 0,
              LcSampleInfo=  (LPSTREAM_SAMPLE_RESP_INFO)(LcPResp + 1);
              (LcSize < LcPResp->rhSize) && (i < LcNbStream);
              i ++, LcSampleInfo ++, LcSize += sizeof(STREAM_SAMPLE_RESP_INFO) )
        {
            PmSamplePosition[i] = LcSampleInfo->sspHour;
        }
    }
    COMReleaseExmut();

    // returning the error code
    return LcError;
} // PCXGetStreamSamplePosition



// ****************************************************************************
// WORD PCXGetStreamByteNumber()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// DWORD PmByteMask : Mask of stream.
//
// Output parameters
// *****************
//
// PPCX_TIME PmByteNumber: Array of number of bytes since the last
// stop on the concerned streams.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// The DSP is involved in this command. It is important for the
// application to watch its calls frequency.
//
// Only one pipe can be specified.
// This command can concern several streams of the pipe.
//
// ****************************************************************************
WORD _CDECL_ PCXGetStreamByteNumber(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN DWORD PmStreamMask,
    OUT PPCX_TIME PmByteNumber )
{
    LPSTREAM_SAMPLE_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                         // request block
    LPRESP_HEADER_INFO LcPResp = 0;      // will point to the beginning of the
                                         // response block
    LPSTREAM_SAMPLE_RESP_INFO LcSampleInfo;
    DWORD LcNbStream = 0;
    DWORD LcStreamMask = PmStreamMask;
    WORD i, LcSize;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // counting the number of streams.
    while ( LcStreamMask )
    {
        LcNbStream += LcStreamMask & 0x1;
        LcStreamMask >>= 1;
    }

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_STATUS_FAM,
                    STREAM_BYTE_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof(STREAM_SAMPLE_REQ_INFO),
                    MAX_STREAM_SAMPLE_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->ssqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->ssqInPipeMask64.QuadPart = PmPipeInMask;
    LcPReq->ssqStreamMask = PmStreamMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        &LcPResp                    ,
        NULL                        );

    // reading the response
    // --------------------
    LcError = LcPResp->rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        for ( i = 0, LcSize = 0,
              LcSampleInfo=  (LPSTREAM_SAMPLE_RESP_INFO)(LcPResp + 1);
              (LcSize < LcPResp->rhSize) && (i < LcNbStream);
              i ++, LcSampleInfo ++, LcSize += sizeof(STREAM_SAMPLE_RESP_INFO) )
        {
            PmByteNumber[i] = LcSampleInfo->sspHour;
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXGetOutStreamLevel()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
// DWORD PmPipeOutMask: Output pipe mask.
// DWORD PmStreamMask : Mask of stream.
//
// Output parameters
// *****************
//
// POUT_STREAM_SET_LEVEL_INFO PmOutStreamLevelInfo : points to a descriptor array
//                                                   for stream levels
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// The DSP is involved in this command. It is important for the
// application to watch its calls frequency.
//
// Only one pipe can be specified.
// This command can concern several streams of the pipe.
//
// ****************************************************************************
WORD _CDECL_ PCXGetOutStreamLevel(
    IN  PCX_HANDLE PmMyProcess,
    IN  DWORD PmPipeOutMask,
    IN  DWORD PmStreamMask,
   OUT  POUT_STREAM_SET_LEVEL_INFO PmOutStreamLevelInfo )
{
    LPGET_OUT_STREAM_LEVELS_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                                  // request block
    LPRESP_HEADER_INFO LcPResp = 0;               // will point to the beginning of the
                                                  // response block
    LPGET_OUT_STREAM_LEVELS_RESP_INFO LcLevelInfo;
    DWORD       LcNbStream = 0;
    DWORD       LcStreamMask = PmStreamMask;
    WORD        i, LcSize;
    WORD        LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // counting the number of streams.
    while ( LcStreamMask )
    {
        LcNbStream += LcStreamMask & 0x1;
        LcStreamMask >>= 1;
    }

    if (!COMWaitExmut())
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_STATUS_FAM,
                    OUT_STREAM_GET_LEVELS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof(GET_OUT_STREAM_LEVELS_REQ_INFO),
                    MAX_GET_OUT_STREAM_LEVELS_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->oslqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->oslqStreamMask = PmStreamMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq   ,
        0                           ,
        &LcPResp                    ,
        NULL                        );

    // reading the response
    // --------------------
    LcError = LcPResp->rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        for ( i = 0, LcSize = 0,
              LcLevelInfo=  (LPGET_OUT_STREAM_LEVELS_RESP_INFO)(LcPResp + 1);
              (LcSize < LcPResp->rhSize) && (i < LcNbStream);
              i ++, LcLevelInfo ++, LcSize += sizeof(GET_OUT_STREAM_LEVELS_RESP_INFO) )
        {
            // Zero-fill the user structure
            //
            MEMSET2( &(PmOutStreamLevelInfo[i]), 0, OUT_STREAM_SET_LEVEL_INFO, 1 );

            // Copy all level values and the associated validation
            // bits
            //
            PmOutStreamLevelInfo[i].osslValidationMask1 |= OUT_STREAM_SET_LEVEL_LEFT_AUDIO1;
            PmOutStreamLevelInfo[i].osslLeftChannelToOut1DigitalLevel =
                Drv2ApiLevel( LcLevelInfo->oslrLeftChannelToOutput1DigitalLevel );

            PmOutStreamLevelInfo[i].osslValidationMask1 |= OUT_STREAM_SET_LEVEL_LEFT_AUDIO2;
            PmOutStreamLevelInfo[i].osslLeftChannelToOut2DigitalLevel =
                Drv2ApiLevel( LcLevelInfo->oslrLeftChannelToOutput2DigitalLevel );

            PmOutStreamLevelInfo[i].osslValidationMask1 |= OUT_STREAM_SET_LEVEL_RIGHT_AUDIO1;
            PmOutStreamLevelInfo[i].osslRightChannelToOut1DigitalLevel =
                Drv2ApiLevel( LcLevelInfo->oslrRightChannelToOutput1DigitalLevel );

            PmOutStreamLevelInfo[i].osslValidationMask1 |= OUT_STREAM_SET_LEVEL_RIGHT_AUDIO2;
            PmOutStreamLevelInfo[i].osslRightChannelToOut2DigitalLevel =
                Drv2ApiLevel( LcLevelInfo->oslrRightChannelToOutput2DigitalLevel );

            PmOutStreamLevelInfo[i].osslValidationMask1 |= OUT_STREAM_SET_LEVEL_STREAM_1;
            PmOutStreamLevelInfo[i].osslDigitalLevel1 =
                Drv2ApiLevel( LcLevelInfo->oslrDigitalLevel1 );

            PmOutStreamLevelInfo[i].osslValidationMask1 |= OUT_STREAM_SET_LEVEL_STREAM_2;
            PmOutStreamLevelInfo[i].osslDigitalLevel2 =
                Drv2ApiLevel( LcLevelInfo->oslrDigitalLevel2 );

            PmOutStreamLevelInfo[i].osslValidationMask1 |= OUT_STREAM_SET_LEVEL_MUTE_1;
            PmOutStreamLevelInfo[i].osslMute1 = LcLevelInfo->oslrMute1;

            PmOutStreamLevelInfo[i].osslValidationMask1 |= OUT_STREAM_SET_LEVEL_MUTE_2;
            PmOutStreamLevelInfo[i].osslMute2 = LcLevelInfo->oslrMute2;
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}




// ****************************************************************************
// WORD PCXGetOutStreamEqualization()
// **************************************
//
// Input parameters :
// ****************
//
//  PCX_HANDLE PmMyProcess: Handle of the calling process.
//  DWORD PmPipeOutMask: Output pipe mask
//  DWORD PmStreamMask: Stream mask for all pipes
//  DWORD PmStreamEqualInfoSize: the entries of the abova arrays
//  PMPEG_EQUAL_SUBBAND_INFO PmSubbandsInfo: array of equal descriptors
//  POUT_STREAM_MPEG_EQUAL_INFO PmStreamEqualInfo: array of equal descriptors
//
// Return value :
// **************
//
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to get the equalization settings
//
// ****************************************************************************
WORD _CDECL_ PCXGetOutStreamEqualization(
    IN  PCX_HANDLE                  PmMyProcess,
    IN  DWORD                       PmPipeOutMask,
    IN  DWORD                       PmStreamMask,
	IN  PCX_EFFECT_HANDLE			PmEffectID,
    IN  DWORD                       PmStreamEqualInfoSize,
    OUT PMPEG_EQUAL_SUBBAND_INFO    PmSubbandsInfo,
    OUT POUT_STREAM_EQUAL_INFO		PmStreamEqualInfo )
{
    LPGET_OUT_STREAM_EFFECTS_RESP_INFO  LcPResp = 0;
    WORD                    LcError = SUCCESS;
    WORD                    i;
    WORD                    LcIndex = 0;

	if (PmEffectID == EFFECT_EQUALIZATION_STREAM_MPEG)
		return PCXGetOutStreamMpegEqualization(	PmMyProcess,
												PmPipeOutMask,
												PmStreamMask,
												PmStreamEqualInfoSize,
												PmSubbandsInfo,
												PmStreamEqualInfo );

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // The caller wants information about the bands levels.
    // ----------------------------------------------------
    //
    if ( PmStreamEqualInfo != NULL )
    {
        // Initialize the response with default values.
        // --------------------------------------------
        //
        for ( i = 0 ; i < PmStreamEqualInfoSize; i++ )
        {
            PmStreamEqualInfo[i].seqValidity = FALSE;
        }

        // Read the current coeff for this stream.
        // ---------------------------------------
        //
        LcError = GENGetOutStreamEqualization( PmMyProcess,
                                               PmPipeOutMask,
                                               PmStreamMask,
											   EFFECT_EQUALIZATION_STREAM_LINEAR,
											   &LcPResp );

        // Fill the response.
        // ------------------
        //
        if ( LcError == SUCCESS )
        {
            // Loop on the bands numbers.
            //
            for ( i = 0 ; i < MIN(PmStreamEqualInfoSize,MAX_EQUALIZATION_LINEAR_COEFF); i++ )
            {

                PmStreamEqualInfo[i].seqValidity   = TRUE;
                PmStreamEqualInfo[i].seqBandNumber = i;

                  // Get the coeff.
                for ( LcIndex = 0; LcIndex < 361 /*sizeof(Linear_DSP_Gain_LUT)*/; LcIndex++)
					{
						if ( LcPResp->gserParam[i] == Linear_DSP_Gain_LUT[LcIndex])
						{
							LcPResp->gserParam[i] = LcIndex;
							break;
						}
					}

                PmStreamEqualInfo[i].seqLevel      = ((float)(LcPResp->gserParam[i])-180)/10;
            }
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}	// Get out stream Equalization


// ****************************************************************************
// WORD PCXGetOutStreamMpegEqualization()
// **************************************
//
// Input parameters :
// ****************
//
//  PCX_HANDLE PmMyProcess: Handle of the calling process.
//  DWORD PmPipeOutMask: Output pipe mask
//  DWORD PmStreamMask: Stream mask for all pipes
//  DWORD PmStreamEqualInfoSize: the entries of the abova arrays
//  PMPEG_EQUAL_SUBBAND_INFO PmSubbandsInfo: array of equal descriptors
//  POUT_STREAM_MPEG_EQUAL_INFO PmStreamEqualInfo: array of equal descriptors
//
// Return value :
// **************
//
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to get the equalization settings
//
// ****************************************************************************
WORD _CDECL_ PCXGetOutStreamMpegEqualization(
    IN  PCX_HANDLE                  PmMyProcess,
    IN  DWORD                       PmPipeOutMask,
    IN  DWORD                       PmStreamMask,
    IN  DWORD                       PmStreamEqualInfoSize,
    OUT PMPEG_EQUAL_SUBBAND_INFO    PmSubbandsInfo,
    OUT POUT_STREAM_MPEG_EQUAL_INFO PmStreamEqualInfo )
{
    LPGET_OUT_STREAM_EFFECTS_RESP_INFO  LcPResp = 0;
    WORD                                LcError = SUCCESS;
    WORD                                i;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // The caller wants information about the bands frequencies.
    // ---------------------------------------------------------
    //
    if ( PmSubbandsInfo != NULL )
    {
        CLOCK_INFO  LcPipeClockInfo;

        // Initialize the response with default values.
        // --------------------------------------------
        //
        for ( i = 0 ; i < PmStreamEqualInfoSize; i++ )
        {
            PmSubbandsInfo[i].esbValidity = FALSE;
        }

        // Read the pipe frequency.
        // ------------------------
        //
        LcError = GENGetPipeClock( FALSE /* no force read */,
                                   PmMyProcess,
                                   PmPipeOutMask,
                                   0,               // PmPipeInMask
                                   & LcPipeClockInfo );

        // Fill the response.
        // ------------------
        //
        if ( LcError == SUCCESS )
        {
            DWORD LcPipeFreq = LcPipeClockInfo.ckFrequency; // in Hz
            DWORD LcWidth    = ( LcPipeFreq / 2 ) / MAX_EQUALIZATION_MPEG_COEFF; // In Hz

            // Loop on the bands numbers.
            //
            for ( i = 0 ; i < MIN(PmStreamEqualInfoSize,MPEG_EQUAL_MAX_BANDS); i++ )
            {
                DWORD   LcMinIndex;
                DWORD   LcMaxIndex;
                DWORD   LcDummy;

                GENMpegEqualBandToIndex( i,
                                         & LcMinIndex,      // Min Index
                                         & LcDummy,         // Medium Index
                                         & LcMaxIndex );    // Max Index

                PmSubbandsInfo[i].esbValidity   = TRUE;
                PmSubbandsInfo[i].esbBandNumber = i;
                PmSubbandsInfo[i].esbLowerFrequency = LcWidth * LcMinIndex;
                PmSubbandsInfo[i].esbUpperFrequency = LcWidth * ( LcMaxIndex + 1 );
            }
        }
    }

    // The caller wants information about the bands levels.
    // ----------------------------------------------------
    //
    if ( PmStreamEqualInfo != NULL )
    {
        // Initialize the response with default values.
        // --------------------------------------------
        //
        for ( i = 0 ; i < PmStreamEqualInfoSize; i++ )
        {
            PmStreamEqualInfo[i].seqValidity = FALSE;
        }

        // Read the current coeff for this stream.
        // ---------------------------------------
        //
        LcError = GENGetOutStreamEqualization( PmMyProcess,
                                               PmPipeOutMask,
                                               PmStreamMask,
											   EFFECT_EQUALIZATION_STREAM_MPEG,
											   &LcPResp );

        // Fill the response.
        // ------------------
        //
        if ( LcError == SUCCESS )
        {

            // Loop on the bands numbers.
            //
            for ( i = 0 ; i < MIN(PmStreamEqualInfoSize,MPEG_EQUAL_MAX_BANDS); i++ )
            {
                DWORD   LcMedIndex;
                DWORD   LcDummy;

                GENMpegEqualBandToIndex( i,
                                         & LcDummy,        // Min Index
                                         & LcMedIndex,     // Medium Index
                                         & LcDummy );      // Max Index

                PmStreamEqualInfo[i].seqValidity   = TRUE;
                PmStreamEqualInfo[i].seqBandNumber = i;
                PmStreamEqualInfo[i].seqLevel      = GENMpegEqualCoeffToLevel( LcPResp->gserParam[LcMedIndex] );
            }
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}



// ****************************************************************************
// WORD PCXGetStreamInternalBufferSize()
// **************************************
//
// Input parameters :
// ****************
//
//   PCX_HANDLE          PmMyProcess,
//   DWORD               PmPipeOutMask,
//   DWORD               PmPipeInMask,
//   DWORD               PmStreamMask,
//   LPDWORD             PmPtrNbArrayEntries,
//
// Output parameters :
// *****************
//
//   LPDWORD             PmPtrNbArrayEntries,
//   LPDWORD             PmBufferSize
//
// Return value :
// **************
//
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to get the latency due to internal buffers ( in bytes )
//
// ****************************************************************************
//
WORD _CDECL_ PCXGetStreamInternalBufferSize(
    IN  PCX_HANDLE          PmMyProcess,
    IN  DWORD               PmPipeOutMask,
    IN  DWORD               PmPipeInMask,
    IN  DWORD               PmStreamMask,
 INOUT  LPDWORD             PmPtrNbArrayEntries,
   OUT  LPDWORD             PmBufferLatencyArray )
{
    WORD LcIndex;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if ( PmPtrNbArrayEntries && *PmPtrNbArrayEntries && PmBufferLatencyArray )
    {
        MEMSET2( PmBufferLatencyArray, 0, DWORD, *PmPtrNbArrayEntries );
    }

    if ( PmStreamMask == 0 ) return( EA_BAD_STREAM_NUMBER ) ;

    if ( GENLookupWaitCmdTab( PmPipeOutMask, PmPipeInMask, &LcIndex ) )
    {
        //  If only one stream is requested..
        //
        if (   PmBufferLatencyArray
        && PmPtrNbArrayEntries
        && ( PmStreamMask == UTIMaskDWord(UTIDWMask2Word(PmStreamMask)) ) )
    {

      DWORD LcLatencyArray[MAX_STREAM_PIPE] ;
      DWORD LcNbStream = MAX_STREAM_PIPE ;
      WORD  LcRet ;

      LcRet = GENGetStreamInternalBufferInfo( LcIndex,
                          &LcNbStream,
                          LcLatencyArray );

      PmBufferLatencyArray[0] =
            LcLatencyArray[UTIDWMask2Word(PmStreamMask)];
      *PmPtrNbArrayEntries = 1 ;

      return( LcRet );
    }
    else
    {
      return( GENGetStreamInternalBufferInfo( LcIndex,
                          PmPtrNbArrayEntries,
                          PmBufferLatencyArray ) );
    }
    }
    else
    {
        return( EA_INVALID_PIPE );
    }
}

