// FindDevice.cpp : implementation file
//

#include <windows.h>
#include <ks.h>
#include <tchar.h>

#if !defined(_UNICODE) && !defined(UNICODE)
#include <stdio.h>
#endif

#ifdef _DEBUG
#define DOUT2(t,a)  { TCHAR szDebug[256]; _stprintf(szDebug,t,a); OutputDebugString(szDebug);}
#define DOUT3(t,a,b){ TCHAR szDebug[256]; _stprintf(szDebug,t,a,b); OutputDebugString(szDebug);}
#else
#define DOUT2(t,a)
#define DOUT3(t,a,b)
#endif //_DEBUG


#define MAX_KEY_LENGTH 255


static BOOL IsWave1DeviceLinked( IN HKEY hSubKey )
{
    DWORD   retCode;
    HKEY    hParamKey;
    DWORD   dwLinked = 0;
    DWORD   cLinkedSize = sizeof(dwLinked);

    retCode = RegOpenKeyEx( hSubKey,
                            _T("#Wave01\\Control"),
                            0,
                            KEY_READ,
                            &hParamKey);

    if( retCode != ERROR_SUCCESS )
    {
        //DOUT2( TEXT("RegOpenKeyEx( #Wave01\\Control ) failed 0x%x\n"), retCode );
        return FALSE;
    }

    retCode = RegQueryValueEx(  hParamKey,
                                _T("Linked"),
                                0,
                                NULL,
                                (LPBYTE)&dwLinked,
                                &cLinkedSize);
    RegCloseKey( hParamKey );

    if( retCode != ERROR_SUCCESS )
    {
        DOUT2( TEXT("RegQueryValueEx( Linked ) failed 0x%x\n"), retCode );
        return FALSE;
    }

    // return TRUE if the device is linked (and not disabled or uninstalled or removed)
    return ( dwLinked != 0 );
}


static BOOL IsWave1DeviceFromDriverName( IN HKEY hSubKey, IN const PTCHAR szDriverName, IN PTCHAR szDevInstanceName)
{
    DWORD   retCode, i, count;
    HKEY    hEnumKey;
    TCHAR   szEnumName[8];
    TCHAR   szKeyName[MAX_KEY_LENGTH];
    DWORD   cKeySize = MAX_KEY_LENGTH * sizeof(TCHAR);  // sizeof(szDevInstanceName)
    BOOL    bRet = FALSE;

    // hSubKey::DeviceInstance is to compare with the Enum strings of our service
    //
    retCode = RegQueryValueEx(  hSubKey,
                                _T("DeviceInstance"),
                                0,
                                NULL,
                                (LPBYTE)szDevInstanceName,
                                &cKeySize);

    if( retCode != ERROR_SUCCESS )
    {
        DOUT2( TEXT("RegQueryValueEx( DeviceInstance ) = %x\n"), retCode );
        return FALSE;
    }

    // open handle to our service
    //
    _stprintf(szKeyName, _T("SYSTEM\\CurrentControlSet\\Services\\%s\\Enum"), szDriverName);

    retCode = RegOpenKeyEx( HKEY_LOCAL_MACHINE,
                            szKeyName,
                            0,
                            KEY_READ,
                            &hEnumKey);

    if( retCode != ERROR_SUCCESS )
    {
        DOUT3( TEXT("RegOpenKeyEx( %s ) failed 0x%x\n"), szKeyName, retCode );
        return FALSE;
    }

    // get number of enum strings
    //
    cKeySize = sizeof(count);
    count = 0;
    retCode = RegQueryValueEx(  hEnumKey,
                                _T("Count"),
                                0,
                                NULL,
                                (LPBYTE)&count,
                                &cKeySize);

    if( retCode != ERROR_SUCCESS )
    {
        DOUT2( TEXT("RegQueryValueEx( Count ) failed 0x%x\n"), retCode );
        RegCloseKey( hEnumKey );
        return FALSE;
    }

    // search for DeviceInstance in the enum strings
    //
    for(i=0; i<count; i++)
    {
        _stprintf(szEnumName, _T("%d"), i);

        cKeySize = sizeof(szKeyName);
        retCode = RegQueryValueEx(  hEnumKey,
                                    szEnumName,
                                    0,
                                    NULL,
                                    (LPBYTE)szKeyName,
                                    &cKeySize);

        if( retCode != ERROR_SUCCESS )
        {
            DOUT3( TEXT("RegQueryValueEx( %d ) failed 0x%x\n"), i, retCode );
            break;
        }

        DWORD lcid = MAKELCID(MAKELANGID(LANG_ENGLISH, SUBLANG_ENGLISH_US), SORT_DEFAULT);
        if (CompareString (lcid, NORM_IGNORECASE, szDevInstanceName, -1, szKeyName, -1) == CSTR_EQUAL)
        {
            // the device belongs to szDriverName !
            bRet = TRUE;
            break;
        }
    }

    RegCloseKey( hEnumKey );

    return bRet;
}


static BOOL GetWave1DeviceSymbolicLink( IN HKEY hSubKey, OUT PTCHAR szFileName)
{
    DWORD   retCode;
    HKEY    hParamKey;
    DWORD   cSize = MAX_KEY_LENGTH * sizeof(TCHAR);

    retCode = RegOpenKeyEx( hSubKey,
                            _T("#Wave01"),
                            0,
                            KEY_READ,
                            &hParamKey);

    if( retCode != ERROR_SUCCESS )
    {
        DOUT2( TEXT("RegOpenKeyEx( #Wave01 ) failed 0x%x\n"), retCode );
        return FALSE;
    }

    retCode = RegQueryValueEx(  hParamKey,
                                _T("SymbolicLink"),
                                0,
                                NULL,
                                (LPBYTE)szFileName,
                                &cSize);
    RegCloseKey( hParamKey );

    if( retCode != ERROR_SUCCESS )
    {
        DOUT2( TEXT("RegQueryValueEx( SymbolicLink ) failed 0x%x\n"), retCode );
        return FALSE;
    }

    DOUT2( TEXT("SymbolicLink = %s\n"), szFileName );
    return TRUE;
}


extern "C" BOOL FindDeviceInterfaceName(IN const PTCHAR szDriverName, OUT PTCHAR szFileName)
{
    HKEY    hKey;
    TCHAR   szKeyName[MAX_KEY_LENGTH];
    DWORD   cSubKeys=0;             // number of subkeys
    DWORD   cbMaxSubKey;            // longest subkey size
    DWORD   i, retCode;
    BOOL    bFound = FALSE;
    const   LPGUID  guid = (const LPGUID)&KSCATEGORY_RENDER;

    // open the key where are all the systems RENDER devices :
    // HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\DeviceClasses\{65E8773E-8F56-11D0-A3B9-00A0C9223196}

    _stprintf( szKeyName,_T("SYSTEM\\CurrentControlSet\\Control\\DeviceClasses\\{%08lX-%04X-%04x-%02X%02X-%02X%02X%02X%02X%02X%02X}"),
                            guid->Data1, guid->Data2, guid->Data3,
                            guid->Data4[0], guid->Data4[1], guid->Data4[2], guid->Data4[3],
                            guid->Data4[4], guid->Data4[5], guid->Data4[6], guid->Data4[7]);

    retCode = RegOpenKeyEx( HKEY_LOCAL_MACHINE,
                            szKeyName,
                            0,
                            KEY_READ,
                            &hKey);

    if( retCode != ERROR_SUCCESS )
    {
        DOUT3( TEXT("RegOpenKeyEx(%s) = %x\n"), szKeyName, retCode );
        return FALSE;
    }

    // Query all the SubKeys
    //
    retCode = RegQueryInfoKey(  hKey,               // key handle
                                NULL,               // buffer for class name
                                NULL,               // size of class string
                                NULL,               // reserved
                                &cSubKeys,          // number of subkeys
                                &cbMaxSubKey,       // longest subkey size
                                NULL,               // longest class string
                                NULL,               // number of values for this key
                                NULL,               // longest value name
                                NULL,               // longest value data
                                NULL,               // security descriptor
                                NULL);              // last write time

    if( retCode != ERROR_SUCCESS )
    {
        DOUT3( TEXT("RegQueryInfoKey(%s) failed %x\n"), szKeyName, retCode );
        RegCloseKey( hKey );
        return FALSE;
    }
    DOUT3( TEXT("Query to %s found %d SubKeys\n"), szKeyName, cSubKeys );

    // Enumerate all SubKeys, until RegEnumKeyEx fails
    //
    for (i=0; i<cSubKeys; i++)
    {
        HKEY    hSubKey;
        DWORD   cbName = MAX_KEY_LENGTH;    // size of name string

        retCode = RegEnumKeyEx( hKey, i,
                                szKeyName,
                                &cbName,
                                NULL,
                                NULL,
                                NULL,
                                NULL);

        if( retCode != ERROR_SUCCESS )
        {
            DOUT3( TEXT("RegEnumKeyEx(%d) failed %x\n"), i, retCode );
            break;
        }

        //  open each SubKey
        //
        retCode = RegOpenKeyEx( hKey,
                                szKeyName,
                                0,
                                KEY_READ,
                                &hSubKey);

        if( retCode != ERROR_SUCCESS )
        {
            DOUT3( TEXT("RegOpenKeyEx(%s) failed %x\n"), szKeyName, retCode );
            break;
        }

        // is there a #Wave01 device and is it linked ?
        //
        if( IsWave1DeviceLinked(hSubKey) )
        {
            // look if szDriverName is the service of SubKey\#Wave1 device
            //
            if( IsWave1DeviceFromDriverName(hSubKey, szDriverName, szKeyName) ) // re-use szKeyName in IsWave1DeviceFromDriverName for STACK reasons
            {
                // we found it, finally get its SymbolicLink
                //
                bFound = GetWave1DeviceSymbolicLink(hSubKey, szFileName);
            }
        }

        RegCloseKey( hSubKey );

        if(bFound) break;
    }

    RegCloseKey( hKey );

    return bFound;
}


#if(0)
// version avec SetupApi.dll


#include <windows.h>
#include <winioctl.h>
#include <ks.h>
#include <setupapi.h>

#ifdef _DEBUG
#define NPAPITRACE(x)   OutputDebugString((x))
#else
#define NPAPITRACE(x)
#endif //_DEBUG


static BOOL FindRenderDevice(   const PTCHAR szDriverName,
                                PSP_PROPSHEETPAGE_REQUEST  pspRequest,
                                PSP_DEVINFO_DATA DeviceInfoData)
{
    TCHAR           szServiceName[128];

    //
    // Prepare the pspRequest structure...
    //
    //pspRequest->cbSize = sizeof (SP_PROPSHEETPAGE_REQUEST);
    pspRequest->DeviceInfoData = DeviceInfoData;
    //pspRequest->PageRequested = SPPSR_ENUM_ADV_DEVICE_PROPERTIES;

    // ...and the DeviceInfoData structure.
    DeviceInfoData->cbSize = sizeof (SP_DEVINFO_DATA);

    // Create a list of devices with Render interface.
    pspRequest->DeviceInfoSet = SetupDiGetClassDevs ((LPGUID)&KSCATEGORY_RENDER,
                                     NULL, NULL, DIGCF_PRESENT | DIGCF_DEVICEINTERFACE);

    // None found?
    if (pspRequest->DeviceInfoSet == INVALID_HANDLE_VALUE)
    {
        NPAPITRACE( TEXT("Error SetupDiGetClassDevs\n") );
        return FALSE;
    }

    //
    // Go through the list of all devices found.
    //
    int nIndex = 0;

    while (SetupDiEnumDeviceInfo (pspRequest->DeviceInfoSet, nIndex, DeviceInfoData))
    {
        //
        // Get the service name for that device.
        //
        if (!SetupDiGetDeviceRegistryProperty (pspRequest->DeviceInfoSet, DeviceInfoData,
                                               SPDRP_SERVICE, NULL, (PBYTE)szServiceName,
                                               sizeof (szServiceName), NULL))
        {
            NPAPITRACE( TEXT("Error SetupDiGetDeviceRegistryProperty\n") );
            SetupDiDestroyDeviceInfoList (pspRequest->DeviceInfoSet);
            return FALSE;
        }

        //
        // We only care about service "szDriverName"
        //
        DWORD lcid = MAKELCID(MAKELANGID(LANG_ENGLISH, SUBLANG_ENGLISH_US), SORT_DEFAULT);
        if (CompareString (lcid, NORM_IGNORECASE, szServiceName,
                            -1, szDriverName, -1) == CSTR_EQUAL)
        {
            //
            // We found it! The information is already stored, just return.
            // Note that we have the device list still open - we have to destroy
            // the list later.
            //
            return TRUE;
        }

        // Take the next in the list.
        nIndex++;
    }

    //
    // We did not find the service.
    //
    SetupDiDestroyDeviceInfoList (pspRequest->DeviceInfoSet);
    return FALSE;
}



/////////////////////////////////////////////////////////////////////////////////
// GetDeviceInterfaceDetail
/////////////////////////////////////////////////////////////////////////////////
// This function gets called by the property page provider (in this module) to
// get the device interface details. The device interface detail contains a
// path to the device driver that can be used to open the device driver.
// When we parse the driver we look for the topology interface since this
// interface exposes the private property.
//
// Arguments:
//    pPropPageRequest           - points to SP_PROPSHEETPAGE_REQUEST
//    pDeviceInterfaceDetailData - device interface details returned.
//
// Return Value:
//    BOOL: FALSE if something went wrong, TRUE on success.
static BOOL GetDeviceInterfaceDetail (  PSP_PROPSHEETPAGE_REQUEST pPropPageRequest,
                                        PSP_DEVICE_INTERFACE_DETAIL_DATA *ppDeviceInterfaceDetailData)
{
    BOOL                        fSuccess;
    ULONG                       ulDeviceInstanceIdSize = 0;
    PTSTR                       pDeviceInstanceID = NULL;
    HDEVINFO                    hDevInfoWithInterface;
    SP_DEVICE_INTERFACE_DATA    DeviceInterfaceData;
    ULONG                       ulDeviceInterfaceDetailDataSize = 0;

    // Get the device instance id (PnP string).  The first call will retrieve
    // the buffer length in characters.  fSuccess will be FALSE.
    fSuccess = SetupDiGetDeviceInstanceId (pPropPageRequest->DeviceInfoSet,
                                           pPropPageRequest->DeviceInfoData,
                                           NULL,
                                           0,
                                           &ulDeviceInstanceIdSize);
    // Check for error.
    if ((GetLastError () != ERROR_INSUFFICIENT_BUFFER) || (!ulDeviceInstanceIdSize))
    {
        NPAPITRACE( TEXT("Error GetDeviceInterfaceDetail: SetupDiGetDeviceInstanceId\n") );
        return FALSE;
    }

    // Allocate the buffer for the device instance ID (PnP string).
    pDeviceInstanceID = (PTSTR)LocalAlloc (LPTR, ulDeviceInstanceIdSize * sizeof (TCHAR));
    if (!pDeviceInstanceID)
    {
        NPAPITRACE( TEXT("Error GetDeviceInterfaceDetail: LocalAlloc\n") );
        return FALSE;
    }

    // Now call again, this time with all parameters.
    fSuccess = SetupDiGetDeviceInstanceId (pPropPageRequest->DeviceInfoSet,
                                           pPropPageRequest->DeviceInfoData,
                                           pDeviceInstanceID,
                                           ulDeviceInstanceIdSize,
                                           NULL);
    // Check for error.
    if (!fSuccess)
    {
        NPAPITRACE( TEXT("Error GetDeviceInterfaceDetail: SetupDiGetDeviceInstanceId\n") );
        LocalFree (pDeviceInstanceID);
        return FALSE;
    }

    // Now we can get the handle to the dev info with interface.
    // We parse the device specifically for render interfaces.
    hDevInfoWithInterface = SetupDiGetClassDevs ((LPGUID)&KSCATEGORY_RENDER,
                                                 pDeviceInstanceID,
                                                 NULL,
                                                 DIGCF_DEVICEINTERFACE);
    // We don't need pDeviceInstanceID anymore.
    LocalFree (pDeviceInstanceID);

    // Check for error.
    if (hDevInfoWithInterface == INVALID_HANDLE_VALUE)
    {
        NPAPITRACE( TEXT("Error GetDeviceInterfaceDetail: SetupDiGetClassDevs\n") );
        return FALSE;
    }

    // Go through the list of render device interface of this device.
    // We assume that there is only one topology device interface and
    // we will store the device details in our private structure.
    DeviceInterfaceData.cbSize = sizeof (DeviceInterfaceData);
    fSuccess = SetupDiEnumDeviceInterfaces (hDevInfoWithInterface,
                                            NULL,
                                            (LPGUID)&KSCATEGORY_RENDER,
                                            0,
                                            &DeviceInterfaceData);
    // Check for error.
    if (!fSuccess)
    {
        NPAPITRACE( TEXT("Error GetDeviceInterfaceDetail: SetupDiEnumDeviceInterfaces\n") );
        SetupDiDestroyDeviceInfoList (hDevInfoWithInterface);
        return FALSE;
    }

    // Get the details for this device interface.  The first call will retrieve
    // the buffer length in characters.  fSuccess will be FALSE.
    fSuccess = SetupDiGetDeviceInterfaceDetail (hDevInfoWithInterface,
                                                &DeviceInterfaceData,
                                                NULL,
                                                0,
                                                &ulDeviceInterfaceDetailDataSize,
                                                NULL);
    // Check for error.
    if ((GetLastError () != ERROR_INSUFFICIENT_BUFFER) || (!ulDeviceInterfaceDetailDataSize))
    {
        NPAPITRACE( TEXT("Error GetDeviceInterfaceDetail: SetupDiGetDeviceInterfaceDetail\n") );
        SetupDiDestroyDeviceInfoList (hDevInfoWithInterface);
        return FALSE;
    }

    // Allocate the buffer for the device interface detail data.
    if (!(*ppDeviceInterfaceDetailData = (PSP_DEVICE_INTERFACE_DETAIL_DATA)
            LocalAlloc (LPTR, ulDeviceInterfaceDetailDataSize)))
    {
        NPAPITRACE( TEXT("Error GetDeviceInterfaceDetail: LocalAlloc\n") );
        SetupDiDestroyDeviceInfoList (hDevInfoWithInterface);
        return FALSE;
    }
    // The size contains only the structure, not the additional path.
    (*ppDeviceInterfaceDetailData)->cbSize = sizeof (SP_DEVICE_INTERFACE_DETAIL_DATA);

    // Get the details for this device interface, this time with all paramters.
    fSuccess = SetupDiGetDeviceInterfaceDetail (hDevInfoWithInterface,
                                                &DeviceInterfaceData,
                                                *ppDeviceInterfaceDetailData,
                                                ulDeviceInterfaceDetailDataSize,
                                                NULL,
                                                NULL);
    // We don't need the handle anymore.
    SetupDiDestroyDeviceInfoList (hDevInfoWithInterface);

    if (!fSuccess)
    {
        NPAPITRACE( TEXT("Error GetDeviceInterfaceDetail: SetupDiGetDeviceInterfaceDetail\n") );
        LocalFree (*ppDeviceInterfaceDetailData), *ppDeviceInterfaceDetailData = NULL;
        return FALSE;
    }

    return TRUE;
}


extern "C" BOOL FindDeviceInterfaceName(IN const PTCHAR szDriverName, OUT PTCHAR szFileName)
{
    SP_PROPSHEETPAGE_REQUEST    pspRequest;         // structure passed
    SP_DEVINFO_DATA             DeviceInfoData;     // pspRequest points to it.

	PSP_DEVICE_INTERFACE_DETAIL_DATA pDeviceInterfaceDetailData = NULL;
    BOOL ret;

    if (!FindRenderDevice(szDriverName, &pspRequest, &DeviceInfoData))
    {
        NPAPITRACE( TEXT("Error FindDeviceInterfaceName : No Digigram Driver\n") );
        return FALSE;
    }

    // Get the device interface detail which return a path to the device
    // driver that we need to open the device.
    ret = GetDeviceInterfaceDetail (&pspRequest, &pDeviceInterfaceDetailData);

    SetupDiDestroyDeviceInfoList (pspRequest.DeviceInfoSet);

    if(ret && (szFileName != NULL))
    {
        if( ! lstrcpy( szFileName, pDeviceInterfaceDetailData->DevicePath))
            ret = FALSE;
        NPAPITRACE( TEXT("FindDeviceInterfaceName : ") );
        NPAPITRACE(szFileName);
    }
    else
    {
        NPAPITRACE( TEXT("Error FindDeviceInterfaceName : No Digigram Driver Details\n") );
    }

    LocalFree (pDeviceInterfaceDetailData);

    return ret;
}
#endif // 0